#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 10669 2017-12-22 09:26:52Z David $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-12-22 17:26:52 +0800 (Fri, 22 Dec 2017) $
# $Revision: 10669 $

from django import forms
from django.contrib import admin
from django.utils.translation import ugettext as _
from django.conf import settings
from django.contrib.contenttypes.models import ContentType

from emencia.django.newsletter.models import Newsletter
from emencia.django.newsletter.admin import NewsletterAdmin


from editor.TinyMCEWidget import CustomTinyMCEWidget

from selectable.forms import AutoCompleteWidget, AutoComboboxWidget

from Theophrastus.lookups import (
    SenderLookup, ReplyLookup, HeaderLookup, FooterLookup)
from Theophrastus.newsletter_extension.models import Header, Footer

class NewsletterTinyMCEForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        
        newsletterObj = Newsletter.objects.all().order_by('-id')

        if newsletterObj:
            header = newsletterObj[0].header
            footer = newsletterObj[0].footer
            if header:
                headerTitle = header.title
            else:
                headerTitle = Header.objects.get(pk=1).title
            if footer:
                footerTitle = footer.title
            else:
                footerTitle = Footer.objects.get(pk=1).title
            
            sender = newsletterObj[0].header_sender
            reply = newsletterObj[0].header_reply
        else:
            sender = getattr(
                settings,
                'NEWSLETTER_DEFAULT_HEADER_SENDER',
                "Nuwa<service@tracedig.com>"
            )
            reply = getattr(
                settings,
                'NEWSLETTER_DEFAULT_HEADER_REPLY',
                "Nuwa<service@tracedig.com>"
            )
            headerTitle = Header.objects.get(pk=1).title
            footerTitle = Footer.objects.get(pk=1).title

        super(NewsletterTinyMCEForm, self).__init__(*args, **kwargs)
        self.fields['header_sender'].initial = sender
        self.fields['header_reply'].initial = reply
        self.fields['header'].initial = headerTitle
        self.fields['footer'].initial = footerTitle

    content = forms.CharField(
        widget=CustomTinyMCEWidget(attrs={'cols': 150, 'rows': 80}),
        label=_("Content"))
    
    header = forms.CharField(
        label=_('header'),
        widget=AutoComboboxWidget(lookup_class=HeaderLookup),
        required=True,
    )

    footer = forms.CharField(
        label=_('footer'),
        widget=AutoComboboxWidget(lookup_class=FooterLookup),
        required=True,
    )

    header_sender = forms.CharField(
        label=_('sender'),
        widget=AutoComboboxWidget(lookup_class=SenderLookup),
        required=True,
    )

    header_reply = forms.CharField(
        label=_('reply to'),
        widget=AutoComboboxWidget(lookup_class=ReplyLookup),
        required=True,
    )

    def clean_header_reply(self):
        """
        Validate reply value.
        """
        reply = self.cleaned_data["header_reply"]
        try:
            reply.encode()
        except UnicodeEncodeError:
            raise forms.ValidationError(
                _("Incorrect value for reply, it can only include English "
                "word, number and symbol."))
        return reply

    class Meta:
        model = Newsletter
        fields = '__all__'

class NewsletterTinyMCEAdmin(NewsletterAdmin):
    list_display = ('title', 'mailing_list', 'server', 'status',
                    'formatedSendingDate', 'formatedCreationDate',
                    'formatedModificationDate',
                    'historic_link', 'statistics_link')
    fieldsets = ((None, {'fields': ('title', 'subject', 
                         'header', 'footer',  
                         'content',)}),
                 (_('Receivers'), {'fields': (
                     'mailing_list', 'test_contacts',)}),
                 (_('Sending'), {'fields': ('sending_date', 'status',)}),
                 (_('Miscellaneous'), {'fields': ('server', 'header_sender',
                                                  'header_reply', 'slug'),
                                       'classes': ('collapse',)}),
                 )
    def save_model(self, request, obj, form, change):
        header = ContentType.objects.get(
            app_label="newsletter_extension", model="header")
        footer = ContentType.objects.get(
            app_label="newsletter_extension", model="footer")
        headerTitle = form.cleaned_data['header']
        footerTitle = form.cleaned_data['footer']
        headerId = Header.objects.filter(title=headerTitle)[0].id
        footerId = Footer.objects.filter(title=footerTitle)[0].id
        obj.headerContentType = header
        obj.footerContentType = footer
        obj.headerId = headerId
        obj.footerId = footerId

        return super(NewsletterTinyMCEAdmin, self).save_model(
            request, obj, form, change)  
    def formatedSendingDate(self, obj):
        return obj.sending_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedSendingDate.short_description = _('sending date')

    def formatedCreationDate(self, obj):
        return obj.creation_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreationDate.short_description = _('creation date')

    def formatedModificationDate(self, obj):
        return obj.modification_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedModificationDate.short_description = _('modification date')

    class Media:
        try:
            js = [
                'js/Newsletter.js',
            ]
        except AttributeError:
            pass

    form = NewsletterTinyMCEForm

admin.site.unregister(Newsletter)
admin.site.register(Newsletter, NewsletterTinyMCEAdmin)

from emencia.django.newsletter.models import Contact
from emencia.django.newsletter.admin import ContactAdmin

class ContactTheophrastusAdmin(ContactAdmin):
    list_display = (
        'email', 'first_name', 'last_name', 'tags', 'tester', 'subscriber',
        'valid', 'total_subscriptions', 'formatedCreationDate',
        'related_object_admin')

    def formatedCreationDate(self, obj):
        return obj.creation_date.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreationDate.short_description = _('creation date')

admin.site.unregister(Contact)
admin.site.register(Contact, ContactTheophrastusAdmin)
