#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: DjangoUtil.py 12417 2020-07-23 07:07:01Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $ (last)
# $Date: 2020-07-23 15:07:01 +0800 (週四, 23 七月 2020) $
# $Revision: 12417 $
"""
Utilities for django.
"""

import time
import warnings
import datetime
import mimetypes
import os
import urllib
import base64
import json

from django.http import HttpResponse
from Iuppiter.Encryption import Encryptor
from Iuppiter.Util import extendUnique

try:
    import django

    DJANGO_VERSION = (django.VERSION[0] * 10000 + django.VERSION[1] * 100 +
                      django.VERSION[2])
except ImportError:
    warnings.warn("Unable to import django, many Iuno packages require it.")
    
if DJANGO_VERSION < 20000: # 2.0.0:
    from django.core.urlresolvers import reverse
else:
    from django.urls import reverse
    
def encryptJson(data, key):
    data = json.dumps(data) 
    encryptor = Encryptor(key)
    secret = encryptor.encrypt(data.encode())  
    secret = base64.urlsafe_b64encode(secret)
    return secret.decode()
    
def decryptJson(secret, key):
    encryptor = Encryptor(key)
    secret = base64.urlsafe_b64decode(secret)
    data = encryptor.decrypt(secret)
    data = json.loads(data)
    return data

def patterns(prefix, *args): 
    if DJANGO_VERSION >= 11100: # django >= 1.11
        return list(args)
    else:
        from django.conf.urls import patterns as _patterns
        
        return _patterns(prefix, *args)
        
def extendInstalledApps(settingsLocals, apps, key=None, addBefore=False):
    """
    Extend apps into settings.INSTALLED_APPS.
    INSTALLED_APPS is required existed in settings.

    @param settingsLocals locals() in settings.py.
    @param apps Apps tuple.
    @param key Reference key to insert apps.
    @param addBefore Insert apps before reference key or not.
    """
    installed = settingsLocals['INSTALLED_APPS']
    settingsLocals['INSTALLED_APPS'] = extendUnique(installed, apps,
                                                    referenceKey=key,
                                                    addBefore=addBefore)
                                                    
def extendAuthenticationBackend(settingsLocals, backends, key=None, 
                                addBefore=False):
    """
    Extend apps into settings.AUTHENTICATION_BACKENDS.
    AUTHENTICATION_BACKENDS is required existed in settings.

    @param settingsLocals locals() in settings.py.
    @param backends Backends tuple.
    @param key Reference key to insert apps.
    @param addBefore Insert apps before reference key or not.
    """
    
    if 'AUTHENTICATION_BACKENDS' in settingsLocals:
        installed = settingsLocals['AUTHENTICATION_BACKENDS']
        settingsLocals['AUTHENTICATION_BACKENDS'] = extendUnique(installed, 
                                                                 backends,
                                                             referenceKey=key,
                                                            addBefore=addBefore)
    else:
        settingsLocals['AUTHENTICATION_BACKENDS'] = backends
    
def extendTemplateContextProcessors(settingsLocals, processors, key=None,
                                    addBefore=False):
    """
    Extend processors into settings.TEMPLATE_CONTEXT_PROCESSORS.

    @param settingsLocals locals() in settings.py.
    @param processors Template context processors tuple.
    @param key Reference key to insert processors.
    @param addBefore Insert processors before reference key or not.
    """
    p = settingsLocals.get('TEMPLATES', ())
    for templateSettings in p:

        if templateSettings['BACKEND'] == \
               'django.template.backends.django.DjangoTemplates':
            
            options = templateSettings.get('OPTIONS', ())
            contextProcessors = options.get('context_processors', ())
            options['context_processors'] = extendUnique(contextProcessors, 
                                                         processors,
                                                         referenceKey=key,
                                                         addBefore=addBefore)
            
def extendMiddlewareClasses(settingsLocals, classes, key=None, addBefore=False):
    """
    Extend classes into settings.MIDDLEWARE_CLASSES.

    @param settingsLocals locals() in settings.py.
    @param classes Middleware classes tuple.
    @param key Reference key to insert classes.
    @param addBefore Insert classes before reference key or not.
    """
    if DJANGO_VERSION >= 11100: # django >= 1.11
        middleware = 'MIDDLEWARE'
    else:
        middleware = 'MIDDLEWARE_CLASSES'
        
    p = settingsLocals.get(middleware, ())
    settingsLocals[middleware] = extendUnique(p, classes,
                                                 referenceKey=key,
                                                 addBefore=addBefore)

def extendTemplateDirs(settingsLocals, dirs, key=None, addBefore=False):
    """
    Extend classes into settings.TEMPLATE_DIRS.

    @param settingsLocals locals() in settings.py.
    @param classes TEMPLATES dirs list.
    @param key Reference key to insert classes.
    @param addBefore Insert classes before reference key or not.
    """
    p = settingsLocals.get('TEMPLATES', ())
    for templateSettings in p:
        if templateSettings['BACKEND'] == \
                   'django.template.backends.django.DjangoTemplates':
            templateDirs = templateSettings.get('DIRS', ())
            templateSettings['DIRS'] = extendUnique(dirs, templateDirs,
                                                    referenceKey=key,
                                                    addBefore=addBefore)


def extendDatabaseDefaults(settingsLocals, defaultValues):
    """
    Extend default key-value into settings.DATABASES.

    @param settingsLocals locals() in settings.py.
    @param defaultValues Key-value of database default dict.
    @param key Reference key to insert classes.
    @param addBefore Insert classes before reference key or not.
    """

    p = settingsLocals.get('DATABASES', ())
    defaultSettings = p.get('default', ())
    for key in defaultValues:
        defaultSettings.setdefault(key, defaultValues[key])

class Expiration(object):
    """
    Expiration datetime wrapper.
    """

    def __init__(self, expireMillisecond):
        """
        Constructor.

        @param expireMillisecond Milliseconds of expiration.
        """
        self.expireMillisecond = expireMillisecond
        self.expireDatetime = datetime.datetime.now() + \
            datetime.timedelta(milliseconds=self.expireMillisecond)

    def expired(self):
        """
        Is expired or not?

        @return True if it is expired.
        """
        return datetime.datetime.now() >= self.expireDatetime

class TemporaryKeyGenerator(object):
    """
    Temporary key generator to generate keys for temporary usage and clear it
    while expired.
    It also manages container's content, this generator will provide three
    functions to let you get/set/delete data for container.
    """

    def __init__(self, container, expireMillisecond=60000,
                 factory=lambda: str(time.clock())):
        """
        Constructor.

        @param container The container to store generated temporary key and
                         value.
        @param expireMillisecond Milliseconds of expiration.
        @param factory Key generation factory. It can be any callable thing.
        """
        from django.core.cache.backends.base import BaseCache
        
        super(TemporaryKeyGenerator, self).__init__()       

        self.container = container
        if isinstance(container, dict):
            def _get(k):
                v, exp = self.container[k]
                return None if exp.expired() else v
            self.get = _get

            def _set(k, v, exp):
                self.container[k] = (v, exp)
            self.set = _set

            def _del(k):
                del self.container[k]
            self.delete = _del
        elif isinstance(container, BaseCache):
            self.get = self.container.get

            def _set(k, v, exp):
                self.container.set(k, v, exp.expireMillisecond / 1000)
            self.set = _set

            self.delete = self.container.delete
        elif isinstance(container, list):
            def _get(k):
                for _k, _v, _exp in self.container:
                    if k == _k:
                        return None if _exp.expired() else _v
            self.get = _get

            def _set(k, v, exp):
                self.container.append((k, v, exp))
            self.set = _set

            def _delete(k):
                for i, (_k, _v, _exp) in enumerate(self.container):
                    if k == _k:
                        del self.container[i]
                        return
            self.delete = _delete
        else:
            raise RuntimeError('This type is not supported.')

        self.factory = factory
        self.expireMillisecond = expireMillisecond

    def generate(self, value, *args, **kws):
        """
        Generate temporary key and its expiration time.

        @param value The value you want to put.
        @param *args Arguments that will pass to factory.
        @param **kws Keyword arguments that will pass to factory.
        @return (key, expiration time)
        """
        key = self.factory(*args, **kws)
        exp = Expiration(self.expireMillisecond)

        self.set(key, value, exp)
        return (key, exp)

    def clear(self, criteria=500):
        """
        Clear expired data from given container which contains temporary key
        if the length of container over the maximum criteria.

        @param criteria Criteria.
        @return True if clear operation is done.
        """
        from django.core.cache.backends.base import BaseCache
        
        # Cache backend has its own expiration mechanism.
        if isinstance(self.container, BaseCache):
            return True

        size = len(self.container)
        if size >= criteria:
            if isinstance(self.container, dict):
                for key in list(self.container.keys()):
                    if self.container[key][-1].expired():
                        del self.container[key]
            elif isinstance(self.container, list):
                i = size - 1
                for j in range(size):
                    item = self.container[i]
                    if item[-1].expired():
                        del self.container[i]
                    i -= 1
            else:
                raise NotImplementedError('This type is not supported.')

            return True
        else:
            return False

# https://gist.github.com/527113/307c2dec09ceeb647b8fa1d6d49591f3352cb034
def checkTableExists(table, cursor=None):
    """
    Check database table exists or not.

    @param table Table name.
    @return True if table exists.
    """
    try:
        if not cursor:
            from django.db import connection
            cursor = connection.cursor()
        if not cursor:
            raise Exception
        tables = connection.introspection.get_table_list(cursor)
    except:
        raise Exception("Unable to determine if the table '%s' exists" % table)
    else:
        return table in tables
        
def getDownloadHttpResponse(path, name=None, suffixIncluded=True):
    """
    Read file and return http response with the file content.

    @param path The location of file.
    @param name The name of file downloaded, default including the suffix.
    @param suffixIncluded The boolean value to control whether name contain suffix. Default is True.
    @return HTTP reponse with the file content.
    """
    
    # used to get filename by adding probably correct suffix 
    def returnFilenameWithSuffix(contentType, guessTypeSuccess, nameWithoutSuffix, baseName):
        # if the path contain suffix, use it as priority
        if '.' in baseName:
            fileName = name + '.' + baseName.split('.')[-1]
        else:
            # the contentType guess is accurate
            if guessTypeSuccess:
                # guess suffix by contentType
                suffix = mimetypes.guess_extension(contentType)
                if suffix:
                    fileName = name + suffix
                else:
                    fileName = name
            else:
                fileName = name 
        return fileName
    
    with open(path, 'rb') as f:
        file = f.read()
        
    mimeType = mimetypes.guess_type(path)[0]
    if mimeType is not None:
        contentType = mimeType
        guessTypeSuccess = True
    else:
        contentType = 'application/octet-stream'
        guessTypeSuccess = False
    baseName = os.path.basename(path)
    if name:
        
        if suffixIncluded:
            fileName = name
        else:
            fileName = returnFilenameWithSuffix(contentType, guessTypeSuccess, name, baseName)
            
    else:
        fileName = baseName
    contentType = contentType + ';charset=utf-8'
    response = HttpResponse(
        file,
        content_type=contentType
    )
    # for chinese filename
    safeFileName = urllib.parse.quote(fileName)
    # https://medium.com/@hyWang/
    # %E5%A6%82%E4%BD%95%E5%9C%A8%E4%B8%8D%E5%90%8C%E7%80%8F%E8%A6%BD%E5%99%A8%E4%B8%8B%E8%BC%89%E6%AD%A3%E7%A2
    # %BA%E7%9A%84%E6%AA%94%E6%A1%88%E5%90%8D%E7%A8%B1-content-disposition-7ef13555b1ba
    response['Content-Disposition'] = (
                f"attachment; filename={safeFileName}; filename*={safeFileName}")
    response['Content-Transfer-Encoding'] = 'binary'
    return response
