#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: MyI18n.py 11265 2018-11-23 05:42:44Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-11-23 13:42:44 +0800 (週五, 23 十一月 2018) $
# $Revision: 11265 $

import django

from django.utils.translation import ugettext_lazy
from django import template
from django.template import Library

register = Library()

@register.tag
def languages(parser, token):
    """
    Render Languages.html template but translate all language names first.

    @param parser Template parser.
    @param token Current token.
    @return LanguagesNode instance of class.
    """
    arg = (token.split_contents()[-1] if len(token.split_contents()) > 1
           else None)

    if arg and arg[0] == arg[-1] and arg[0] in ('"', "'"):
        arg = arg[1:-1]

    return LanguagesNode(*([arg] if arg else []))

class LanguagesNode(template.Node):
    """
    Process how to render LanguagesNode tag.
    """

    def __init__(self, action='/i18n/setlang/'):
        """
        Constructor.

        @param action Action variable of the languages setting form.
        """
        self.action = action

    def render(self, context):
        """
        Render.

        @param context Context instance.
        @return Rendered page.
        """
        langs = []

        if 'LANGUAGES' not in context:
            raise LookupError('You must install LocaleMiddleware first!')

        for lang in context['LANGUAGES']:
            langs.append((lang[0], ugettext_lazy(lang[1])))

        context['LANGUAGES'] = langs
        context['languages__action'] = self.action


        t = template.loader.get_template('Languages.html')

        if django.__version__ == '1.8':
            return t.render(context)
        else:
            return t.render(context.flatten())

from django.utils import translation
from django.conf import settings

@register.simple_tag
def jsi18n(js):
    """
    Generate script tag for i18n according current language.

    @param js Javascript file name.
    @return Generated javascript tag.
    """
    code = translation.get_language()
    s = js.split('.')
    prefix = '.'.join(s[:-1])
    ext = s[-1]
    return ('<script type="text/javascript" src="%s%s_%s.%s"></script>' %
            (settings.MEDIA_URL, prefix, code, ext))
