#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: cms_plugins.py 13373 2022-01-10 07:18:23Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2022-01-10 15:18:23 +0800 (Mon, 10 Jan 2022) $
# $Revision: 13373 $

import re
import requests
import json

from bs4 import BeautifulSoup
from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool
from django.utils.translation import ugettext as _
from django.conf import settings
from django.utils.safestring import mark_safe
from django.template.loader import select_template
from django.urls import reverse

from Iuppiter.Logging import createLogger
from Zephyrus.facebook import settings as zephyrusSettings
from Zephyrus.facebook.models import (
    FacebookFan, FacebookPhoto, ShareLike, FacebookSettings)
from Zephyrus.facebook.models import FanPage as FanPageModel

logger = createLogger(__name__)

class Subattachment(object):
    def __init__(self, title, subattachmentType, img, url):
        self.title = title
        self.type = subattachmentType
        self.img = img
        self.url = url


class FanPage(object):
    def __init__(self, pageId, createdTime, msg, mainImgUrl, postType, url, 
                 subattachments):
        self.id = pageId
        self.createdTime = createdTime
        self.type = postType
        self.img = mainImgUrl
        self.msg = msg
        self.url = url
        self.subattachments = subattachments
        
class Photo(object):
    def __init__(self, name, url, link):
        self.name = name
        self.url = url
        self.link = link
        
def getAccessToken():
    # accessToken = '%s|%s' % (
        # settings.SOCIAL_AUTH_FACEBOOK_KEY, 
        # settings.SOCIAL_AUTH_FACEBOOK_SECRET
    # )
    config, created = FacebookSettings.objects.get_or_create()
    return config.accessToken
    
def getAuthorizationMsg():
    popScript = '''
<script>
    function popupCallback(str){
        location.reload(true);
    }
    $('#authorization').click(function(){
        var popupWindow = window.open("%s", "popupWindow", "status=1, height=600, width=600, toolbar=0, resizable=0");
        popupWindow.window.focus();
    })
</script>
''' % reverse("facebookAccessTokenCreate")

    if not getAccessToken():
        accessMessage = _(
            '<p>We are not authorized to read your fan information, please use the following link:</p>'
            '<a href="#" id="authorization">Allow this plugin to read your fan information</a>'
            '<p>If we are not authorized, This will cause restrictions on access to FB fan-specific information '
            '(e.g. filtering function).</p>'
            )
    else:
        accessMessage = _(
            '<p>We have obtained your fan special authorization and'
            ' can read your fan special information normally</p>'
            '<a href="#" id="authorization">Re-verify</a>')
            
    return accessMessage + popScript
        
def getFanPageId(accessToken, url):
    # get fan page id
    fanPageUrl = url
    if not fanPageUrl.endswith('/'):
        fanPageUrl += '/'
    req = requests.get(
        ("https://graph.facebook.com/v6.0/%s?access_token=%s&debug=all"
         "&format=json&method=get&pretty=0&suppress_http_code=1") % (
         fanPageUrl, accessToken), proxies=zephyrusSettings.ZEPHYRUS_FACEBOOK_PROXY)
    text = req.text

    data = json.loads(text)
    
    if data.get('name') and data.get('id'):
        fanPageId = json.loads(text)['id']
        return fanPageId
    else:
        return None
        
def getFanPage(url, limit):
    try:
        # get data
        # req = requests.get(
            # "%sposts/?ref=page_internal&_fb_noscript=1" % url, proxies=zephyrusSettings.ZEPHYRUS_FACEBOOK_PROXY)  
        req = requests.get(
            "%sposts/?ref=page_internal&_fb_noscript=1" % url)
        content = req.text
        soup = BeautifulSoup(content, 'html.parser')
        
        # create object
        fpList = []
        for i, div in enumerate(soup.find_all('div', class_="_4-u2 _4-u8"), 1):
            pageId = i
            
            # createdTime
            createdTime = div.find_all('span', class_="timestampContent")[0].text
            
            # message
            msg = div.find_all('div', class_='text_exposed_root')
            if msg:
                msg = mark_safe(str(div.find_all('div', class_='text_exposed_root')[0]))
            else:
                msg = div.find_all('div', {'data-testid': 'post_message'})
                if msg:
                    TextWithBackground = msg[0].find_all('div', {'style':re.compile("background-image")})
                    if TextWithBackground:
                        # for those text with background, only use extract the text without style
                        msg = mark_safe(TextWithBackground[0].find_all('span', {'aria-hidden': "true"})[0].text)
                    else:
                        msg = mark_safe(str(msg[0]))
                else:
                    msg = ''
            
            # image
            images = div.find_all('img', class_="scaledImageFitWidth img")
            if images:
                mainImg = div.find_all('img', class_="scaledImageFitWidth img")[0].get("data-src", "#")
            else:
                # there might be more than one picture in one post
                images = div.find_all('img', class_="scaledImageFitHeight img")
                if images:
                    mainImg = images[0].get("data-src", "#")
                else:
                    mainImg = None
               
            # type
            postType = "abc"
            
            # link
            links = div.find_all('a', class_="_5pcq")
            if links:
                link = "https://www.facebook.com%s" % div.find_all('a', class_="_5pcq")[0].get("href", "#")
            else:
                continue
                
            subattachments = []
            fp = FanPage(pageId, createdTime, msg, mainImg, postType, link, subattachments)
            fpList.append(fp)
            if len(fpList) == limit:
                break
       
        return fpList
    except Exception as e:
        logger.error(str(e))
        raise
        return [] # no post

class FacebookFanPlugin(CMSPluginBase):
    model = FacebookFan
    module = _("Zephyrus Facebook")
    name = _("Facebook Fan Page")
    render_template = zephyrusSettings.ZEPHYRUS_FACEBOOK_FAN_DEFAULT_TEMPLATE

    def get_fieldsets(self, request, obj=None):
        accessMessage = getAuthorizationMsg()
    
        fieldsets = (
            (None, {
                'description': _('<h2>Main:</h2>%s') % accessMessage,
                'fields': ('url', 'limit',),
            }),
            (None, {
                'description': _('<h2>Choose article display mode:</h2>'),
                'fields': ('unknownDisplay', 'videoDisplay', 'albumDisplay', 
                           'shareDisplay', 'commerceDisplay', 
                           'onlyMessageDisplay',),
            }),
            (None, {
                'description': _('<h2>Other:</h2>'),
                'fields': ('renderTemplate',),
            }),
        )
        return fieldsets

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            zephyrusSettings.ZEPHYRUS_FACEBOOK_FAN_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        accessToken = getAccessToken()
        if not accessToken:
            fpList = getFanPage(instance.url, instance.limit)
           
            context.update({
                'fpList': fpList,
                'crawler': True,
            })
            
            return context
        else:
            context.update({
                'crawler': False,
            })
            fanPageUrl = instance.url
            fanPageId = getFanPageId(accessToken, instance.url)

            # article type filter
            articleTypes = []
            onlyMsg = instance.onlyMessageDisplay
            if instance.unknownDisplay:
                articleTypes.append('unknown')
            if instance.videoDisplay:
                articleTypes.append('video') 
            if instance.albumDisplay:
                articleTypes.append('photo')
                articleTypes.append('album')
            if instance.shareDisplay:
                articleTypes.append('share')
            if instance.commerceDisplay:
                articleTypes.append('commerce')
            
            if not fanPageId:
                context.update({
                    'error': _('Url must conform correct format, '
                               'EX: https://www.facebook.com/xxx/'),
                })
                return context
            
            # get data
            query = ("posts?fields=message,attachments,permalink_url"
                     "&limit=100")
            req = requests.get(
                'https://graph.facebook.com/v6.0/%s/%s&access_token=%s' % (
                fanPageId, query, accessToken)
                )   

            if json.loads(req.text).get('error', {}).get('message'):
                context.update({
                    'error': json.loads(req.text).get('error', {}).get('message'),
                })
                return context

            nextPage = None

            fpList = []

            while len(fpList) < instance.limit: 
                if nextPage:
                    if nextPage == 'End':
                        break
                    req = requests.get(nextPage)

                content = json.loads(req.text).get('data')
                nextPage = json.loads(req.text).get('paging', {}).get('next', 'End')

                # create objects
                for data in content:
                    pageId = data.get('id')
                    createdTime = data.get('created_time')
                    link = data.get('permalink_url')
                    msg = mark_safe(data.get('message', '').replace('\n', '<br />'))
                    mainImg = None
                    postType = 'unknown'

                    if not msg and onlyMsg:
                        continue

                    subattachments = []
                    if 'attachments' in data:
                        data2 = data['attachments']['data'][0]

                        mainImg = data2.get("media", {}).get("image", {}).get("src")
                        postType = data2.get('type', 'article')

                        if "subattachments" in data2:
                            for data3 in data2["subattachments"]['data']:
                                title = data3.get('title')
                                subattachmentType = data3.get('type')
                                url = data3.get('url')
                                image = data3.get(
                                    "media", {}).get("image", {}).get("src")

                                subattachments.append(
                                    Subattachment(
                                        title, subattachmentType, image, url))

                    addToFpList = False
                    for t in articleTypes:
                        if t in postType:
                            addToFpList = True

                    if addToFpList:
                        fpList.append(
                            FanPage(
                                pageId, createdTime, msg, mainImg, postType, link, 
                                subattachments))

                        if len(fpList) == instance.limit:
                            break
                
            context.update({
                'fpList': fpList,
            })
            
            return context
        
def getFanPhoto(url, limit):
    try:
        # get data
        req = requests.get(url, proxies=zephyrusSettings.ZEPHYRUS_FACEBOOK_PROXY)  
        content = req.text
        startStr = '''"media":{"edges":['''
        endStr = '''],"page_info":'''
        content = "[%s]" % content[content.find(startStr) + len(startStr):content.find(endStr)]
        data = json.loads(content)
        
        # create objects
        photoList = []
        for d in data:
            node = d.get("node")
            if not node:
                continue
                
            name = None
            url = None
            link = None
            if not 'image' in node:
                continue
           
            url = node.get("image", {}).get("uri", '#')
            #name = mark_safe(node.get("album", {}).get("name", '').replace('\n', '<br />'))
            link = node.get("url", '#')
            
            photoList.append(Photo(name, url, link))
            if len(photoList) == limit:
                break
                
        return photoList
    except Exception as e:
        logger.error(str(e))
        raise
        return [] # no photo
        
class FacebookPhotoPlugin(CMSPluginBase):
    model = FacebookPhoto
    module = _("Zephyrus Facebook")
    name = _("Facebook Fan Photo")
    render_template = zephyrusSettings.ZEPHYRUS_FACEBOOK_PHOTO_DEFAULT_TEMPLATE
    
    def get_fieldsets(self, request, obj=None):
        accessMessage = getAuthorizationMsg()
    
        fieldsets = (
            (None, {
                'description': _('<h2>Main:</h2>%s') % accessMessage,
                'fields': ('url', 'limit',),
            }),
            (None, {
                'description': _('<h2>Other:</h2>'),
                'fields': ('renderTemplate',),
            }),
        )
        return fieldsets

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            zephyrusSettings.ZEPHYRUS_FACEBOOK_PHOTO_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        accessToken = getAccessToken()
        if not accessToken:
            photoList = getFanPhoto(instance.url, instance.limit)
            context.update({
                'photoList': photoList,
                'crawler': True,
            })
            return context
        else:
            context.update({
                'crawler': False,
            })
            albumId = None
            fanPageId = getFanPageId(accessToken, instance.url)
            if 'album_id=' in instance.url:
                index = instance.url.index('album_id=') + len('album_id=')
                albumId = instance.url[index:]
            elif 'set=' in instance.url:
                index = instance.url.index('set=') + len('set=')
                albumId = str(instance.url[index:]).strip('a.')
            else:
                if not fanPageId:
                    
                    context.update({
                        'error': _('Url must conform correct format, '
                                   'EX: https://www.facebook.com/pg/xxx/photos/'
                                   '?tab=album&album_id=xxx'),
                    })
                    return context

            if albumId:
                query = "?fields=photos.limit(100){name,images,link}"
                req = requests.get(
                    'https://graph.facebook.com/v6.0/%s/%s&access_token=%s' % (
                    albumId, query, accessToken)
                    )
            else:
                query = ("photos/uploaded?fields=images,name,link&limit=%d" % 
                         instance.limit)
                if not fanPageId:
                    
                    context.update({
                        'error': _('Url must conform correct format, '
                                   'EX: https://www.facebook.com/xxx/'),
                    })
                    return context
                req = requests.get(
                    'https://graph.facebook.com/v6.0/%s/%s&access_token=%s' % (
                    fanPageId, query, accessToken)
                )     
            
            text = json.loads(req.text)
            
            if 'photos' in text and 'data' in text['photos']:
                content = text['photos']['data']
            elif 'data' in text:
                content = text['data']
            else:
                
                context.update({
                    'error': _("Album data doesn't be searched"),
                })
                return context
                    
            
            # create objects
            photoList = []
            for data in content:
                name = None
                url = None
                link = None
                if not 'images' in data:
                    continue
                for data2 in data['images']:
                    if 'source' in data2:
                        url = data2['source']
                        break
                if 'name' in data:
                    name = mark_safe(data['name'].replace('\n', '<br />'))
                if 'link' in data:
                    link = data['link']
                photoList.append(Photo(name, url, link))
                if len(photoList) == instance.limit:
                    break
            
            context.update({
                'photoList': photoList,
            })
            
            return context

class ShareLikePlugin(CMSPluginBase):
    model = ShareLike
    module = _("Zephyrus Facebook")
    name = _("Facebook Share Like")
    render_template = "facebook/ShareLike.html"

    def render(self, context, instance, placeholder):
        request = context['request']
        url = request.build_absolute_uri(request.path)

        if not instance.appId:
            appId = zephyrusSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID
        else:
            appId = instance.appId

        context.update({
            'url': url,
            'obj': instance,
            'appId': appId,
        })

        return context

class FanPagePlugin(CMSPluginBase):
    model = FanPageModel
    module = _("Zephyrus Facebook")
    name = _("Facebook Insert Fan Page")
    render_template = "facebook/FanPage.html"

    def render(self, context, instance, placeholder):
        if not instance.appId:
            appId = zephyrusSettings.ZEPHYRUS_FACEBOOK_DEFAULT_APP_ID
        else:
            appId = instance.appId

        context.update({
            'obj': instance,
            'appId': appId,
        })
        return context

plugin_pool.register_plugin(FacebookFanPlugin)
plugin_pool.register_plugin(FacebookPhotoPlugin)
plugin_pool.register_plugin(ShareLikePlugin)
plugin_pool.register_plugin(FanPagePlugin)
