#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Patch.py 11526 2019-05-07 08:36:46Z Andy $
#
# Copyright (c) 2019 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $ (last)
# $Date: 2019-05-07 17:36:46 +0900 (週二, 07 五月 2019) $
# $Revision: 11526 $
#
# Contributors:
#  Bear
#  Yachu

import json
import base64
import hashlib
import os
import six
import urllib

from django.conf import settings
from django.urls import reverse
from django.utils.html import conditional_escape 
from django.contrib.sites.shortcuts import get_current_site
from django.contrib.auth.decorators import permission_required
from django.http import HttpResponse, Http404
from django.core.mail import send_mail
from des.models import DynamicEmailConfiguration

import statsd
from django_statsd.utils import get_client

from Iuppiter.Logging import createLogger
from Iuppiter.Encryption import Encryptor

SECRET_USE_FIELDS = ['emails', 'type']

STATSD_COUNTER = 'counter'
STATSD_TIMER_START = 'timer_start'
STATSD_TIMER_STOP = 'timer_stop'
STATSD_GAUGE = 'gauge'
STATSD_TYPES = [
    STATSD_COUNTER, STATSD_TIMER_START, STATSD_TIMER_STOP, STATSD_GAUGE]
    
counter = get_client("Iuno.metrics.counter", class_=statsd.Counter)
timer = get_client("Iuno.metrics.timer", class_=statsd.Timer)
gauge = get_client("Iuno.metrics.gauge", class_=statsd.Gauge)

logger = createLogger(__name__)
    
def getClientIP(request):
    xForwardedFor = request.META.get('HTTP_X_FORWARDED_FOR')
    if xForwardedFor:
        ip = xForwardedFor(',')[0]
    else:
        ip = request.META.get('REMOTE_ADDR')
    return ip

def getHashKey(keys):
    data = settings.SECRET_KEY
    for key, val in keys.items():
        data += str(key) + str(val)
        
    if six.PY3:
        hashKey = hashlib.sha256(data.encode()).hexdigest()
    else:
        hashKey = hashlib.sha256(data).hexdigest()
        
    return hashKey
    
def getData(request):
    keys = {
        k: request.GET.get(k, '') 
        for k in request.GET if not k in SECRET_USE_FIELDS
    }
    
    secret = {
        k: request.GET.get(k, '') 
        for k in request.GET if k in SECRET_USE_FIELDS
    }
    
    if not secret.get('type', STATSD_COUNTER) in STATSD_TYPES:
         raise Http404()
    else:
        secret['type'] = secret.get('type', STATSD_COUNTER)
    
    return (keys, secret)
    
def encryptSecret(secret):
    secret = json.dumps(secret) 
    if six.PY3:
        encryptor = Encryptor(settings.SECRET_KEY.encode())
        secret = encryptor.encrypt(secret.encode())
    else:
        encryptor = Encryptor(settings.SECRET_KEY)
        secret = encryptor.encrypt(secret)
    secret = base64.b64encode(secret) # urlsafe_b64encode
    secret = secret.decode() 
    if six.PY3:
        secret = urllib.parse.quote(secret) # ' '
    else:
        secret = urllib.quote(secret)
    return secret
    
def decryptSecret(secretKey):
    if six.PY3:
        encryptor = Encryptor(settings.SECRET_KEY.encode())
        secret = urllib.parse.unquote(secretKey)
        secret = base64.b64decode(secret.encode())
        secret = encryptor.decrypt(secret)
    else:
        encryptor = Encryptor(settings.SECRET_KEY)
        secret = urllib.parse.unquote(secretKey)
        secret = base64.b64decode(secret)
        secret = encryptor.decrypt(secret)
        
    secret = json.loads(secret)
    
    return secret
    

def sendStatsd(request):
    # referer
    referer = request.META.get('HTTP_REFERER')
    site = get_current_site(request)
    
    # img file
    imagePath = os.path.join(os.path.dirname(__file__), 'static', "1px.png")
    with open(imagePath, "rb") as f:
        image = f.read()

    try:
        if not referer:
            logger.error("referer error: %s" % referer)
            raise Http404()
        if not site.domain in referer:
            logger.error("referer error: %s" % referer)
            raise Http404()
         
        # decrypt data
        keys, secret = getData(request)
        checksum = keys.pop('checksum', '')
        secretKey = keys.pop('secret', '')
        secret = decryptSecret(secretKey)
        allData = {}
        allData.update(keys)
        allData.update(secret)
       
        # certification
        hashKey = getHashKey(allData)
        if not hashKey == request.GET.get('checksum'):
            logger.error("checksum error: %s" % request.build_absolute_uri())
            raise Http404()
        
        # statsd
        # https://pythonhosted.org/python-statsd/
        if secret.get('type') == STATSD_COUNTER:
            for key, val in keys.items():
                try:
                    val = int(val)
                except Exception as e:
                    val = 1 # counter + 1
                counter.increment(key, val)
        elif secret.get('type') == STATSD_TIMER_START:
            print("timer_start")
            timer.start()
        elif secret.get('type') == STATSD_TIMER_STOP:
            print("timer_stop")
            
        elif secret.get('type') == STATSD_GAUGE:
            for key, val in keys.items():
                # assert isinstance(value, compat.NUM_TYPES)
                try:
                    val = int(val)
                except Exception as e:
                    val = 1
                gauge.send(key, val)
        else:
            logger.error("statsd type error: %s" % secret.get('type'))
            raise Http404()
            
        # send emails
        emails = secret.get('emails')
        keyData = "&".join(["%s=%s" % (k, v) for k, v in keys.items()])
        if secret.get('type') == STATSD_COUNTER:
            keyData.replace("=", '')
        emailTitle = "[Iuno.metrics] %s - %s" % (site.name, keyData)
        statsdContent = "\n".join(
            ["%s: %s" % (k, v) for k, v in allData.items()])  
        emailContent = "IP: %s\nReferer: %s\n%s" % (
            getClientIP(request), referer, statsdContent)
        
        if emails:
            emails = emails.split(",")
            send_mail(
                emailTitle,
                emailContent,
                DynamicEmailConfiguration.objects.get().from_email,
                emails,
                fail_silently=False,
            )
            
            
        
        return HttpResponse(image, content_type="image/png")
    except Exception as e:
        
        return HttpResponse(image, content_type="image/png")

  
@permission_required('metrics.get_statsd', raise_exception=True)
def getStatsdImage(request):
    
    # get data
    keys, secret = getData(request)
    if not keys:
        raise Http404()
    
    # create hashKey and secretKey
    allData = {}
    allData.update(keys)
    allData.update(secret)
   
    hashKey = getHashKey(allData)
    secretKey = encryptSecret(secret)
    
    # create img tag
    keyStr = "&".join(["%s=%s" % (k, v) for k, v in keys.items()])
    url1 = "<img src=\"%s?%s%s%s\"/>" % (
        reverse("sendStatsd"), keyStr, 
        "&secret=%s" % secretKey,
        "&checksum=%s" % hashKey
    )
    url2 = "\n<img src=\"{% url 'sendStatsd' %}" + "?%s%s%s\"/>" % (
        keyStr, 
        "&secret=%s" % secretKey,
        "&checksum=%s" % hashKey
    )
    
    temp = "%s<br>%s" % (conditional_escape(url1), conditional_escape(url2))

    return HttpResponse(temp)


