#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: AttachSettingsTest.py 11772 2019-11-01 08:50:14Z Andy $
#
# Copyright (c) 2016 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $
# $Date: 2019-11-01 17:50:14 +0900 (週五, 01 十一月 2019) $
# $Revision: 11772 $

import os
import warnings

from django.test import TestCase

from Iuno import attachSettings


try:
    from Iuppiter.DjangoUtil import DJANGO_VERSION
                      
    if DJANGO_VERSION >= 11100: # django >= 1.11
        MIDDLEWARE = 'MIDDLEWARE'
    else:
        MIDDLEWARE = 'MIDDLEWARE_CLASSES'
        
except ImportError:
    warnings.warn("Unable to import django, many Iuno packages require it.")

class TestAttachSettings(TestCase):

    REQUIRED_SETTINGS = {
        'ROOT_URLCONF' : "urls",
        'SECRET_KEY' : "456456",
        '__file__' : os.path.dirname(
            os.path.dirname(os.path.abspath(__file__))),
        'IUNO_SITE_DOMAIN_NAMES': {
                1: '127.0.0.1:8000',
                2: 'www.caramelo-jd.com',
                3: 'www.caramelo-jd.com',           
            },
    }

    def testPopulateEnvToSettings(self):
        """
        Test for populateEnvToSettings function, see #2747.        
        """
        settings = TestAttachSettings.REQUIRED_SETTINGS.copy()
        
        attachSettings(settings)        
        self.assertEqual(settings['DEBUG'], True)
        
        os.environ['DEBUG'] = 'false'
        attachSettings(settings)    
        self.assertEqual(settings['DEBUG'], False)        
        
        os.environ['DEBUG'] = 'True'
        attachSettings(settings)    
        self.assertEqual(settings['DEBUG'], True)        
        
        os.environ.pop('DEBUG')
        
        settings = TestAttachSettings.REQUIRED_SETTINGS.copy()
        os.environ['SERVER_MODE'] = 'production'        
        attachSettings(settings)    
        self.assertEqual(settings['DEBUG'], False)   
        from Iuno import PRODUCTION
        self.assertEqual(settings['SERVER_MODE'], PRODUCTION)
        
        os.environ.pop('SERVER_MODE')
        
        settings = TestAttachSettings.REQUIRED_SETTINGS.copy()
        os.environ['ADMIN_EMAIL'] = 'devops@nuwainfo.com'
        os.environ['ALLOWED_HOST'] = '.example.com'
        attachSettings(settings)    
        self.assertEqual(settings['ADMINS'][-1], 
            ('devops', os.environ['ADMIN_EMAIL']))
        self.assertTrue(os.environ['ALLOWED_HOST'] in settings['ALLOWED_HOSTS'])
        
        settings = TestAttachSettings.REQUIRED_SETTINGS.copy()
        os.environ['DATABASE_URL'] = (
            'postgres://'
            'user3123:passkja83kd8@'
            'ec2-117-21-174-214.compute-1.amazonaws.com:6212/db982398'
        )
        attachSettings(settings)    
        self.assertEqual(settings['DATABASES']['default'], {
            'ENGINE': 'django.db.backends.postgresql_psycopg2',
            'NAME': 'db982398',
            'USER': 'user3123',
            'PASSWORD': 'passkja83kd8',
            'HOST': 'ec2-117-21-174-214.compute-1.amazonaws.com',
            'PORT': 6212,   
        })        
        
        os.environ.update({
            'DEFAULT_FROM_EMAIL': 'devops@nuwainfo.com',
            'EMAIL_PORT': '25',
            'EMAIL_USE_TLS': 'False',
        })
        attachSettings(settings)    
        self.assertEqual(settings['DEFAULT_FROM_EMAIL'], 
                         os.environ['DEFAULT_FROM_EMAIL'])
        self.assertEqual(settings['EMAIL_PORT'], eval(os.environ['EMAIL_PORT']))
        self.assertEqual(settings['EMAIL_USE_TLS'], False)
        
        settings = TestAttachSettings.REQUIRED_SETTINGS.copy()
        os.environ['IUNO_CLOUD_ENABLE'] = 'True'
        os.environ['IUNO_CLOUD_SERVICES'] = "()"
        attachSettings(settings)    
        self.assertEqual(settings['IUNO_CLOUD_ENABLE'], True)
        self.assertEqual(settings['IUNO_CLOUD_SERVICES'], ())
    
    def testEmptyTemplatesSettings(self):
    
        settingsContent = TestAttachSettings.REQUIRED_SETTINGS.copy()
            
        settingsContent.update({
                               
            'INSTALLED_APPS': (
                'django.contrib.admin',
                'django.contrib.auth',
                'django.contrib.contenttypes',
                'django.contrib.sessions',
                'django.contrib.messages',
                'django.contrib.staticfiles',
            ),
            MIDDLEWARE: (
                'django.contrib.sessions.middleware.SessionMiddleware',
                'django.middleware.common.CommonMiddleware',
                'django.middleware.csrf.CsrfViewMiddleware',
                'django.contrib.auth.middleware.AuthenticationMiddleware',
                ('django.contrib.auth.middleware.'
                 'SessionAuthenticationMiddleware'),
                'django.contrib.messages.middleware.MessageMiddleware',
                'django.middleware.clickjacking.XFrameOptionsMiddleware',
                'django.middleware.security.SecurityMiddleware',
            ),
        })

        attachSettings(settingsContent)
            
        from django.conf import settings
        
        projectPath = os.path.abspath(os.path.dirname(settings.BASE_DIR))
        
        templatesSettings = {
            'TEMPLATES' : [
                {
                    'BACKEND': ('django.template.backends.django.'
                                'DjangoTemplates'),
                    'DIRS': [
                        os.path.join(projectPath, "templates"),
                    ],
                    'APP_DIRS': True,
                    'OPTIONS': {
                        'context_processors': [
                            'django.template.context_processors.debug',
                            'django.template.context_processors.request',
                            'django.contrib.auth.context_processors.auth',
                            ('django.contrib.messages.context_processors.'
                            'messages'),
                        ],
                    'debug': True,
                    },
                    
                },
            ],
        }
        
        self.assertEqual(settingsContent['TEMPLATES'],
                         templatesSettings['TEMPLATES'])
        
    def testMissingTemplatesSettings(self):
        from django.conf import settings
        
        projectPath = os.path.abspath(os.path.dirname(settings.BASE_DIR))
        
        settingsContent = TestAttachSettings.REQUIRED_SETTINGS.copy()
        
        default = {              
            'INSTALLED_APPS': (
                'django.contrib.admin',
                'django.contrib.auth',
                'django.contrib.contenttypes',
                'django.contrib.sessions',
                'django.contrib.messages',
                'django.contrib.staticfiles',
            ),
            'TEMPLATES': [
                {
                    'BACKEND': ('django.template.backends.django.'
                                'DjangoTemplates'),
                },
            ],     
            MIDDLEWARE: (
                'django.contrib.sessions.middleware.SessionMiddleware',
                'django.middleware.common.CommonMiddleware',
                'django.middleware.csrf.CsrfViewMiddleware',
                'django.contrib.auth.middleware.AuthenticationMiddleware',
                ('django.contrib.auth.middleware.'
                 'SessionAuthenticationMiddleware'),
                'django.contrib.messages.middleware.MessageMiddleware',
                'django.middleware.clickjacking.XFrameOptionsMiddleware',
                'django.middleware.security.SecurityMiddleware',
            )
        }
        
        settingsContent.update(default)
        
        
        settingsContent['__file__'] = os.path.dirname(
            os.path.dirname(os.path.abspath(__file__)))
            
        attachSettings(settingsContent)
            
        templatesSettings = {
            'TEMPLATES' : [
                {
                    'BACKEND': ('django.template.backends.django.'
                                'DjangoTemplates'),
                    'DIRS': [
                        os.path.join(projectPath, "templates"),
                    ],
                    'APP_DIRS': True,
                    'OPTIONS': {
                        'context_processors': [
                            'django.template.context_processors.debug',
                            'django.template.context_processors.request',
                            'django.contrib.auth.context_processors.auth',
                            ('django.contrib.messages.context_processors.'
                             'messages'),
                        ],
                    'debug': True,
                    },
                    
                },
            ],
        }
        
        self.assertEqual(settingsContent['TEMPLATES'],
                         templatesSettings['TEMPLATES'])

    def testHttpsSettings(self):
        settingsContent = TestAttachSettings.REQUIRED_SETTINGS.copy()

        # Test if we specify SECURE_PROXY_SSL_HEADER in settings.py, 
        # it should follow the settings value.

        settingsContentForHTTP = settingsContent.copy()
        settingsContentForHTTP['SECURE_PROXY_SSL_HEADER'] = (
                                            'HTTP_X_FORWARDED_PROTO', 'http')
        attachSettings(settingsContentForHTTP)

        httpSettings = {
            'SECURE_PROXY_SSL_HEADER': ('HTTP_X_FORWARDED_PROTO', 'http'),
        }

        self.assertEqual(settingsContentForHTTP['SECURE_PROXY_SSL_HEADER'],
                         httpSettings['SECURE_PROXY_SSL_HEADER'])

        # Test if we set HTTPS=on in env, it should set  HTTP_X_FORWARDED_PROTO
        # to https if we didn't specify the value in settings.py, otherwise, it 
        # it should follow the value in settings.py.

        os.environ["HTTPS"] = "on"
        settingsContentForHTTPS = settingsContent.copy()

        attachSettings(settingsContentForHTTP)
        attachSettings(settingsContentForHTTPS)

        httpsSettings = {
            'SECURE_PROXY_SSL_HEADER': ('HTTP_X_FORWARDED_PROTO', 'https'),
        }

        self.assertEqual(settingsContentForHTTPS['SECURE_PROXY_SSL_HEADER'],
                    httpsSettings['SECURE_PROXY_SSL_HEADER'])

        self.assertEqual(settingsContentForHTTP['SECURE_PROXY_SSL_HEADER'],
                         httpSettings['SECURE_PROXY_SSL_HEADER'])