#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: StatisticsHandler.py 11616 2019-06-07 02:15:46Z Andy $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $
# $Date: 2019-06-07 11:15:46 +0900 (週五, 07 六月 2019) $
# $Revision: 11616 $

from datetime import timedelta, datetime

from django.db.models import Max as modelsMax
from django.http import Http404
from django.utils.translation import ugettext_lazy as _

from Iuppiter.Encoding import _unicode

from emencia.django.newsletter.models import Newsletter
from emencia.django.newsletter.models import ContactMailingStatus
from collections import OrderedDict

_STATUS = [
    'deliverie', 'error', 'invaild', 'open', 'openOnSite', 'click',
    'unsubscribe', 'bounce', 'complain', 'reject', 'sentTest'
]

STATUS_DICT = dict(ContactMailingStatus.STATUS_CHOICES)
STATUS_NAMES = STATUS_DICT.values()
STATUS_CODE = STATUS_DICT.keys()

from django.db import connection
from django.conf import settings
# check if strftime function is in current mysql
dbName = settings.DATABASES['default']['NAME']
setNewMysqlFunc = True

if settings.DATABASE_ENGINE == 'mysql':
    with connection.cursor() as cursor:
        cursor.execute(
            """
                SELECT ROUTINE_NAME
                FROM INFORMATION_SCHEMA.ROUTINES
                WHERE ROUTINE_TYPE="FUNCTION"
                AND ROUTINE_SCHEMA="%s";
            """ % (dbName, )
        )
        row = cursor.fetchall()

        for ele in row:
            if "strftime" in ele:
                setNewMysqlFunc = False

        if setNewMysqlFunc:
            cursor.execute(
                """
                    CREATE FUNCTION strftime(format varchar(255), d date)
                    RETURNS varchar(64)
                    LANGUAGE SQL
                    DETERMINISTIC
                    COMMENT 'synonym for date_format'
                    return date_format(d, format) ;
                """
        )

def getDateRange(startDate, endDate):
    """
    Get range of dates.
    """
    if isinstance(startDate, str) or isinstance(startDate, type(b'')):
        if startDate != "NaN":
            startDate = datetime.strptime(startDate, '%Y-%m-%d %H:%M:%S')
    if isinstance(endDate, str):
        if endDate != "NaN":
            endDate = datetime.strptime(endDate, '%Y-%m-%d %H:%M:%S')

    deltaDays = endDate - startDate
    daysAmount = deltaDays.days + 1

    def _createDate(daysAmount):
        days = [startDate.strftime('%Y-%m-%d')]
        for i in range(1, daysAmount):
            d = startDate + timedelta(days=i)
            d = d.strftime('%Y-%m-%d')
            days.append(d)

        class DateRange(object):
            def __init__(self, days):
                self._days = days
                self.startDate = '%s 00:00:00' % days[0]
                self.endDate = '%s 23:59:59' % days[-1]

        return DateRange(days)

    return _createDate(daysAmount)

#def countStatus(queryset):
#    """
#    Calculate the status number of occurrences.
#    ex: (c.newsletter.id, c.status, c.creation_date)
#
#    SENT_TEST = -1
#    SENT = 0
#    ERROR = 1
#    INVALID = 2
#    OPENED = 4
#    OPENED_ON_SITE = 5
#    LINK_OPENED = 6
#    UNSUBSCRIPTION = 7
#
#    BOUNCED = 8
#    COMPLAINT = 9
#    REJECTED = 10
#
#    @param queryset The queryset of ContactMailingStatus.
#    """
#    sentTestNumber = 0
#    sentNumber = 0
#    errorLogs = set()
#    invaildLogs = set()
#    openLogs = set()
#    openOnSiteLogs = set()
#    clickLogs = set()
#    unsubscriptionNumber = 0
#
#    bouncedLogs = set()
#    complainLogs = set()
#    rejectedLogs = set()
#
##    subscribesList = []
#
#    for (id, status, logTime) in queryset:
#        if status == ContactMailingStatus.SENT_TEST:
#            sentTestNumber += 1
#        elif status == ContactMailingStatus.SENT:
#            sentNumber += 1
#        elif status == ContactMailingStatus.ERROR:
#            errorLogs.add(id)
#        elif status == ContactMailingStatus.INVALID:
#            invaildLogs.add(id)
#        elif status == ContactMailingStatus.OPENED:
#            openLogs.add(id)
#        elif status == ContactMailingStatus.OPENED_ON_SITE:
#            openOnSiteLogs.add(id)
#        elif status == ContactMailingStatus.LINK_OPENED:
#            clickLogs.add(id)
#        elif status == ContactMailingStatus.UNSUBSCRIPTION:
#            unsubscriptionNumber += 1
#        ##
#        elif status == ContactMailingStatus.BOUNCED:
#            bouncedLogs.add(id)
#        elif status == ContactMailingStatus.COMPLAINT:
#            complainLogs.add(id)
#        elif status == ContactMailingStatus.REJECTED:
#            rejectedLogs.add(id)
##        elif status == ContactMailingStatus.STATUS_SUBSCRIBE:
##            subscribesList.append(id)
#
#    return {
#        'sentTest': sentTestNumber,
#        'deliverie': sentNumber,
#        'error': len(errorLogs),
#        'invaild': len(invaildLogs),
#        'open': len(openLogs),
#        'openOnSite': len(openOnSiteLogs),
#        'click': len(clickLogs),
#        'unsubscribe': unsubscriptionNumber,
#        ##
#        'bounce': len(bouncedLogs),
#        'complain': len(complainLogs),
#        'reject': len(rejectedLogs),
#        #'subscribe': len(subscribesList),
#    }

def getAccumulatedValues(listData, dataType):
    """
    'sentTest': sentTestNumber,
    'deliverie': sentNumber,
    'error': len(errorLogs),
    'invaild': len(invaildLogs),
    'open': len(openLogs),
    'openOnSite': len(openOnSiteLogs),
    'click': len(clickLogs),
    'unsubscribe': unsubscriptionNumber,
    ##
    'bounce': len(bouncedLogs),
    'complain': len(complainLogs),
    'reject': len(rejectedLogs),
    #'subscribe': len(subscribesList),
    """
    returnInfo = {}
    _returnInfo = {}

    tmpInfo = dict([(s, 0.0) for s in _STATUS])

    if dataType == 'number':
        for (day, d) in listData:
            for k in tmpInfo.keys():
                if k not in d:
                    continue
                v = tmpInfo[k]
                v += d[k]
                tmpInfo[k] = v

            returnInfo[day] = dict(tmpInfo)

        return returnInfo
    else:
        for (day, d) in listData:
            for k in tmpInfo.keys():
                if k not in d:
                    continue
                v = tmpInfo[k]
                v += d[k]
                tmpInfo[k] = v

            tt = {}
            for k in tmpInfo.keys():
                if k == 'subscribe':
                    continue
                if k == 'unsubscribe':
                    nv = percentage(tmpInfo[k], tmpInfo['deliverie'],
                                     showPercent=False)
                else:
                    nv = percentage(tmpInfo[k], tmpInfo['deliverie'],
                                     showPercent=False)

                tt[k] = nv

            returnInfo[day] = tt

        return returnInfo

def toChartStructure(dictionary):

    info = {}
    for day in dictionary:
        data = dictionary[day]
        for key in data:
            v = data.get(key, 0)
            ll = info.get(key, [])
            ll.append([day, v])
            info.update({key: ll})

    return info

def percentage(value, total, showPercent=True):
    """
    Compute percentage.

    @param value Value
    @param total Total value.
    @param showPercent Show percent?
    """
    if total <= 0:
        v = 0
    else:
        v = ((value * 1.0) / total)
        if v > 1:
            v = 1.0
        v = v * 100 if (value > 0 and total > 0) else 0

    if showPercent:
        v = ('%s' % str(round(v, 2))) + '%'
    else:
        v = round(v, 2)

    return v

def getEmptyDataStructure():
    return {
        'sentTest': 0,
        'deliverie': 0,
        'error': 0,
        'invaild': 0,
        'open': 0,
        'openOnSite': 0,
        'click': 0,
        'unsubscribe': 0,
        'bounce': 0,
        'complain': 0,
        'reject': 0,
    }

def getEmptyDataList():
    return []

def countStatus(queryset, historyMode='general'):
    """
    Calculate the status number of occurrences.
    ex: (c.newsletter.id, c.status, c.creation_date)

    SENT_TEST = -1
    SENT = 0
    ERROR = 1
    INVALID = 2
    OPENED = 4
    OPENED_ON_SITE = 5
    LINK_OPENED = 6
    UNSUBSCRIPTION = 7

    BOUNCED = 8
    COMPLAINT = 9
    REJECTED = 10

    @param queryset The queryset of ContactMailingStatus.
    """
    sentTestNumber = set()
    sentNumber = set()
    errorLogs = set()
    invaildLogs = set()
    openLogs = set()
    openOnSiteLogs = set()
    clickLogs = set()
    unsubscriptionNumber = set()

    openId = []
    openOnSiteId = []
    clickId = []

    bouncedLogs = set()
    complainLogs = set()
    rejectedLogs = set()

    for (id, status) in [(i['contact'], i['status']) for i in queryset]:
        if status == ContactMailingStatus.SENT_TEST:
            sentTestNumber.add(id)
        elif status == ContactMailingStatus.SENT:
            sentNumber.add(id)
        elif status == ContactMailingStatus.ERROR:
            errorLogs.add(id)
        elif status == ContactMailingStatus.INVALID:
            invaildLogs.add(id)
        elif status == ContactMailingStatus.UNSUBSCRIPTION:
            unsubscriptionNumber.add(id)
        elif status == ContactMailingStatus.OPENED:
            openLogs.add(id)
            openId.append(id)
        elif status == ContactMailingStatus.OPENED_ON_SITE:
            openOnSiteLogs.add(id)
            openOnSiteId.append(id)
            if not id in openLogs:
                openLogs.add(id)
                openId.append(id)
        elif status == ContactMailingStatus.LINK_OPENED:
            clickLogs.add(id)
            clickId.append(id)
            if not id in openLogs:
                openLogs.add(id)
                openId.append(id)
        ##
        elif status == ContactMailingStatus.BOUNCED:
            bouncedLogs.add(id)
        elif status == ContactMailingStatus.COMPLAINT:
            complainLogs.add(id)
        elif status == ContactMailingStatus.REJECTED:
            rejectedLogs.add(id)

    # formula reference: emencia.django.utils.statistics.py
    # openings
#     openings = queryset.filter(Q(status=ContactMailingStatus.OPENED) |
#                                Q(status=ContactMailingStatus.OPENED_ON_SITE))
#     openingsByLinks = len(
#         set(queryset.filter(status=ContactMailingStatus.LINK_OPENED).exclude(
#         contact__in=openings.values_list('contact', flat=True)
#     ).values_list('contact', flat=True)))
#     uniqueOpenings = len(set(openings.values_list('contact', flat=True))) + \
#                      openingsByLinks
#
#     # on site openings
#     onSiteOpenings = queryset.filter(
#        status=ContactMailingStatus.OPENED_ON_SITE)
#     uniqueOnSiteOpenings = len(set(onSiteOpenings.values_list(
#                               'contact', flat=True)))
#
#     # clicked
#     clickedLinks = queryset.filter(status=ContactMailingStatus.LINK_OPENED)
#     uniqueClicked = len(set(clickedLinks.values_list(
#                               'contact', flat=True)))
    if historyMode == 'history':
        return [
            {
                'sentTest': len(sentTestNumber),
                'deliverie': len(sentNumber),
                'error': len(errorLogs),
                'invaild': len(invaildLogs),
                'open': len(openId),
                'openOnSite': len(openOnSiteId),
                'click': len(clickId),
                'unsubscribe': len(unsubscriptionNumber),
                'bounce': len(bouncedLogs),
                'complain': len(complainLogs),
                'reject': len(rejectedLogs),
            },
            {
                'openId': openId,
                'openOnSiteId': openOnSiteId,
                'clickId': clickId
            }
        ]
    else:
        return [
            {
                'sentTest': len(sentTestNumber),
                'deliverie': len(sentNumber),
                'error': len(errorLogs),
                'invaild': len(invaildLogs),
                'open': len(openLogs),
                'openOnSite': len(openOnSiteLogs),
                'click': len(clickLogs),
                'unsubscribe': len(unsubscriptionNumber),
                'bounce': len(bouncedLogs),
                'complain': len(complainLogs),
                'reject': len(rejectedLogs),
            },
            {
                'openId': openId,
                'openOnSiteId': openOnSiteId,
                'clickId': clickId
            }
        ]

from django.db.models import Q

from emencia.django.newsletter.models import ContactMailingStatus as Status

def getStartDateEndDate():
    startDate = ContactMailingStatus.objects.order_by('creation_date')[0
                ].creation_date.strftime('%Y-%m-%d')
    endDate = ContactMailingStatus.objects.order_by('-creation_date')[0
               ].creation_date.strftime('%Y-%m-%d')
    return startDate, endDate

def getAllDataDate():
    firstDate = ContactMailingStatus.objects.order_by('creation_date')[0
                ].creation_date
    lastDate = ContactMailingStatus.objects.order_by('-creation_date')[0
               ].creation_date

    deltaDays = lastDate - firstDate + timedelta(1)
    daysAmount = deltaDays.days + 1

    days = [firstDate.strftime('%Y-%m-%d')]
    for i in range(1, daysAmount):
        d = firstDate + timedelta(days=i)
        d = d.strftime('%Y-%m-%d')
        days.append(d)

    class DateRange(object):
        def __init__(self, days):
            self._days = days
            self.startDate = '%s 00:00:00' % days[0]
            self.endDate = '%s 23:59:59' % days[-1]

    return DateRange(days)

def getStatisticsTotalData(reqestGetData, subscriberCount, testSubscriberCount,
                           conditions):
    """
    Get statistics of subscriber's every rates.
    Data for line chart.

    @param slug Newsletter slug.
    @param request Request instance.
    """

    dataType = reqestGetData.get('dataType', 'number')
    mode = reqestGetData.get('mode', 'accumulate')
    historyMode = reqestGetData.get('historyMode', 'general')
    allDataDates = getAllDataDate()

#     period = reqestGetData.get('period', 'week')
#
#     _firstCreationDate = ContactMailingStatus.objects.filter(**conditions)
#     _s = {"d": "strftime('%%Y-%%m-%%d', creation_date)"}
#     _firstCreationDate = _firstCreationDate.extra(_s)
#     _firstCreationDate = _firstCreationDate.values('d').distinct().order_by('d')
#     _t = datetime.strptime(_firstCreationDate[0]['d'], '%Y-%m-%d')
#     dates = getDateRange(period, defaultStart=_t)


    def getContactStatus(conditions, history):
        contactStatus = ContactMailingStatus.objects.filter(**conditions)
        selectData = {"d": "strftime('%%Y-%%m-%%d', "
                            "newsletter_contactmailingstatus.creation_date)"}
        if historyMode == 'history':
            contactStatus = contactStatus.extra(select=selectData).values(
                'contact', 'status', 'd').order_by('contact', 'status', 'd')
        else:
            contactStatus = contactStatus.extra(select=selectData).values(
                'contact', 'status', 'd').distinct(
                ).order_by('contact', 'status', 'd')

        # [{'status': 0, 'contact': 1, 'd': u'2012-12-20'},
        #  {'status': 4, 'contact': 1, 'd': u'2012-12-20'}, ...]
        return contactStatus

    def sortByDatetime(contactStatus):
        completed = OrderedDict()

        dates = []
        for c in contactStatus:
            date = c['d']
            if not date in dates:
                dates.append(date)
        dates = sorted(dates)

        for date in dates:
            year = date.split('-')[0]
            month = date.split('-')[1]
            day = date.split('-')[2]
            queryset = contactStatus.filter(creation_date__day=day,
                                            creation_date__month=month,
                                            creation_date__year=year)
            completed[date] = queryset
        return completed

    # -- get all date content -------------------------------------------------
    conditions.update(
        {'creation_date__range': [allDataDates.startDate,
                                  allDataDates.endDate]})

    if 'newsletter__id__in' in conditions:
        newsletterIds = conditions.pop('newsletter__id__in')


        allDates = []
        allNewslettersData = []
        openIds = []
        openOnSiteIds = []
        linkOpenIds = []
        # data structure of allNewslettersData is something like:
        # [ {u'2017-12-30': {'bounce': 0, 'click': 0, ...}}, ...}
        for _id in newsletterIds:
            conditions.update({'newsletter__id': _id})
            _cs = getContactStatus(conditions, historyMode)
            if _cs:
                # _cs:[ {'status': 0, 'contact': 5, 'd': u'2018-01-04'},
                # {'status': 4, 'contact': 5, 'd': u'2018-01-07'}...]

                for ele in _cs:
                    contactId = ele['contact']
                    _dateKey = ele['d']
                    if _dateKey not in allDates:
                        allDates.append(_dateKey)

                    _statusCount = countStatus([ele, ], historyMode)[0]

                    if _statusCount['open']:
                        if contactId in openIds and historyMode == 'general':
                            _statusCount['open'] = 0
                        else:
                            openIds.append(contactId)
                    if _statusCount['openOnSite']:
                        if contactId in openOnSiteIds and historyMode == 'general':
                            _statusCount['openOnSite'] = 0
                        else:
                            openIds.append(contactId)
                    if _statusCount['click']:
                        if contactId in openOnSiteIds and historyMode == 'general':
                            _statusCount['click'] = 0
                        else:
                            openOnSiteIds.append(contactId)

                    dataTmp = {}
                    dataTmp[_dateKey] = _statusCount
                    allNewslettersData.append(dataTmp)
        # Merge.
        totalData = []
        for _dt in allDates:
            data = {}
            for newsletterData in allNewslettersData:
                if _dt in newsletterData:
                    allDatas = newsletterData[_dt]
                    for _key in allDatas:
                        v = allDatas[_key]
                        if _key not in data:
                            data[_key] = 0
                        data[_key] += v
            totalData.append([_dt, data])
    else:
        contactStatus = getContactStatus(conditions, historyMode)
        completed = sortByDatetime(contactStatus)
        totalData = []

        if historyMode == 'history':
            for c in completed:
                _d = countStatus(completed[c], historyMode)
                data = _d[0]
                totalData.append([c, data])
        else:
            openIds = []
            openOnSiteIds = []
            linkIds = []
            for c in completed:
                _d = countStatus(completed[c])
                open = _d[1]['openId']
                openOnSite = _d[1]['openOnSiteId']
                link = _d[1]['clickId']

                openCount = 0
                openOnSiteCount = 0
                linkCount = 0

                for id in open:
                    if not id in openIds:
                        if not id in openOnSiteIds:
                            if not id in linkIds:
                                openIds.append(id)
                                openCount += 1

                for id in openOnSite:
                    if not id in openOnSiteIds:
                        openOnSiteIds.append(id)
                        openOnSiteCount += 1

                for id in link:
                    if not id in linkIds:
                        linkIds.append(id)
                        linkCount += 1

                data = _d[0]
                data['open'] = openCount
                data['openOnSite'] = openOnSiteCount
                data['click'] = linkCount
                totalData.append([c, data])

    allDates = sorted(dict(totalData).keys())
    for day in allDataDates._days:
        if day not in allDates:
            totalData.append([day, getEmptyDataStructure()])
    totalData = sorted(totalData)

    allContent = {}
    if dataType == 'number':
        #allContent[day] = [(status, value), ...]
        if mode == 'single':
            for (day, data) in totalData:
                allContent[day] = [(i, data[i]) for i in data]
        else:
            accumulativeData = getEmptyDataStructure()
            for (day, data) in totalData:
                result = []
                for status in data:
                    accumulativeData[status] += data[status]
                    result.append((status, accumulativeData[status]))
                allContent[day] = result

    else:
        def _do(info, subscriberCount, testSubscriberCount):
            rates = []
            newsletter = conditions.get('newsletter', None)
            if newsletter:
                totalSentNewsletter = ContactMailingStatus.objects.filter(
                                    newsletter=newsletter,
                                    status=ContactMailingStatus.SENT
                                ).count()
            else:
                totalSentNewsletter = subscriberCount
            for i in info:
                if i in ('unsubscribe', 'deliverie', 'error'):
                    totalValue = subscriberCount
                elif i == 'sentTest':
                    totalValue = testSubscriberCount
                else:
                    totalValue = totalSentNewsletter

                value = percentage(info[i], totalValue, showPercent=False)

                rates.append((i, value))
            return rates

        if mode == 'single':
            for (day, data) in totalData:
                result = _do(data, subscriberCount, testSubscriberCount)
                allContent[day] = result
                # {'day': (status, value)}
                # {'2017-12-25': [('invaild', 0.0), ('sentTest', 0.0), ...]}
        else:
            accumulativeData = getEmptyDataStructure()

            for (day, data) in totalData:
                result = _do(data, subscriberCount, testSubscriberCount)
                accResult = []
                for r in result:
                    status = r[0]
                    value = r[1]
                    accumulativeData[status] += value
                    _result = (status, accumulativeData[status])
                    accResult.append(_result)
                allContent[day] = accResult
    # -- get all date content end ----------------------------------------------

    # -- get range date content start ------------------------------------------
    deltaDays = None

    startDate = reqestGetData.get('startDate', None)
    if isinstance(startDate, str) or isinstance(startDate, type(b'')):
        if startDate != "NaN":
            startDate = datetime.strptime(startDate, '%Y/%m/%d')

    endDate = reqestGetData.get('endDate', None)
    if isinstance(endDate, str) or isinstance(endDate, type(b'')):
        if endDate != "NaN":
            endDate = datetime.strptime(endDate, '%Y/%m/%d')

    period = reqestGetData.get('period', 'week')
    if period == 'week':
        deltaDays = 6
        endDate = datetime.now()
        startDate = endDate - timedelta(days=deltaDays)
    elif period == 'month':
        deltaDays = int(1 * (365 / 12))
        endDate = datetime.now()
        startDate = endDate - timedelta(days=deltaDays)
    elif period == '3month':
        deltaDays = int(3 * (365 / 12))
        endDate = datetime.now()
        startDate = endDate - timedelta(days=deltaDays)
    dateRange = getDateRange(startDate, endDate)

    content = {}
    _days = dateRange._days
    lastDate = allDates[-1]
    lastDateObj = datetime.strptime(allDates[-1], '%Y-%m-%d')
    for _d in _days:
        if _d in allContent.keys():
            content[_d] = allContent[_d]
        else:
            _emptyData = getEmptyDataStructure()
            if mode == 'single':
                if dataType == 'number':
                    content[_d] = [(i, _emptyData[i]) for i in _emptyData]
                else:
                    content[_d] = [(i, 0.0) for i in _emptyData]
            else:
                _dDateObj = datetime.strptime(_d, '%Y-%m-%d')
                if _dDateObj > lastDateObj:
                    content[_d] = allContent[lastDate]
                else:
                    if dataType == 'number':
                        content[_d] = [(i, _emptyData[i]) for i in _emptyData]
                    else:
                        content[_d] = [(i, 0.0) for i in _emptyData]
   # -- get range date content end --------------------------------------------


    context = {}
    for day in content:
        for (statusName, value) in content[day]:
            dData = context.get(statusName, [])
            dData.append((day, value))
            context[statusName] = dData

    # Find the max value of Y-axis.
    tmp = []
    for k in context:
        for i in context[k]:
            tmp.append(i[-1])

    yaxisMaxValue = max(tmp)
    del tmp


    _STATUS_NAMES = []
    for i in STATUS_NAMES:
        _STATUS_NAMES.append(str(i))

    response = {
        'context': {
            'email': context,
        },
        'rateType': dataType,
        'dateRange': dateRange._days,
        'xaxis': {'minDate': dateRange._days[0], 'maxDate': dateRange._days[-1]},
        'yaxis': {'max': yaxisMaxValue},
        'status': _STATUS,
        'settings': {
            'labels': _STATUS_NAMES,
        },
    }
    return response

def getStatisticsTotalRateData(reqestGetData, subscriberCount,
                               testSubscriberCount, conditions):
    """
    Get statistics of subscriber's every rates.
    Data for pie chart and statistics table.

    @param slug Newsletter slug.
    @param request Request instance.
    """

    # select distinct contact_id, status from newsletter_contactmailingstatus
    # where newsletter_id=3
    def getContactStatus(conditions):
        contactStatus = ContactMailingStatus.objects.filter(**conditions)
        # FIXME: Distinct multiple fields?
        #contactStatus = contactStatus.order_by().values(
        #    'contact', 'status').distinct(['contact', 'status'])
        contactStatus = contactStatus.order_by().values(
            'contact', 'status').distinct()
        # [{'status': 0, 'contact': 1}, {'status': 4, 'contact': 1}, ...]
        return contactStatus

    if 'newsletter__id__in' in conditions:
        newsletterIds = conditions.pop('newsletter__id__in')
        data = {}
        for _id in newsletterIds:
            conditions.update({'newsletter__id': _id})
            _cs = getContactStatus(conditions)

            _data = countStatus(_cs)[0]
            for _d in _data:
                v = _data[_d]
                if _d not in data:
                    data[_d] = 0
                data[_d] += v
    else:
        contactStatus = getContactStatus(conditions)
        data = countStatus(contactStatus)[0]

    rates = []

    totalValue = 0

    for d in data:
        totalValue += data[d]


    for d in data:
        if d in ('unsubscribe', 'deliverie', 'error'):
            totalValue = subscriberCount
        elif d == 'sentTest':
            totalValue = testSubscriberCount
        else:
            totalValue = data['deliverie']

        value = percentage(data[d], totalValue)

        rates.append((d, value))

    _STATUS_NAMES = []
    for i in STATUS_NAMES:
        _STATUS_NAMES.append(str(i))

    response = {
        'context': dict(rates),
        'contextCount': data,
        'status': _STATUS,
        'statusNames': _STATUS_NAMES,
    }
    return response

def getSubscriberHistory(requestGetData, conditions):
    """
    Get history data of subscribers.

    @param slug Slug.
    """

    statusCode = conditions.pop('status', None)
    QList = conditions.pop('QList', [])

    contactStatus = ContactMailingStatus.objects.filter(*QList, **conditions)
    contactStatus = contactStatus.annotate(d=modelsMax("creation_date"),
                                           qid=modelsMax("id"))
    contactStatus = contactStatus.values('contact', 'qid')
    contactStatus.query.group_by = ['contact_id']
    qs = contactStatus.order_by('id')

    qids = [_i['qid'] for _i in qs]

    # Sort data.
    #sortedFieldName = requestGetData.get('sort', 'lastActivity')
    sortType = requestGetData.get('sortType', 'desc')
    if sortType == 'desc':
        sortType = True
    else:
        sortType = False

    #emailData = sortList(emailData, sortedFieldName, desc=sortType)
    info = []
    if statusCode is None:
        _query = ContactMailingStatus.objects.filter(
            id__in=qids).order_by('-creation_date')
    else:
        _query = ContactMailingStatus.objects.filter(
            id__in=qids, status=statusCode).order_by('-creation_date')

    total = _query.count()

    # Get query range.
    more = False
    start = int(requestGetData.get('start', 0))
    if (len(_query) - start) > 20:
        more = True
    _query = _query[start:start + 20]

    for q in _query:
        info.append({
            'contact': {
                'id': q.contact.id,
                'name': _unicode(q.contact),
                'email': q.contact.email,
            },
            'status': q.status,
            'creationDate': q.creation_date.strftime('%Y-%m-%d %I:%M%p'),
        })

    return {
        'content': info,
        'sortType': sortType,
        'more': more,
        'startNum': start,
        'totalNum': total,
        #'sortedFieldName': sortedFieldName,
    }
