#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: AdvancedMailServer.py 9589 2016-03-14 07:45:28Z Judy $
#
# Copyright (c) 2014 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $ (Last)
# $Date: 2013-01-30 12:13:45 +0800$
# $Revision: 9589 $

from datetime import datetime, timedelta

from django.conf import settings

from Iuppiter.schedule import Scheduler
from Iuppiter.dispatch import WorkRequest, ThreadPool
from Iuppiter.Logging import createLogger

from Theophrastus.models import Newsletter, ContactMailingStatus
from Theophrastus.cloud.Mailer import CloudMailer
from Theophrastus.cloud.service import CloudService

LOG_LEVEL = settings.LOG_LEVEL
SEND_FAILED_COUNT = settings.SEND_FAILED_COUNT
WAITTING_TIMEOUT = settings.WAITTING_TIMEOUT
ENABLE_FAKE_MAIL = settings.ENABLE_FAKE_MAIL
ENABLE_FAKE_SMTP = settings.ENABLE_FAKE_SMTP
APSCHEDULER_LOGFILE = settings.APSCHEDULER_LOGFILE
ENABLE_APSCHEDULER_LOGFILE = settings.ENABLE_APSCHEDULER_LOGFILE
MAIL_SERVER_LOGFILE = settings.MAIL_SERVER_LOGFILE
MAILER_LOGFILE = settings.MAILER_LOGFILE

MAILER_THREAD_POOL_COUNT = settings.MAILER_THREAD_POOL_COUNT
SENDBOX_THREAD_POOL_COUNT = settings.SENDBOX_THREAD_POOL_COUNT

cloudSrv = CloudService.getInstance()

class AdvancedMailServer(object):
    def __init__(self):
        """
        Constructor.
        """
        logLevel = LOG_LEVEL

        # Create loggers.
        if ENABLE_APSCHEDULER_LOGFILE:
            createLogger(
                'apscheduler.scheduler', APSCHEDULER_LOGFILE,
                loggingLevel=logLevel)

        if MAIL_SERVER_LOGFILE:
            self.logger = createLogger(
                'theophrastusMailServer', MAIL_SERVER_LOGFILE,
                loggingLevel=logLevel)
        else:
            raise RuntimeError(
                'Need to set "THEOPHRASTUS_MAIL_SERVER_LOGFILE" in '
                'settings.py.')

        if MAILER_LOGFILE:
            self.mailerLogger = createLogger(
                'theophrastusMailer', MAILER_LOGFILE, loggingLevel=logLevel)
        else:
            raise RuntimeError(
                'Need to set "THEOPHRASTUS_MAILER_LOGFILE" in settings.py.')

        # Start the scheduler
        self.sched = Scheduler(daemonic=True)
        # Create mailer pool.
        self.mailerPool = ThreadPool(MAILER_THREAD_POOL_COUNT)
        # Create sendbox pool.
        self.sendboxPool = ThreadPool(SENDBOX_THREAD_POOL_COUNT)

        # Schedule job_function to be called every two hours
        self.sched.addIntervalJob(self.seeker, seconds=30)
        self.sched.addIntervalJob(self.checkStatus, seconds=60)

    def seeker(self):
        """
        Find out newsletters that status wasn't "DRAFT", "SENDING" or "SENT".
        """
        syncs = []
        if settings.ENABLE_INTEGRATION:
            syncs = cloudSrv.findNewsletterInformations(sync=True)
            syncs = syncs.values('newsletter')
            syncs = [i['newsletter'] for i in syncs]

        for newsletter in Newsletter.objects.filter(
            status=Newsletter.WAITING).exclude(id__in=syncs):
            try:
                self.logger.info(
                    "[%d]_START: Preparing to put a newsletter to pool ." %
                    newsletter.id)

                mailer = CloudMailer(newsletter, pool=self.sendboxPool,
                                     fakeMail=ENABLE_FAKE_MAIL,
                                     fakeSMTP=ENABLE_FAKE_SMTP,
                                     logger=self.mailerLogger)

                self.logger.info(
                    "[%d]_READY: Newsletter was ready to process." %
                    newsletter.id)

                self.mailerPool.putRequest(WorkRequest(mailer.run))
                self.mailerPool.poll()

                self.logger.info(
                    "[%d]_SUCCESS: Putting a newsletter pool success." %
                    newsletter.id)
            except Exception as e:
                self.logger.info(
                    "[%d]_FAILED: Putting a newsletter pool failed. "
                    "Reason: %s" % (newsletter.id, str(e)))
            finally:
                self.logger.info(
                    "[%d]_END: Put a newsletter end." % newsletter.id)

    def checkStatus(self):
        """
        Check newsletter's status, if it's status was "watting" and modification
        time small than now time for "WAITTING_TIMEOUT" minutes, we should to
        re-send it.
        If it's failed count was big than "SEND_FAILED_COUNT", we should to
        cancel it.

        SQL:
        SELECT COUNT(distinct `newsletter_contactmailingstatus`.`contact_id`),
               COUNT(`newsletter_contactmailingstatus`.`status`)
        FROM `newsletter_contactmailingstatus`
        WHERE (`newsletter_contactmailingstatus`.`newsletter_id` =  3677 AND
               `newsletter_contactmailingstatus`.`status` IN (1, 2, 10) AND
               NOT (`newsletter_contactmailingstatus`.`contact_id` IN (13, 14)))
        GROUP BY `newsletter_contactmailingstatus`.`contact_id`
        ORDER BY NULL
        become to django query:
        SELECT DISTINCT
               COUNT(`newsletter_contactmailingstatus`.`contact_id`) AS `d1`
        FROM `newsletter_contactmailingstatus`
        WHERE (`newsletter_contactmailingstatus`.`newsletter_id` =  3677 AND
               `newsletter_contactmailingstatus`.`status` IN (1, 2, 10) AND
               NOT (`newsletter_contactmailingstatus`.`contact_id` IN (13, 14)))
        GROUP BY (contact_id)
        ORDER BY NULL
        """
        from django.db.models import Count
        
        _status = [ContactMailingStatus.ERROR, ContactMailingStatus.INVALID,
                  ContactMailingStatus.REJECTED]
        for newsletter in Newsletter.objects.filter(status=Newsletter.SENDING):
            try:
                self.logger.info(
                    '[%d]_CHECK_START: Start check newsletter.' % newsletter.id)

                hasSent = ContactMailingStatus.objects.filter(
                    newsletter=newsletter).exclude(status__in=_status)
                hasSent.query.group_by = ['contact_id']
                hasSent = hasSent.order_by().values('contact_id')
                hasSentContacts = [i['contact_id'] for i in hasSent]

                hasFailed = ContactMailingStatus.objects.filter(
                    newsletter=newsletter, status__in=_status)
                hasFailed = hasFailed.exclude(contact__id__in=hasSentContacts)
                hasFailed = hasFailed.annotate(
                    d=Count("status"), d1=Count("contact"))
                hasFailed = hasFailed.values('d1').distinct()
                hasFailed.query.group_by = ['contact_id']
                hasFailed = hasFailed.order_by()

                failContactCount = len(hasFailed)
                noPass = 0
                if failContactCount > 0:
                    for r in hasFailed:
                        num = r['d1']
                        if num >= SEND_FAILED_COUNT:
                            noPass += 1

                    if noPass >= failContactCount:
                        newsletter.status = Newsletter.CANCELED
                        newsletter.save()
                        self.logger.info(
                            '[%d]_SET_CANCELED: Cancel this newsletter, because'
                            ' newsletter has too many fail.' % newsletter.id)
                        continue

                _t = datetime.now() - timedelta(minutes=WAITTING_TIMEOUT)
                if _t > newsletter.modification_date:
                    newsletter.status = Newsletter.WAITING
                    newsletter.save()
                    self.logger.info(
                    '[%d]_SET_RESEND: Resend this newsletter, because this'
                    ' newsletter waited for too long.' % newsletter.id)
            except Exception as e:
                self.logger.info(
                    '[%d]_CHECK_FAILED: Check newsletter fail. Reason: %s' %
                    (newsletter.id, str(e)))
            finally:
                self.logger.info(
                    '[%d]_CHECK_END: Check newsletter end.' % newsletter.id)

    def start(self):
        """
        Start scheduler.
        """
        self.sched.start()

    def stop(self, wait=False):
        """
        Stop scheduler.
        """
        self.sched.stop(wait)
