#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 9579 2016-03-09 01:43:46Z Judy $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $
# $Date: 2016-03-09 09:43:46 +0800 (三, 09  3 2016) $
# $Revision: 9579 $

import re
from urlparse import urljoin

from django.contrib.sites.models import Site
from django.core.urlresolvers import reverse
from django.shortcuts import HttpResponse
from django.utils.translation import ugettext as _
from django.core.cache import cache

from Iuppiter.extension.views import jsonpCallback
from Iuppiter.Encoding import _unicode
from Iuppiter.Logging import createDebugLogger

from django.conf import settings

from Theophrastus.models import ContactMailingStatus, Contact
from Theophrastus.cloud.service import CloudService
from Theophrastus.cloud.Amazon import SES, SNS

cloudSrv = CloudService.getInstance()

def _getDomain():
    site = Site.objects.get_current()
    domain = site.domain.replace('.', '-')
    return domain

_domain = _getDomain()

AWS_ACCESS_KEY_ID = settings.AWS_ACCESS_KEY_ID
AWS_SECRET_ACCESS_KEY = settings.AWS_SECRET_ACCESS_KEY

BOUNCES_TOPIC_NAME = getattr(settings, 'BOUNCES_TOPIC_NAME', 'bounces')
BOUNCES_TOPIC_NAME = '%s_%s' % (_domain, BOUNCES_TOPIC_NAME)
COMPLAINTS_TOPIC_NAME = getattr(settings, 'COMPLAINTS_TOPIC_NAME', 'complaints')
COMPLAINTS_TOPIC_NAME = '%s_%s' % (_domain, COMPLAINTS_TOPIC_NAME)

SUBSCRIBE_RETRY_COUNT = getattr(settings, 'SUBSCRIBE_RETRY_COUNT', 3)

amazonLogger = createDebugLogger('amazon', settings.AMAZON_LOGFILE)

def getNameFromArn(arn):
    s = arn.split(':')
    return s[-1]

class EmailAddressException(Exception):
    def __init__(self, email):
        self.email = email

    def __str__(self):
        if self.email:
            return _("Incorrect e-mail address: %s.") % self.email
        else:
            return _("Please enter e-mail address.")

def filterEmail(email):
    """
    Filter email address.

    @param email Email address.
    """
    email = email.strip().lower()

    if '<' and '>' in email:
        EMAIL_FILTER = re.compile(r'(.*?)\<(?P<email>.*?)\>')
        filterResult = EMAIL_FILTER.search(email)
        if filterResult:
            filterContent = filterResult.groupdict()
            email = filterContent.get('email', '')
        else:
            raise EmailAddressException(email)

    if not email:
        raise EmailAddressException(email)
    return email

def setSenderByCache(sender, timeout=216000):
    """
    Set verification sender.

    @param sender Sender's email address.
    @param timeout The number of seconds the value should be stored in the
           cache.
    """
    cache.set('amazonSender_%s' % sender, True, timeout)

def checkSenderByCache(sender):
    """
    Get verification sender.

    @param sender Sender's email address.
    """
    return cache.get('amazonSender_%s' % sender, False)

def doVerifyEmailAddress(email):
    """
    Verify email address.

    Reference:
        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/index.html#
        m=AmazonSES/verify_email_address

        and

        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
        index.html#m=AmazonSES/list_verified_email_addresses

    @param email Email address.
    """
    try:
        response = {
            'verification': True,
            'continue': False,
            'message': _("This email address already verified.\n")
        }

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        if email in ses.getAllVerifiedSenders():
            checkBounce = None
            checkComplaint = None

            notifyAttrValues = ses.getIdentityNotificationAttributes(email)
            bounceTopic = notifyAttrValues.get('BounceTopic', '')
            if bounceTopic:
                checkBounce = False
                if getNameFromArn(bounceTopic) == BOUNCES_TOPIC_NAME:
                    checkBounce = True

            complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
            if complaintTopic:
                checkComplaint = False
                if getNameFromArn(complaintTopic) == COMPLAINTS_TOPIC_NAME:
                    checkComplaint = True

            if (checkBounce is False) or (checkComplaint is False):
                response['continue'] = True
                response['message'] += _(
                    "Your email has been set for other system.\n"
                    "Are you sure you want to continue?")
            elif (checkBounce is None) or (checkComplaint is None):
                response['message'] += _("You not yet set mail notifications.")

            return response
        else:
            try:
                ses.verifyEmailAddress(email)
                return {
                    'verification': False,
                    'message': _(
                        "You not yet verify your email address.\n"
                        "Your verify request was sent, please to receive "
                        "verify email on your sender email box.\n")
                }

            except Exception as e:
                return {
                    'verification': False,
                    'message': _("Send verify request failed, please check "
                                 "your e-mail address.\nreason: %s" % str(e))
                }
    except EmailAddressException as e:
        return {
            'verification': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'verification': False,
            'message': _("Unknown error: %s") % str(e)
        }

@jsonpCallback()
def verifyEmailAddress(request):
    """
    Verify email address.
    """
    email = filterEmail(request.GET.get('email', ''))
    return doVerifyEmailAddress(email)

@jsonpCallback()
def deleteVerifiedEmailAddress(request):
    """
    Verify email address.

    Reference: http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
               index.html#m=AmazonSES/delete_verified_email_address
    @param email Email address.
    """
    try:
        email = filterEmail(request.GET.get('email', ''))

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        response = ses.deleteVerifiedEmailAddress(email)
        return {
            'result': True,
            'message': _("This email address already delete from "
                         "verified list.")
        }
    except EmailAddressException as e:
        return {
            'result': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'result': False,
            'message': _("Unknown error: %s") % str(e)
        }

@jsonpCallback()
def checkVerifiedEmailAddress(request):
    """
    Check email address.

    Reference:
        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
        index.html#m=AmazonSES/list_verified_email_addresses
    @param email Email address.
    """
    try:
        email = filterEmail(request.GET.get('email', ''))

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        verifiedList = ses.getAllVerifiedSenders()

        if email in verifiedList:
            response = {
                'verification': True,
                'message': _("This email address already verified.\n")
            }

            checkBounce = None
            checkComplaint = None
            notifyAttrValues = ses.getIdentityNotificationAttributes(email)

            bounceTopic = notifyAttrValues.get('BounceTopic', '')
            if bounceTopic:
                if getNameFromArn(bounceTopic) == BOUNCES_TOPIC_NAME:
                    checkBounce = True

            complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
            if complaintTopic:
                if getNameFromArn(complaintTopic) == COMPLAINTS_TOPIC_NAME:
                    checkComplaint = True

            if checkBounce and checkComplaint:
                return response
            else:
                response['verification'] = False
                response['message'] += _(
                    "But you not yet set mail notifications."
                    "Please verify again before you save or add.")

            return response
        else:
            return {
                'verification': False,
                'message': _("You not yet verify your email address.\n"
                             "Please verify email before you save or add.")
            }
    except EmailAddressException as e:
        return {
            'verification': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'verification': False,
            'message': _("Unknown error: %s") % str(e)
        }

from django.views.decorators.csrf import csrf_exempt

from Iuppiter.Json import serializer

def recordStatus(messageId, destination, status):
    amazonLogger.info("START: Start record.")
    try:
        amazonLogger.info(
            "messageId=%s, destination=%s, status=%s" %
            (messageId, destination, status))

        obj = cloudSrv.getCloudMailInformation(
            messageId=messageId, errorOnNotFound=False)
        if obj:
            newsletter = obj.newsletter
            contact = obj.contact
            for contact in Contact.objects.filter(email__in=destination):
                ContactMailingStatus.objects.create(
                    newsletter=newsletter,
                    contact=contact,
                    status=status)
            amazonLogger.info("SUCCESS: Record success.")
        else:
            amazonLogger.info("PASS: Not matched, record pass.")

    except Exception as e:
        amazonLogger.info("FAILED: Record failed, reason: %s" % str(e))
    finally:
        amazonLogger.info("END: Record end.")

def recordBouncedStatus(messageId, destination):
    recordStatus(messageId, destination, ContactMailingStatus.BOUNCED)

def recordRejectedStatus(messageId, destination):
    recordStatus(messageId, destination, ContactMailingStatus.REJECTED)

def recordComplaintStatus(messageId, destination):
    recordStatus(messageId, destination, ContactMailingStatus.COMPLAINT)

def isBounceType(bounceType, bounceSubType):

    if bounceType == 'Undetermined' and bounceSubType == 'Undetermined':
        # Bounce.
        return True
    elif bounceType == 'Permanent' and bounceSubType == 'General':
        # Bounce.
        return True
    elif bounceType == 'Permanent' and bounceSubType == 'NoEmail':
        # Bounce.
        return True
    elif bounceType == 'Transient' and bounceSubType == 'General':
        # Bounce.
        return True
    elif bounceType == 'Transient' and bounceSubType == 'MailboxFull':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'MessageToolarge':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'ContentRejected':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'AttachmentRejected':
        # Rejected.
        return False

import sys
import traceback
from urllib import urlopen

def handleSNS(request):
    """
    http://docs.amazonwebservices.com/ses/latest/DeveloperGuide/
        NotificationContents.html#BounceObject
    http://docs.amazonwebservices.com/ses/latest/DeveloperGuide/
        ExampleNotifications.html
    Format:
        http://docs.amazonwebservices.com/sns/latest/gsg/json-formats.html
    """
    # Get the message type header.
    # String messagetype = request.getHeader("x-amz-sns-message-type");
    # If message doesn't have the message type header, don't process it.
    postData = request._get_raw_post_data()

    try:
        amazonLogger.info("RECEIVED: %s" % str(postData))

        #Parse the JSON message in the message body
        #and hydrate a Message object with its contents
        #so that we have easy access to the name/value pairs
        #from the JSON message.
        postData = _unicode(postData)
        postData = serializer.deserialize(postData)

        # Process the message based on type.
        topicType = postData.get('Type', '')
        if topicType == 'SubscriptionConfirmation':
            amazonLogger.info("Topic type: %s" % topicType)
            # You should make sure that this subscription is from the topic you
            # expect. Compare topicARN to your list of topics
            # that you want to enable to add this endpoint as a subscription.
            # Confirm the subscription by going to the subscribeURL location
            # and capture the return value (XML message body as a string)
            subscribeURL = postData.get('SubscribeURL', '')
            amazonLogger.info("Subscribe URL: %s" % subscribeURL)

            conn = None
            try:
                conn = urlopen(subscribeURL)
                amazonLogger.info("URL response: %s" % conn.read())
            except Exception as e:
                amazonLogger.info("Get URL response failed: %s" % str(e))
            finally:
                conn.close()

        elif topicType == 'Notification':
            amazonLogger.info("Topic type: %s" % topicType)

            message = postData.get('Message', '')

            amazonLogger.info("Message: %s" % message)
            if not isinstance(message, dict):
                message = serializer.deserialize(message)

            notificationType = message.get('notificationType', '')
            if notificationType == 'Bounce': # Bounced.
                amazonLogger.info("Message type: %s" % notificationType)

                mail = message.get('mail', {})
                destination = mail.get('destination', [])
                messageId = mail.get('messageId', '')

                # Get bounce's type.
                bounceType = message.get('bounceType', '')
                bounceSubType = message.get('bounceSubType', '')
                isBounce = isBounceType(bounceType, bounceSubType)
                if isBounce:
                    # record!
                    amazonLogger.info("Got bounce, need record...")
                    recordBouncedStatus(messageId, destination)
                else:
                    # record!
                    amazonLogger.info("Got rejected, need record...")
                    recordRejectedStatus(messageId, destination)

            elif notificationType == 'Complaint': # Complaints.
                amazonLogger.info("Message type: %s" % notificationType)

                mail = message.get('mail', {})
                destination = mail.get('destination', [])
                messageId = mail.get('messageId', '')
                # Record.
                amazonLogger.info("Got complaint, need record...")
                recordComplaintStatus(messageId, destination)

        elif topicType == 'UnsubscribeConfirmation':
            amazonLogger.info("Topic type: %s" % topicType)
            # Handle UnsubscribeConfirmation message.
            # For example, take action if unsubscribing should not have occurred.
            # You can read the SubscribeURL from this message and
            # re-subscribe the endpoint.
            pass
        else:
            # Handle unknown message type.
            amazonLogger.info("Topic type: unknown.")
            pass
    except Exception as e:
        c = '\n'.join(traceback.format_tb(sys.exc_info()[-1]))
        amazonLogger.info("RECEIVED_FAILED: %s\n%s\n" % (c, e))

    return HttpResponse('ok')

@csrf_exempt
def getBounces(request):
    return handleSNS(request)

@csrf_exempt
def getComplains(request):
    return handleSNS(request)

def checkNotificationTopic(email, useCache=False):
    """
    Check notification topic.

    @param email Email address.
    @param useCache Save verification result to cache?
    """
    try:
        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

        notifyAttrValues = ses.getIdentityNotificationAttributes(email)

        bounceTopic = notifyAttrValues.get('BounceTopic', '')
        if getNameFromArn(bounceTopic) != BOUNCES_TOPIC_NAME:
            bounceTopic = ''

        complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
        if getNameFromArn(complaintTopic) != COMPLAINTS_TOPIC_NAME:
            complaintTopic = ''

        if bounceTopic and complaintTopic:
            if useCache:
                # Set cache.
                setSenderByCache(email)
            return {'result': True}
        else:
            return {'result': False}
    except Exception as e:
        return {
            'result': False,
            'message': _('Check notification topic failed, reason: %s. '
                         'If this error shown again, please contact us.') %
                         _unicode(e)
        }

def doSetNotificationTopic(email, useCache=False):
    """
    Sets the Amazon SNS topic to which Amazon SES will publish bounce and
    complaint notifications for emails sent with that identity as the Source.
    Publishing to topics may only be disabled when feedback forwarding is
    enabled.
    (Given an identity (email address or domain).)

    Reference: http://docs.amazonwebservices.com/ses/latest/APIReference/
               http://docs.aws.amazon.com/sns/latest/api/
               http://sesblog.amazon.com/post/TxJE1JNZ6T9JXK/Handling-Bounces-
                      and-Complaints

    @param email Email address.
    @param useCache Save verification result to cache?
    """
    try:
        bounceTopic = ''
        complaintTopic = ''

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        # Check topics.
        sns = SNS(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

        for topic in sns.getAllTopicArns():
            if bounceTopic and complaintTopic:
                break
            topicName = getNameFromArn(topic)
            if topicName == BOUNCES_TOPIC_NAME:
                bounceTopic = topic
            if topicName == COMPLAINTS_TOPIC_NAME:
                complaintTopic = topic

        if not bounceTopic: # Create a topic of bounces.
            bounceTopic = sns.createTopic(BOUNCES_TOPIC_NAME)

        if not complaintTopic: # Create a topic of complaints.
            complaintTopic = sns.createTopic(COMPLAINTS_TOPIC_NAME)

        # Check subscription.
        hasBounceTopicExist = False
        hasComplaintTopicExist = False
        for n in range(SUBSCRIBE_RETRY_COUNT):
            if hasBounceTopicExist and hasComplaintTopicExist:
                break
            if not hasBounceTopicExist:
                hasBounceTopicExist = sns.checkSubscriptionExist(
                    bounceTopic, BOUNCES_TOPIC_NAME)
            if not hasComplaintTopicExist:
                hasComplaintTopicExist = sns.checkSubscriptionExist(
                    complaintTopic, COMPLAINTS_TOPIC_NAME)

        # Subscribe to a Topic.
        bounceEndpoint = reverse(getBounces)
        if not hasBounceTopicExist:
            sns.subscribe(
                bounceTopic, 'http',
                urljoin(settings.LOCAL_URL, bounceEndpoint))

        complaintEndpoint = reverse(getComplains)
        if not hasComplaintTopicExist:
            sns.subscribe(
                complaintTopic, 'http',
                urljoin(settings.LOCAL_URL, complaintEndpoint))

        # Set notification topic.
        ses.setIdentityNotificationTopic('Bounce', email, bounceTopic)
        ses.setIdentityNotificationTopic('Complaint', email, complaintTopic)
        if useCache:
            # Set cache.
            setSenderByCache(email)
        return {
            'result': True,
            'bounceTopic': bounceTopic,
            'complaintTopic': complaintTopic
        }
    except Exception as e:
        return {
            'result': False,
            'message': _('Set notification topic failed, reason: %s. '
                         'If this error shown again, please contact us.') %
                         _unicode(e)
        }

@jsonpCallback()
def setNotificationTopic(request):
    """
    Sets the Amazon SNS topic to which Amazon SES will publish bounce and
    complaint notifications for emails sent with that identity as the Source.
    Publishing to topics may only be disabled when feedback forwarding is
    enabled.
    """
    email = filterEmail(request.GET.get('email', ''))
    _checkResult = checkNotificationTopic(email)
    if _checkResult['result']:
        return _checkResult
    else:
        return doSetNotificationTopic(email)
