#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Handlers.py 16181 2025-05-22 12:40:09Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2025-05-22 20:40:09 +0800 (週四, 22 五月 2025) $
# $Revision: 16181 $

import os

from typing import Tuple

from Aristotle.main.DataTypes import MailData, Target
from Pixiu.llm.Prompt import PromptHandler

from .DataTypes import LLMActionResult, LLMActionResultType, LLMProvider, RuleMatch
from .Interfaces import BaseLLMHandler

currDir = os.path.dirname(os.path.abspath(__file__))
RULE_MATCH_PROMPT_PATH = os.path.join(currDir, 'prompts', 'RuleMatch.yml.j2')
LLM_ACTION_PROMPT_PATH = os.path.join(currDir, 'prompts', 'v2', 'LLMAction.yml.j2')
SYSTEM_PROMPT = PromptHandler.getPrompt(
    templatePath=os.path.join(currDir, 'prompts', 'System.yml.j2'),
    templateData={},
)


class LLMRuleHandler(BaseLLMHandler):

    @classmethod
    def isMailMatched(
        cls,
        provider: LLMProvider,
        mailData: MailData,
        rulePrompt: str,
        apiKey: str,
        model: str,
        target: Target,
        maxRetries: int=3,
        temperature: float=1,
        topP: float=1,
        instructorMaxRetries: int=3,
    ) -> bool:
        """
        Determines if an email matches a specified rule prompt using an AI model.

        This method initializes an AI client with a given API key and model ID,
        constructs a prompt using the provided mail data and rule prompt, and
        retrieves the response from the AI client. It returns whether the rule
        matches based on the AI's response.

        Args:
            mailData (MailData): The mail data object containing details about the email.
            rulePrompt (str): The rule prompt string to be matched against the mail data.

        Returns:
            bool: True if the rule is matched, False otherwise.
        """
        kwargs = {
            "apiKey": apiKey,
            "model": model,
            "respModel": RuleMatch,
            "multiple": False,
            "metadata": {
                "session_id": "aristotle_llm_rule"
            },
            "maxRetries": maxRetries,
            "temperature": temperature,
            "topP": topP,
            "instructorMaxRetries": instructorMaxRetries,
        }
        client = cls.getClient(provider=provider, **kwargs)

        if isinstance(target, int):
            target = Target(target)

        templateData = {
            "mailData": mailData if target == Target.ALL.value else {
                target.name: mailData.targetValue(target)
            },
            "rulePrompt": rulePrompt,
        }
        prompt = PromptHandler.getPrompt(templatePath=RULE_MATCH_PROMPT_PATH, templateData=templateData)
        ruleMatch: RuleMatch = client.getResponse(prompt, SYSTEM_PROMPT)
        return ruleMatch.matached


class LLMActionHandler(BaseLLMHandler):

    @classmethod
    def execute(
        cls,
        provider: LLMProvider,
        mailData: MailData,
        prompt: str,
        apiKey: str,
        model: str,
        maxRetries: int=3,
        temperature: float=1,
        topP: float=1,
        instructorMaxRetries: int=3,
    ) -> Tuple[LLMActionResultType, str | MailData, bool]:
        # NOTE: Does not support update on attachments
        client = cls.getClient(
            provider=provider,
            apiKey=apiKey,
            model=model,
            respModel=LLMActionResult,
            multiple=False,
            metadata={"session_id": "aristotle_llm_action"},
            maxRetries=maxRetries,
            temperature=temperature,
            topP=topP,
            instructorMaxRetries=instructorMaxRetries,
        )
        templateData = {
            "mailData": mailData,
            "prompt": prompt,
        }
        prompt = PromptHandler.getPrompt(templatePath=LLM_ACTION_PROMPT_PATH, templateData=templateData)
        result: LLMActionResult = client.getResponse(prompt, SYSTEM_PROMPT)
        output = result.response

        if result.resultType == LLMActionResultType.MAIL:
            resultMail = result.resultMail
            output = MailData(
                subject=resultMail.subject,
                sender=resultMail.sender,
                receivers=resultMail.receivers,
                body=resultMail.body,
                msg=resultMail.msg,
                date=resultMail.date,
                attachments=mailData.attachments, # NOTE: keep original attachments
                textPlain=resultMail.textPlain,
                textHTML=resultMail.textHTML,
                timezone=resultMail.timezone
            )

        return result.resultType, output, result.passToNext
