#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: DataTypes.py 15557 2025-02-20 04:30:31Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2025-02-20 12:30:31 +0800 (週四, 20 二月 2025) $
# $Revision: 15557 $

from dataclasses import asdict, dataclass
from enum import Enum, IntEnum
from typing import List, Optional


class RuleType(IntEnum): # sync with aristotle api RuleType class
    IS_REGEX = 1
    IS_NOT_REGEX = 2
    INCLUDE = 3
    EXCLUDE = 4
    PROMPT = 5 # LLM


class Target(IntEnum):
    SENDER = 1 # 寄件人
    RECEIVER = 2 # , _("收件人")
    SUBJECT = 3 # , _("主旨")
    BODY = 4 # , _("內容")
    X_SPAM_FLAG = 5 # , _("X_SPAM_FLAG")
    DATE = 6 # , _("日期")
    ALL = 7 # , _("全部")


class FilterOption(IntEnum):
    ALL_RULES = 1 # , _("符合所有規則")
    ANY_RULE = 2 # , _("符合任一規則")
    ALL_EMAILS = 3 # , _("所有信件")

class ExecutionOption(IntEnum):
    BY_ORDER = 1 # , _("依序執行") 失敗了不執行下個動作
    EXECUTE_ALL = 2 # , _("全部執行") 失敗不影響其他動作執行

@dataclass(frozen=True) # make parameter immutable
class Attachment:
    """raw attachment"""
    name: str
    content: str
    type: str


@dataclass(frozen=True)
class MailData:
    sender: str
    receivers: List[str]
    subject: str
    body: str
    attachments: Optional[List[Attachment]] = None
    msg: Optional[str] = None # the complete mail message from Message.as_string()
    date: Optional[str] = None
    textPlain: List[str] = None
    textHTML: List[str] = None
    timezone: str = None

    def _dict(self, complete=True):
        if complete:
            return asdict(self)
        else:
            mail = asdict(self)
            mail.pop("msg")
            mail.pop("textPlain")
            mail.pop("textHTML")
            return mail

    def targetValue(self, target: Target):
        if isinstance(target, int):
            target = Target(target)
        return self.targetMapper.get(target)

    @property
    def targetMapper(self) -> dict[Target, List[str] | str]:
        return {
            Target.SENDER: self.sender,
            Target.RECEIVER: self.receivers,
            Target.SUBJECT: self.subject,
            Target.BODY: self.body,
            Target.DATE: self.date,
        }

class TaskScheduleType(IntEnum):
    ADD = 1
    REMOVE = 2
    REMOVE_ALL_JOBS = 3

class APITokenType(IntEnum):
    API = 1
    TRIGGER_TASK_ACTION = 2
    TASK_SCHEDULE = 3
    WASM_UPLOAD = 4
    EXECUTE_ACTION = 5
    EXECUTE_ROUTE = 6

class ActionStatus(IntEnum):
    PENDING = 0
    FAILED = 1
    SUCCESS = 2
    CANCELLED = 3