#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Handlers.py 15430 2025-01-16 04:45:34Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2025-01-16 12:45:34 +0800 (週四, 16 一月 2025) $
# $Revision: 15430 $

import logging
import re

from typing import List

import mailparser

from mailparser import MailParser

from .DataTypes import Attachment, MailData, RuleType, Target

logger = logging.getLogger(__name__)

class MailDataHandler:
    @classmethod
    def _parseMail(cls, mail: MailParser) -> MailData:
        attachments = []
        for attachment in mail.attachments:
            attachments.append(
                Attachment(name=attachment.filename, content=attachment.payload, type=attachment.mail_content_type)
            )
        # mail.to => list of tuple [(name, email), (name, email)]
        return MailData(
            sender=mail.from_[0][1],
            receivers=[email for _, email in mail.to],
            subject=mail.subject,
            body=mail.body,
            attachments=attachments,
            msg=mail.message_as_string,
            date=mail.date,
            textPlain=mail.text_plain,
            textHTML=mail.text_html,
            timezone=mail.timezone
        )

    @classmethod
    def parseEML(cls, emlFilePath: str) -> MailData:
        mail = mailparser.parse_from_file(emlFilePath)
        return cls._parseMail(mail)


class RuleHandler:

    @classmethod
    def _isMatch(cls, pattern: re.Pattern, text: str) -> bool:
        return bool(re.search(pattern, text))

    @classmethod
    def _matchesExist(cls, pattern: re.Pattern, items: List[str]) -> bool:
        return any(cls._isMatch(pattern, item) for item in items)

    @classmethod
    def _evaluateMatchCondition(cls, matches: bool, ruleType: RuleType) -> bool:
        if matches and ruleType in [RuleType.IS_REGEX.value, RuleType.INCLUDE.value]:
            return True
        if not matches and ruleType in [RuleType.IS_NOT_REGEX.value, RuleType.EXCLUDE.value]:
            return True
        return False

    @classmethod
    def isRuleMatched(cls, ruleType: RuleType, target: Target, value: str, mailData: MailData) -> bool:
        # check if the mail data match the rule
        if ruleType == RuleType.PROMPT.value:
            raise ValueError("Prompt rule is not allowed to be executed directly. Use LLMRuleHandler instead.")

        if ruleType in [RuleType.INCLUDE.value, RuleType.EXCLUDE.value]:
            # turn the value into regex form
            value = rf".*{re.escape(value)}.*"

        # Compile the regex pattern
        pattern = re.compile(value)

        if target == Target.RECEIVER.value:
            receivers = mailData.targetValue(target)
            if receivers:
                matches = cls._matchesExist(pattern, receivers)
                return cls._evaluateMatchCondition(matches, ruleType)
            return False

        # For all targets, if any of the targets match, return True
        if target == Target.ALL.value:
            for _target, items in mailData.targetMapper.items():
                if _target == Target.RECEIVER.value:
                    matches = cls._matchesExist(pattern, items)
                else:
                    matches = cls._isMatch(pattern, items)

                if cls._evaluateMatchCondition(matches, ruleType):
                    return True
            return False

        # For single target
        targetValue = mailData.targetValue(target)
        isTargetMatch = cls._isMatch(pattern, targetValue)
        return cls._evaluateMatchCondition(isTargetMatch, ruleType)
