#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 12785 2021-02-25 16:33:33Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License, 
# you may not use this file except in compliance with the License. 
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and 
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2021-02-26 00:33:33 +0800 (Fri, 26 Feb 2021) $
# $Revision: 12785 $ 

import json
import logging
import hashlib
import datetime
import base64
import urllib
import six
import re

from collections import OrderedDict

import requests

from lxml import etree
from django.shortcuts import render
from django.contrib.sites.shortcuts import get_current_site
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.shortcuts import redirect, get_object_or_404
from django.contrib import messages
from django.views import generic
from django.utils.translation import ugettext as _
from django.conf import settings
from django.http import HttpResponse, Http404, HttpResponseRedirect
from django.utils import timezone
from django.template.loader import get_template
from django.template import Context
from django.db.models import Q
from paypal.express.views import SuccessResponseView
from oscar.core.loading import get_model, get_class
from phonenumber_field.phonenumber import to_python
from oscar.core.compat import get_user_model
from oscar import VERSION

from Iuppiter.Encoding import utf8
from Iuppiter.Encryption import Encryptor
from Iuppiter.DjangoUtil import DJANGO_VERSION, reverse

from Iuno.shop import forms
from Iuno.shop import buildAPIForm
from Iuno.shop.models import (
    ECPayTrade, AllPayTrade, SpgatewayTrade, CathaybkCreditTrade,
    ECPayLogisticTrade, OrderOrdererAddress, OrdererAddress, 
    BonusSettings, BonusRecord, Bonus, calcReceiveBouns, getATMInfo, TradeSession)
from Iuno.shop.Util import createCheckValue

_PaymentDetailsView = get_class('checkout.views', 'PaymentDetailsView')

Source = get_model('payment', 'Source')
RawHTML = get_model('promotions', 'RawHTML')
SourceType = get_model('payment', 'SourceType')
Basket = get_model('basket', 'Basket')
Order = get_model('order', 'Order')
Selector = get_class('partner.strategy', 'Selector')
try:
    Applicator = get_class('offer.applicator', 'Applicator')
except ModuleNotFoundError:
    # fallback for django-oscar<=1.1
    Applicator = get_class('offer.utils', 'Applicator')

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
logger.addHandler(logging.StreamHandler())

fileHandler = logging.FileHandler('Payment.log')
fileHandler.setLevel(logging.WARNING)
logger.addHandler(fileHandler)

BONUS_VOUCHER_CODE = r"^BONUS-\d*-\d*-[a-fA-F\d]{32}$"

DEVELOPMENT = 1

SPGATEWAY = 'spgateway'
ALLPAY = 'allpay'
ECPAY = 'ecpay'
ECPAY_LOGISTIC = 'ecpay_logistic'
ECPAY_MAP = 'ecpay_map'
ATM = 'ATM'

CATHAYBK_CREDIT = 'cathaybk-credit' # /cathaybk-credit/checkxml/
CATHAYBK_CUP = 'cathaybk-cup' # /cathaybk-cup/checkxml/
    
class SuccessResponseView(SuccessResponseView):
    def get_shipping_address(self, basket):
        return _PaymentDetailsView.get_shipping_address(self, basket)

# patch
from oscar.apps.checkout import views
from oscar.apps.dashboard.orders.views import OrderListView
from oscar.core.compat import UnicodeCSVWriter

if DJANGO_VERSION >= 10900: # django >= 1.9.0
    from collections import OrderedDict as SortedDict
else:
    from django.utils.datastructures import SortedDict
from oscar.core.utils import format_datetime

def get_context_data(self, **kwargs):
    kwargs = super(views.PaymentDetailsView, self).get_context_data(**kwargs)
    
    submission = self.build_submission()

    ecpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ECPAY_ENABLE'):
        ecpayEnable = settings.IUNO_SHOP_ECPAY_ENABLE

    allpayEnable = False
    if hasattr(settings, 'IUNO_SHOP_ALLPAY_ENABLE'):
        allpayEnable = settings.IUNO_SHOP_ALLPAY_ENABLE

    spgatewayEnable = False
    if hasattr(settings, 'IUNO_SHOP_SPGATEWAY_ENABLE'):
        spgatewayEnable = settings.IUNO_SHOP_SPGATEWAY_ENABLE

    cathaybkEnable = False
    if hasattr(settings, 'IUNO_SHOP_CATHAYBK_ENABLE'):
        cathaybkEnable = settings.IUNO_SHOP_CATHAYBK_ENABLE
        
    atmEnable = False
    if hasattr(settings, 'IUNO_SHOP_ATM_ENABLE'):
        atmEnable = settings.IUNO_SHOP_ATM_ENABLE
    
    if ecpayEnable:
        kwargs['ecpay'] = buildAPIForm(submission, self.request, ECPAY)
        kwargs['ecpay_map'] = \
            buildAPIForm(submission, self.request, ECPAY_MAP)
        kwargs['ecpay_logistic'] = \
            buildAPIForm(submission, self.request, ECPAY_LOGISTIC)
    if allpayEnable:
        kwargs['allpay'] = buildAPIForm(submission, self.request, ALLPAY)
    if spgatewayEnable:
        kwargs['spgateway'] = buildAPIForm(
                                        submission, self.request, SPGATEWAY)
    if cathaybkEnable:
        kwargs['cathaybk'] = buildAPIForm(
                                        submission, self.request, 'cathaybk')                                       
    if atmEnable:
        kwargs['ATM'] = buildAPIForm(
                                        submission, self.request, ATM) 

    if self.request.session.get('logisticMap'):
        mapData = json.loads(self.request.session['logisticMap'])
        kwargs.update(**mapData)
    return kwargs

views.PaymentDetailsView.get_context_data = get_context_data

def download_selected_orders(self, request, orders):
    response = HttpResponse(content_type='text/csv')
    response['Content-Disposition'] = 'attachment; filename=%s' \
        % self.get_download_filename(request)
    writer = UnicodeCSVWriter(open_file=response)

    metaData = (('number', _('Order number')),
                    ('value', _('Order value')),
                    ('date', _('Date of purchase')),
                    ('num_items', _('Number of items')),
                    ('status', _('Order status')),
                    ('customer', _('Customer email address')),
                    ('shipping_address_name', _('Deliver to name')),
                    ('billing_address_name', _('Bill to name')),
                    ('address', _('Address')),
                    ('phone', _('Phone')),
                    ('product', _('Product')),
                    )
    columns = SortedDict()
    for k, v in metaData:
        columns[k] = v

    writer.writerow(columns.values())
    for order in orders:
        row = columns.copy()
        row['number'] = order.number
        row['value'] = order.total_incl_tax
        row['date'] = format_datetime(order.date_placed, 'DATETIME_FORMAT')
        row['num_items'] = order.num_items
        row['status'] = order.status
        row['customer'] = order.email
        row['address'] = ''
        
        phone = '%s' % order.shipping_address.phone_number
        
        # FIXME:
        # 目前電話去掉+886換0是給台灣用的
        row['phone'] = phone.replace('+886', '0')
        
        row['product'] = ''
        
        for field in order.shipping_address.active_address_fields():
            row['address'] += "%s \n" % field
        
        for i, line in enumerate(order.basket.lines.all(), 1):
            product = line.product
            attributes = product.attribute_summary
            description = line.description
            row['product'] += \
                "%d : %s(%s) X %d \n" % (i, description, 
                                            attributes, line.quantity)
        
        if order.shipping_address:
            row['shipping_address_name'] = order.shipping_address.name
        else:
            row['shipping_address_name'] = ''
        if order.billing_address:
            row['billing_address_name'] = order.billing_address.name
        else:
            row['billing_address_name'] = ''
        writer.writerow(row.values())
      
    if six.PY3:
        response.content = '\xEF\xBB\xBF' + response.content.decode('utf8')
    else:
        response.content = '\xEF\xBB\xBF' + response.content
    
    return response

OrderListView.download_selected_orders = download_selected_orders

# https://code.nuwainfo.com/trac/Mercurius/ticket/3155#ticket
from django.db import models

class OrderManager(models.Manager):
    def get_queryset(self):
        return Order._default_manager.exclude(status=u'測試用訂單')

Order.objects = OrderManager()

# https://code.nuwainfo.com/trac/Mercurius/ticket/3282#ticket
OrderCreator = get_class('order.utils', 'OrderCreator')
ShippingAddress = get_model('order', 'ShippingAddress')

_placeOrder = OrderCreator.place_order
def placeOrder(self, basket, total,  # noqa (too complex (12))
               shipping_method, shipping_charge, user=None,
               shipping_address=None, billing_address=None,
               order_number=None, status=None, request=None, **kwargs):
    order = _placeOrder(
        self, basket, total, 
        shipping_method, shipping_charge, user=user,
        shipping_address=shipping_address, billing_address=billing_address,
        order_number=order_number, status=status, request=request, **kwargs)
    
    # send mail
    msg = "%s: %s" % (_("Order number"), order_number)
    sendMessageToAdmin(_("Customer order notification"), msg)
   
    if user:
        if (user.is_authenticated and 
            OrdererAddress.objects.filter(user=user).exists()):
            orderAddress = OrdererAddress.objects.get(user=user)
            
            address = ShippingAddress.objects.create(
                first_name=orderAddress.address.first_name,
                last_name=orderAddress.address.last_name,
                line1=orderAddress.address.line1,
                line4=orderAddress.address.line4,
                country=orderAddress.address.country,
                postcode=orderAddress.address.postcode,
            )
            
            orderOrderAddress, created = \
                OrderOrdererAddress.objects.get_or_create(
                    order=order,
                    defaults={
                        'address': address,
                    }
                )
        
        # bonus   
        config, created = BonusSettings.objects.get_or_create(id=1)
        
        if user.is_authenticated:
            b, created = Bonus.objects.get_or_create(user=user)
                
            # receive
            receive = calcReceiveBouns(total.excl_tax, shipping_charge.excl_tax)
                
            # use
            use = 0
            bonusVoucher = None
            for v in basket.vouchers.all():
                if re.match(BONUS_VOUCHER_CODE, v.code):
                    bonusVoucher = v
              
            # len(BONUS-[bouns use]-[bouns discount]-XXXX.split("-")) == 4
            if bonusVoucher:
                if len(bonusVoucher.code.split("-")) == 4: 
                    use = int(bonusVoucher.code.split("-")[1])
                    
            # update use
            b.quantity = b.quantity - use
            b.save()
            
            if config.receiveStatus == settings.OSCAR_INITIAL_ORDER_STATUS:
                # update receive
                b.quantity = b.quantity + receive
                b.save()
                
                messages.info(
                    request, _("You earn %d bonus on this order.") % receive)
            else:
                messages.info(
                    request, 
                    _("When you do complete the order transaction, "
                      "you will earn %d bonus on this order.") % receive)
                receive = 0 # 只先記錄 use 的部分
                      
            # record
            record = BonusRecord.objects.createBonusRecord(
                user=user,
                order=order,
                use=use,
                receive=receive,
            )
            
        
            
    return order
    
OrderCreator.place_order = placeOrder

# https://code.nuwainfo.com/trac/Mercurius/ticket/3273#ticket
if VERSION >= (2, 0, 0):
    from oscar.apps.dashboard.catalogue import views as dashboardCatalogueViews
    from oscar.apps.dashboard.catalogue import \
        tables as dashboardCatalogueTables
    from django_tables2 import A, TemplateColumn

    Product = get_model('catalogue', 'Product')
    _ProductTable = get_class('dashboard.catalogue.tables', 'ProductTable')

    class ProductTable(_ProductTable):
        isPublic = TemplateColumn(
            verbose_name=_('Is public'),
            template_name='shop/tables/ProductRowIsPublic.html',
            accessor=A('is_public'),
            order_by='is_public')
        actions = TemplateColumn(
            verbose_name=_('Actions'),
            template_name='shop/tables/ProductRowActions.html',
            orderable=False)

        class Meta(_ProductTable.Meta):
            sequence = ('title', 'upc', 'image', 'product_class', 'variants',
                        'stock_records', '...', 'isPublic', 
                        'date_updated', 'actions')

    dashboardCatalogueTables.ProductTable = ProductTable  
    
    class ProductPublicView(generic.View):
    
        def get(self, request, *args, **kwargs):
            pk = kwargs.get('pk', None)
            action = kwargs.get('action', None)
            
            if pk and action in ['public', 'not_public']:
                product = Product.objects.get(id=pk)
                if action == 'public':
                    product.is_public = True
                    messages.info(
                        request, _("Set Product to public successfully."))
                else:    
                    product.is_public = False
                    messages.info(
                        request, _("Set Product to not public successfully."))
                product.save()
                return redirect('dashboard:catalogue-product-list')
            else:
                raise Http404()

# https://code.nuwainfo.com/trac/Mercurius/ticket/3156#ticket
from django.core.mail import send_mail
from des.models import DynamicEmailConfiguration
from django.contrib.sites.models import Site
from oscar.apps.order.processing import EventHandler

def sendOrderEmail(order, subjectTemp, htmlTemp, txtTemp, context={}):
    try:
        temp = get_template(subjectTemp)
        ctx = {
            "order": order,
            "site": Site.objects.get(id=settings.SITE_ID)
        }
        ctx.update(context)
        subject = temp.render(ctx)
        
        temp = get_template(htmlTemp)
        htmlContent = temp.render(ctx)
                    
        temp = get_template(txtTemp)
        txtContent = temp.render(ctx)
        
        if order.user:
            email = order.user.email
        else:
            email = order.guest_email
                            
        success = send_mail(
            subject,
            txtContent,
            DynamicEmailConfiguration.objects.get().from_email,
            [email,],
            html_message=htmlContent,
            fail_silently=False,
        )
                            
        if success:
            logger.info("Send order email success: %s" % email)
        else:
            logger.error("Send order email fail: %s" % email)
    except Exception as e:
        logger.error("Error send order email: %s" % e)
        
def sendMessageToAdmin(subject, message):
    try:
        if hasattr(settings, 'IUNO_SHOP_SEND_MAIL_ADMINS'):
            emails = settings.IUNO_SHOP_SEND_MAIL_ADMINS
        else:
            User = get_user_model()
            staff = User.objects.filter(Q(is_staff=True) | Q(is_superuser=True))
            emails = [u.email for u in staff]
        
        if emails:
            send_mail(
                subject,
                message,
                DynamicEmailConfiguration.objects.get().from_email,
                emails,
                fail_silently=False,
            )
    except Exception as e:
        logger.error(
            "Error send order email: %s, %s, %s" % (str(e), subject, message))
    

def handle_order_status_change(self, order, new_status, note_msg=None):
    """
    Handle a requested order status change

    This method is not normally called directly by client code.  The main
    use-case is when an order is cancelled, which in some ways could be
    viewed as a shipping event affecting all lines.
    """
    order.set_status(new_status)
    
    hasSettings = hasattr(settings, 'OSCAR_PROCESSING_SEND_EMAIL_STATUS')
    
    if (hasSettings and 
        new_status == settings.OSCAR_PROCESSING_SEND_EMAIL_STATUS):
        sendOrderEmail(
            order, 
            'shop/customer/emails/BeingProcessedSubject.txt', 
            'shop/customer/emails/BeingProcessedBody.html', 
            'shop/customer/emails/BeingProcessedBody.txt'
        )
    hasSettings = hasattr(settings, 'OSCAR_ARRIVE_SEND_EMAIL_STATUS')
    
    if hasSettings and new_status == settings.OSCAR_ARRIVE_SEND_EMAIL_STATUS:
        sendOrderEmail(
            order, 
            'shop/customer/emails/ArriveSubject.txt', 
            'shop/customer/emails/ArriveBody.html', 
            'shop/customer/emails/ArriveBody.txt'
        )
        
    
    if note_msg:
        self.create_note(order, note_msg)
    
    # bonus
    config = BonusSettings.objects.get(id=1)
        
    if (not order.is_anonymous and 
        config.receiveStatus == new_status):
        user = order.user
        b, created = Bonus.objects.get_or_create(user=user)
        
        # receive
        receive = order.canReceiveBonus
            
        # use
        use = 0
        bonusVoucher = None
        for v in order.basket_discounts.all():
            if re.match(BONUS_VOUCHER_CODE, v.voucher_code):
                bonusVoucher = v
          
        # len(BONUS-[bouns use]-[bouns discount]-XXXX.split("-")) == 4
        if bonusVoucher:
            if len(bonusVoucher.voucher_code.split("-")) == 4: 
                use = int(bonusVoucher.voucher_code.split("-")[1])
                
        # update 
        b.quantity = b.quantity + receive
        b.save()
        
        # record
        record = BonusRecord.objects.createBonusRecord(
            user=user,
            order=order,
            use=0, # 已在購買訂單時紀錄
            receive=receive,
        )

EventHandler.handle_order_status_change = handle_order_status_change

# https://code.nuwainfo.com/trac/Mercurius/ticket/3285#ticket
from oscar.apps.offer import conditions

def is_satisfied(self, offer, basket):
    """
    Determines whether a given basket meets this condition
    """
    num_matches = 0
    for line in basket.all_lines():
        if self.can_apply_condition(line):
            num_matches += line.quantity_without_offer_discount(offer)
        if num_matches >= self.value:
            return True
    return False 
    
conditions.CountCondition.is_satisfied = is_satisfied

class BonusSettingsView(generic.TemplateView):
    
    template_name = 'shop/offer/dashboard/Bonus.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)    
        config, created = BonusSettings.objects.get_or_create(id=1)        
        context['form'] = forms.BonusSettingsForm(instance=config)
        return context

    def post(self, request, *args, **kwargs):
        config, created = BonusSettings.objects.get_or_create(id=1)  
        form = forms.BonusSettingsForm(request.POST, instance=config)
        if form.is_valid():
            form.save()
            messages.success(request, _("Settings saved"))
            return redirect("bonusSettings")
        else:
            messages.error(request, _("Form error"))
            return render(request, self.template_name, self.get_context_data())
        
from oscar.apps.basket import views as basketViews

originBasketViewGetContextData =  basketViews.BasketView.get_context_data

def basketViewGetContextData(self, **kwargs):
    context = originBasketViewGetContextData(self, **kwargs)
    context['bonusForm'] = forms.BonusForm()
    return context
    
basketViews.BasketView.get_context_data = basketViewGetContextData

originVoucherRemoveViewPost = basketViews.VoucherRemoveView.post

def voucherRemoveViewPost(self, request, *args, **kwargs):
    response = redirect('basket:summary')

    voucher_id = kwargs['pk']
    if not request.basket.id:
        # Hacking attempt - the basket must be saved for it to have
        # a voucher in it.
        return response
    try:
        voucher = request.basket.vouchers.get(id=voucher_id)
    except ObjectDoesNotExist:
        messages.error(
            request, _("No voucher found with id '%s'") % voucher_id)
    else:
        request.basket.vouchers.remove(voucher)
        self.remove_signal.send(
                sender=self, basket=request.basket, voucher=voucher)
        if re.match(BONUS_VOUCHER_CODE, voucher.code):
            messages.info(
                request, _("Bouns discount removed from basket"))
        else:
            messages.info(
                request, _("Voucher '%s' removed from basket") % voucher.code)

    return response
    
basketViews.VoucherRemoveView.post = voucherRemoveViewPost

Voucher = get_model('voucher', 'Voucher')
ConditionalOffer = get_model('offer', 'ConditionalOffer')
Benefit = get_model('offer', 'Benefit')
Range = get_model('offer', 'Range')
Condition = get_model('offer', 'Condition')

PageTitleMixin = get_class('customer.mixins', 'PageTitleMixin')

class CustomerBonusListView(PageTitleMixin, generic.TemplateView):
    template_name = 'shop/bonus/BonusList.html'
    page_title = _('Bonus')
    active_tab = 'bonus'
    
    def get_context_data(self, **kwargs):
        ctx = super().get_context_data(**kwargs)
        return ctx

def useBonus(request):
    quantity = int(request.POST.get('quantity', 0))
    
    if quantity <= 0:
        messages.error(request, _("Use bonus must exceed 0."))
        return redirect("basket:summary")
    b, created = Bonus.objects.get_or_create(user=request.user)
    config, created = BonusSettings.objects.get_or_create(id=1)
    
    if quantity > b.quantity:
        messages.error(request, _("Not enough bonus."))
        return redirect("basket:summary")
    if not config.limit == -1 and quantity > config.limit:
        messages.error(
            request, _("Use bonus over the upper limit %d.") % config.limit)
        return redirect("basket:summary")
    
    # include exclude 
    totalCanUseBonus = 0
    for line in request.basket.all_lines():
        product = line.product
        
        if config.includeRange:
            if not config.includeRange.contains_product(product):
                messages.info(
                    request, _("%s can't use bonus.") % product)
                continue
        if config.excludeRange:
            if config.excludeRange.contains_product(product):
                messages.info(
                    request, _("%s can't use bonus.") % product)
                continue
                    
        totalCanUseBonus += int(line.line_price_excl_tax)
    
    if quantity > totalCanUseBonus:
        messages.info(
            request, _("Only use %d bonus.") % totalCanUseBonus)
        quantity = totalCanUseBonus
        
    value = quantity * config.discount
    
    # create voucher
    if Range.objects.filter(includes_all_products=True).exists():
        range = Range.objects.filter(includes_all_products=True).first()
    else:
        range, created = Range.objects.get_or_create(
            name=_("All Products"),
            includes_all_products=True,
        )
    if Condition.objects.filter(
        range=range, type=Condition.COUNT, value=1).exists():
        condition = Condition.objects.filter(
            range=range, type=Condition.COUNT, value=1).first()
    else:
        condition, created = Condition.objects.get_or_create(
            range=range,
            type=Condition.COUNT,
            value=1, # 商品至少有一個
        )

    if Benefit.objects.filter(
        range=range, type=Benefit.FIXED, value=value).exists():
        benefit = Benefit.objects.filter(
            range=range, type=Benefit.FIXED, value=value).first()
    else:
        benefit, created = Benefit.objects.get_or_create(
            range=range,
            type=Benefit.FIXED,
            value=value
        )
    
    name = _("Bouns %(quantity)d discount %(value)d for %(username)s") % {
        'value': value, 'username': request.user.username, 'quantity': quantity}
    offer, created = ConditionalOffer.objects.get_or_create(
        name=_("Offer for voucher '%s'") % name,
        offer_type=ConditionalOffer.VOUCHER,
        benefit=benefit,
        condition=condition,
        exclusive=False, # False: 可與其他優惠同時使用
    )
    code = "BONUS-%d-%d-%s" % (quantity, value, hashlib.md5(
        (name + settings.SECRET_KEY).encode()).hexdigest())
    voucher, created = Voucher.objects.get_or_create(
        name=name,
        code=code.upper(),
        usage=Voucher.MULTI_USE,
        defaults={
            'start_datetime': datetime.datetime(1999, 1, 1), # FIXME
            'end_datetime': datetime.datetime(2099, 12, 31), # FIXME
        }
    )
    voucher.num_basket_additions += 1
    voucher.save()
    voucher.offers.add(offer)
    
    # add voucher
    for v in request.basket.vouchers.all():
        if re.match(BONUS_VOUCHER_CODE, v.code):
            request.basket.vouchers.remove(v)
    request.basket.vouchers.add(voucher)
        
    return redirect("basket:summary")
    
def updateBonus(request, *args, **kwargs):
    pk = int(kwargs['pk'])
    adjust = int(request.POST.get('adjust', 0))
    note = request.POST.get('note', None)
    user = get_user_model().objects.get(id=pk)
    b, created = Bonus.objects.get_or_create(user=user)
    
    if not b.quantity + adjust >= 0:
        messages.error(
            request, _("Bonus must be greater than 0."))
        
        return redirect(reverse("dashboard:user-detail", kwargs={'pk': pk,}))
        
    # update
    b.quantity += adjust
    b.save()
    
    # record
    record = BonusRecord.objects.createBonusRecord(
        user=user,
        order=None,
        adjust=adjust,
        note=note,
    )
    
    messages.info(
        request, _("Update success"))
    
    return redirect(reverse("dashboard:user-detail", kwargs={'pk': pk,}))
    
def updateRecord(request, *args, **kwargs):
    pk = int(kwargs['pk'])
    recordId = int(kwargs['recordId'])
    date = request.POST.get("expiredTime", None)
    record = BonusRecord.objects.get(id=recordId)
    
    try:
        if date:
            record.expireTimeSetting = date
        else:
            record.expireTimeSetting = None
        record.save()
        
        messages.info(
            request, _("Update success"))
    except Exception as e:
        messages.error(
            request, _("Wrong date format, the correct format must be YYYY-MM-DD."))
        
    return redirect(reverse("dashboard:user-detail", kwargs={'pk': pk,}))
    
# paypal patch
# https://www.paypal-community.com/t5/Sandbox-Environment/
# You-are-not-signed-up-to-accept-payment-for-digitally-delivered/td-p/1468047
from paypal.express import gateway

originFetchResponse = gateway._fetch_response
def newFetchResponse(method, extraParams):
    newExtraParams = {k: v for k, v in extraParams.items() if not k.startswith("L_PAYMENTREQUEST_0_ITEMCATEGORY")}
    
    return originFetchResponse(method, newExtraParams)
    
gateway._fetch_response = newFetchResponse

    

from django_tables2 import Column
_UserTable = get_class('dashboard.users.tables', 'UserTable')

class UserTable(_UserTable):
    bonus = Column(accessor='bonus', verbose_name=_('Bonus'))
    
    class Meta(_UserTable.Meta):
        sequence = ('check', 'email', 'name', 'active', 'staff',
                    'date_registered', 'num_orders', 'bonus', 'actions',)
    
from oscar.apps.dashboard.users import \
        views as dashboardUsersViews
        
dashboardUsersViews.IndexView.table_class = UserTable

# https://code.nuwainfo.com/trac/Mercurius/ticket/3289#ticket   
@csrf_exempt
def sendConfirmCode(request):

    if request.POST:
        email = request.POST.get('email', None)
        timeStamp = request.POST.get('timeStamp', None)

        if not email or not timeStamp:
            raise PermissionDenied()

        m = hashlib.md5()
        m.update(email.encode())
        m.update(timeStamp.encode())
        code = m.hexdigest()[5:10]
        
        site = get_current_site(request)
        
        ctx = {
            'code': code,
            'site': site,
        }
        
        t = get_template('shop/customer/emails/ConfirmCodeSubject.txt')
        subject = t.render(ctx)
        t = get_template('shop/customer/emails/ConfirmCodeBody.txt')
        body = t.render(ctx)
        
        send_mail(
            subject,
            body,
            settings.DEFAULT_FROM_EMAIL,
            [request.POST['email']],
            fail_silently=False,
        )

        s = _("We have send a confirmation code to %(email)s") % {
                'email': email,}
        return HttpResponse(json.dumps(s), content_type="application/json")
        
# https://code.nuwainfo.com/trac/Mercurius/ticket/3290#ticket
from oscar.apps.dashboard.orders import views as dashboardOrderViews
from Iuno.shop.forms import OrderNoteForm

OrderNote = get_model('order', 'OrderNote')
CommunicationEventType = get_model('customer', 'CommunicationEventType')
Dispatcher = get_class('customer.utils', 'Dispatcher')

def get_order_note_form(self):
    kwargs = {
        'order': self.object,
        'user': self.request.user,
        'data': None
    }
    if self.request.method == 'POST':
        kwargs['data'] = self.request.POST
    noteId = self.kwargs.get('note_id', None)
    if noteId:
        note = get_object_or_404(OrderNote, order=self.object, id=noteId)
        if note.is_editable():
            kwargs['instance'] = note
    return OrderNoteForm(**kwargs)
    
def save_note(self, request, order):
    form = self.get_order_note_form()
    if form.is_valid():
        note = form.save()
        if note.note_type == form.DISPLAY_FOR_CUSTOMER:
            code = 'NOTE_PUBLIC'
            ctx = {
                'site': get_current_site(request),
                'order': order,
                'note': note,
            }
            msg = CommunicationEventType.objects.get_and_render(code, ctx)
            Dispatcher().dispatch_order_messages(order, msg)
        messages.success(self.request, _("Note saved"))
        return self.reload_page(fragment='notes')

    ctx = self.get_context_data(note_form=form, active_tab='notes')
    return self.render_to_response(ctx)
    
dashboardOrderViews.OrderDetailView.get_order_note_form = get_order_note_form
dashboardOrderViews.OrderDetailView.save_note = save_note

# https://code.nuwainfo.com/trac/Mercurius/ticket/3284#ticket
try:
    import xapian

    from oscar.apps.search import facets
    from oscar.apps.search import forms as searchForms

    def munge_query_facet(self, key, facet, clean_data):
        clean_data[key] = {
            'name': facet['name'],
            'results': []}
        # Loop over the queries in OSCAR_SEARCH_FACETS rather than the returned
        # facet information from the search backend.
        for field_value, query in facet['queries']:
            field_name = facet['field']
            is_faceted_already = field_name in self.selected_facets

            match = '%s_exact:%s' % (field_name, query)
            if match not in self.facet_counts['queries'][field_name]:
                # This query was not returned
                datum = {
                    'name': field_value,
                    'count': 0,
                    'show_count': True,
                    'disabled': True,
                }
            else:
                count = self.facet_counts['queries'][field_name][match]
                datum = {
                    'name': field_value,
                    'count': count,
                    'show_count': not is_faceted_already,
                    'disabled': count == 0 and not is_faceted_already,
                    'selected': False,
                }
                if query in self.selected_facets.get("%s_exact" % field_name, []):
                    # Selected
                    datum['selected'] = True
                    datum['show_count'] = True
                    url = self.base_url.remove_query_param(
                        'selected_facets', match)
                    datum['deselect_url'] = self.strip_pagination(url)
                else:
                    url = self.base_url.append_query_param(
                        'selected_facets', match)
                    datum['select_url'] = self.strip_pagination(url)
            clean_data[key]['results'].append(datum)
           
    def xapianSearch(self):
        # We replace the 'search' method from FacetedSearchForm, so that we can
        # handle range queries
        # Note, we call super on a parent class as the default faceted view
        # escapes everything (which doesn't work for price range queries)
        sqs = super(searchForms.FacetedSearchForm, self).search()

        # We need to process each facet to ensure that the field name and the
        # value are quoted correctly and separately:
        for field, values in self.selected_multi_facets.items():
            if not values:
                continue
            if field in searchForms.VALID_FACET_QUERIES:
                # Query facet - don't wrap value in speech marks and don't
                # clean value. Query values should have been validated by this
                # point and so we don't need to escape them.
                q = ["%s:%s" % (field, v) for v in values]
                sqs = sqs.narrow(" ".join(q))
            else:
                # Field facet - clean and quote the values
                clean_values = [
                    '"%s"' % sqs.query.clean(val) for val in values]
                sqs = sqs.narrow('%s:(%s)' % (
                    field, " OR ".join(clean_values)))

        if self.is_valid() and 'sort_by' in self.cleaned_data:
            sort_field = self.SORT_BY_MAP.get(
                self.cleaned_data['sort_by'], None)
            if sort_field:
                sqs = sqs.order_by(sort_field)

        return sqs
            
    facets.FacetMunger.munge_query_facet = munge_query_facet
    searchForms.SearchForm.search = xapianSearch
except Exception as e:
    # no use xapian
    pass

# ------------------------------------------------------------------------------
# /cathaybk/preview/
# /cathaybk-credit/checkxml/
@csrf_exempt
def returnCathaybkCreditXml(request):
    if request.POST:
        try:
            # get data
            xml = request.POST.get("strRsXML", None)
            if not xml:
                raise ValueError("Not strRsXM in post data.")

            root = etree.fromstring(utf8(xml))

            data = {}

            for child in root:
                if child.tag == "CAVALUE":
                    data["CAVALUE"] = child.text
                if child.tag in ["ORDERINFO", "AUTHINFO"]:
                    for child2 in child:
                        data[child2.tag] = child2.text

            #record data
            def parseDate(date):
                year = date[0:4]
                month = date[4:6]
                day = date[6:8]
                return "%s-%s-%s" % (year, month, day)

            if not data.get("AUTHCODE", ""):
                tradeNo = "NO_TRADE_NO"
            else:
                tradeNo = data.get("AUTHCODE", "NO_TRADE_NO")

            cathaybkCreditTrade = CathaybkCreditTrade.objects.create(
                tradeNo=tradeNo,
                tradeAmt=int(data.get("AMOUNT", "0")),
                merchantId=data.get("STOREID", ""),
                merchantTradeNo=data.get("ORDERNUMBER", ""),
                rtnCode=data.get("AUTHSTATUS", ""),
                paymentDate=parseDate(data.get("AUTHTIME", "11111111")),
                rtnMsg=data.get("AUTHMSG", ""),
                cardNo=data.get("CARDNO", ""),
            )
            cathaybkCreditTrade.save()

            # check
            status = data.get('AUTHSTATUS', None)
            if not status == '0000':
                raise ValueError("AUTHSTATUS: %s" % status)
        except Exception as e:
            logger.error('%s API ERROR: %s\n%s' % (
                    CATHAYBK_CREDIT, request.POST, str(e)))
        finally:
            # return xml
            scheme = request.scheme
            domain = request.META['HTTP_HOST']

            url = "%s://%s/%s/preview/" % (scheme, domain, CATHAYBK_CREDIT)

            checkValue = hashlib.md5(
                domain + settings.CATHAYBK_API_CUBKEY).hexdigest()
            
            temp = get_template('shop/payment/CathaybkReturn.xml')
            context = Context({
                'checkValue': checkValue,
                'returnUrl': url,
            })
            rtXml = temp.render(context)

            return HttpResponse(rtXml)
    else:
        logger.error("returnCathaybkCreditXml run to GET")
        scheme = request.scheme
        domain = request.META['HTTP_HOST']

        url = "%s://%s/%s/preview/" % (scheme, domain, CATHAYBK_CREDIT)

        checkValue = hashlib.md5(
            domain + settings.CATHAYBK_API_CUBKEY).hexdigest()
        
        temp = get_template('shop/payment/CathaybkReturn.xml')
        context = Context({
            'checkValue': checkValue,
            'returnUrl': url,
        })
        rtXml = temp.render(context)

        return HttpResponse(rtXml)
        
def sendPaymentForm(request):
    if request.method == 'POST':
        encryptor = Encryptor(settings.SECRET_KEY.encode())
        formData = request.POST.get("formData")
        if not formData:
            raise Http404()
        request.session['preBasketId'] = request.basket.id
        request.basket.freeze()
        d = base64.b64decode(formData)
        form = encryptor.decrypt(d)
        if six.PY3:
            form = form.decode()
            
        TradeSession.objects.update_or_create(
            basket=request.basket,
            defaults={
                'sessionKey': request.COOKIES.get(settings.SESSION_COOKIE_NAME),
            }
        )
        
        return render(
                request, 
                "shop/payment/PaymentRedirect.html", 
                {'form': form,}
            )
    else:
        raise Http404()
    

class SuccessResponseView2(_PaymentDetailsView):
    preview = True
    http_method_names = ['post',]
    
    @method_decorator(csrf_exempt)
    def dispatch(self, request, *args, **kwargs):
        self.apiName = kwargs['name']
        return super(SuccessResponseView2, self).dispatch(request, 
                                                         *args, **kwargs)
                                                         
    @property
    def pre_conditions(self):
        return []
        
    def loadFrozenBasket(self, basketId):
        # Lookup the frozen basket that this txn corresponds to
        try:
            basket = Basket.objects.get(id=basketId, status=Basket.FROZEN)
        except Basket.DoesNotExist:
            return None

        # Assign strategy to basket instance
        if Selector:
            basket.strategy = Selector().strategy(self.request)

        # Re-apply any offers
        Applicator().apply(basket, self.request.user, request=self.request)

        return basket

    def get_pre_conditions(self, request):
        conditions = super(
            SuccessResponseView2, self).get_pre_conditions(request)
        if self.apiName == ECPAY_MAP:
            conditions = [
                'check_basket_is_not_empty',
                'check_basket_is_valid',
                'check_user_email_is_captured',
            ]
        return conditions
        
    def post(self, request, *args, **kwargs):
        # apiName
        self.apiName = kwargs['name']
        
        if hasattr(settings, "IUNO_SHOP_SEND_MAIL_WHEN_USER_PAID"):
            if settings.IUNO_SHOP_SEND_MAIL_WHEN_USER_PAID:
                data = ""
                for k, v in request.POST.items():
                    data += " %s: %s" % (k, v)
                msg = "%s: %s\n%s: \n%s" % (
                    _('User'), request.user.username, _("Data"), data)
                
                sendMessageToAdmin(_("User checkout notification"), msg)
        
        if not self.apiName in [
            SPGATEWAY, ALLPAY, ECPAY, CATHAYBK_CREDIT, 
            CATHAYBK_CUP, ECPAY_MAP, ECPAY_LOGISTIC, ATM]:
            raise ValueError("Not pay api name: %s" % apiName)

        try:
            if self.apiName in [ALLPAY, ECPAY]:
                # data
                self.tradeNo = request.POST.get('TradeNo')
                self.tradeAmt = request.POST.get('TradeAmt')
                self.merchantID = request.POST.get('MerchantID')
                self.merchantTradeNo = request.POST.get('MerchantTradeNo')
                self.rtnCode = request.POST.get('RtnCode')
                self.rtnMsg = request.POST.get('RtnMsg')
                self.payAmt = request.POST.get('MerchantID')
                self.tradeDate = request.POST.get('TradeDate')
                self.paymentType = request.POST.get('PaymentType')
                self.simulatePaid = request.POST.get('SimulatePaid')
                self.paymentDate = request.POST.get('PaymentDate')
                self.paymentTypeChargeFee = request.POST.get(
                                                        'PaymentTypeChargeFee')

                # basket
                index = request.POST.get('MerchantTradeNo').find('BASKETID')
                self.basketId = int(
                                request.POST.get('MerchantTradeNo')[index + 8:])

                # error
                if self.apiName == ALLPAY:
                    result = not str(self.rtnCode) == str(0)
                else:
                    result = self.rtnMsg == 'ERROR'
                if result:
                    msg = _(
                        "There was an error in the payment. Please try again.")
                    messages.error(self.request, msg)
                    logger.error(
                        '%s API ERROR: %s' % (self.apiName, request.POST))
                    return redirect('checkout:payment-details')
            
                # record trade data
                if self.apiName == ECPAY:
                    ecpayTrade = ECPayTrade.objects.create(
                        tradeNo=self.tradeNo,
                        tradeAmt=int(self.tradeAmt),
                        merchantId=self.merchantID,
                        merchantTradeNo=self.merchantTradeNo,
                        rtnCode=self.rtnCode,
                        payAmt=int(self.payAmt),
                        tradeDate=self.tradeDate.replace('/', '-'),
                        paymentType=self.paymentType,
                        simulatePaid=self.simulatePaid,
                        paymentDate=self.paymentDate.replace('/', '-'),
                        paymentTypeChargeFee=self.paymentTypeChargeFee,
                        rtnMsg=self.rtnMsg,
                    )
                    ecpayTrade.save()
                else:
                    allpayTrade = AllPayTrade.objects.create(
                        tradeNo=self.tradeNo,
                        tradeAmt=int(self.tradeAmt),
                        merchantId=self.merchantID,
                        merchantTradeNo=self.merchantTradeNo,
                        rtnCode=self.rtnCode,
                        payAmt=int(self.payAmt),
                        tradeDate=self.tradeDate.replace('/', '-'),
                        paymentType=self.paymentType,
                        simulatePaid=self.simulatePaid,
                        paymentDate=self.paymentDate.replace('/', '-'),
                        paymentTypeChargeFee=self.paymentTypeChargeFee,
                        rtnMsg=self.rtnMsg,
                    )
                    allpayTrade.save()
                    
                if "return" in request.GET:
                    return HttpResponse("1|OK")
            elif self.apiName == SPGATEWAY:
                # data
                data = json.loads(request.POST.get('JSONData', '{}'))
                
                self.status = data.get('Status')
                self.message = data.get('Message')
                self.result = json.loads(data.get('Result', '{}'))
                self.tradeNo = self.result.get("TradeNo")
                self.tradeAmt = self.result.get("Amt")
                self.merchantID = self.result.get("MerchantID")
                self.merchantTradeNo = self.result.get("MerchantOrderNo")
                self.tradeDate = self.result.get("PayTime")
                self.paymentType = self.result.get("PaymentType")
                
                # 信用卡才有的欄位
                self.paymentMethod = self.result.get("PaymentMethod")
                self.rtnCode = self.result.get("RespondCode")
                
                # basket
                index = self.result.get('MerchantOrderNo').find('BASKETID')
                self.basketId = int(
                    self.result.get('MerchantOrderNo')[index + 8:])

                # error
                if not self.status == 'SUCCESS':
                    msg = _(
                        "There was an error in the payment. Please try again.")
                    messages.error(self.request, msg)
                    logger.error(
                        '%s API ERROR: %s' % (self.apiName, request.POST))
                    return redirect('checkout:payment-details')

                # record trade data
                spgatewayTrade = SpgatewayTrade.objects.create(
                    tradeNo=self.tradeNo,
                    tradeAmt=int(self.tradeAmt),
                    merchantId=self.merchantID,
                    merchantTradeNo=self.merchantTradeNo,
                    rtnCode=self.rtnCode,
                    paymentDate=self.tradeDate.replace('/', '-'),
                    paymentType=self.paymentType,
                    paymentMethod=self.paymentMethod,
                    rtnMsg=self.message,
                )
                spgatewayTrade.save()
            elif self.apiName == CATHAYBK_CREDIT:
                self.status = 'XXXX'
                msg = _("There was an error in the payment. Please try again.")

                # get data
                xml = request.POST.get("strRsXML", None)
                if not xml:
                    logger.error(
                        "Not strRsXML in post data.: %s" % request.POST)
                    messages.error(self.request, msg)
                    return redirect('checkout:payment-details')

                root = etree.fromstring(utf8(xml))
                data = {}
                for child in root:
                    if child.tag == "CAVALUE":
                        data["CAVALUE"] = child.text
                    if child.tag in ["ORDERINFO",]:
                        for child2 in child:
                            data[child2.tag] = child2.text

                # get obj & check
                self.merchantTradeNo = data.get("ORDERNUMBER", "")
                if not CathaybkCreditTrade.objects.filter(
                        merchantTradeNo=self.merchantTradeNo).exists():
                    messages.error(self.request, msg)
                    return redirect('checkout:payment-details')
                cathaybkCreditTrade = CathaybkCreditTrade.objects.get(
                    merchantTradeNo=self.merchantTradeNo)
                self.status = cathaybkCreditTrade.rtnCode
                if not cathaybkCreditTrade.rtnCode == '0000':
                    messages.error(
                        self.request, 
                        msg + ": %s %s" % (cathaybkCreditTrade.rtnCode, 
                                           cathaybkCreditTrade.rtnMsg))
                    return redirect('checkout:payment-details') 

                # data
                index = cathaybkCreditTrade.merchantTradeNo.find('BASKETID')
                self.basketId = int(
                    cathaybkCreditTrade.merchantTradeNo[index + 8:])

                self.tradeNo = cathaybkCreditTrade.tradeNo
                self.tradeAmt = cathaybkCreditTrade.tradeAmt
            elif self.apiName == ECPAY_MAP:
                if hasattr(settings, "ECPAY_API_LOGISTIC_SUB_TYPES"):
                    subTypes = settings.ECPAY_API_LOGISTIC_SUB_TYPES
                else:
                    subTypes = {
                        "FAMI": u"全家便利商店",
                        "UNIMART": u"統一7-11超商" ,
                        "HILIFE": u"萊爾富",
                    }

                extra = request.POST.get("ExtraData")
                # 防呆 or submission 內會有地址資料
                splitData = extra.split(",")
                name = splitData[1][len("name="):]
                phone = splitData[0][len("phone="):]

                data = {
                    'storeName': self.request.POST.get('CVSStoreName'),
                    'storeAddress': self.request.POST.get('CVSAddress'),
                    'storeTypeName': subTypes[
                        self.request.POST.get('LogisticsSubType')],
                    'storeId': request.POST.get("CVSStoreID"),
                    'storeType': self.request.POST.get('LogisticsSubType'),
                    'storePhone': self.request.POST.get('CVSTelephone'),
                    'receiverName': name,
                    'receiverPhone': phone,
                }

                #FIXME: phone 判斷
                pyPhone = to_python(
                    data['receiverPhone'].replace('09', '+8869'))

                addressData = {
                    "first_name": " ", 
                    "last_name": data['receiverName'],
                    
                    "country_id": u'TW', 
                    "line4": data['storeAddress'], 
                    "line1": "%s%s" % (
                        data['storeTypeName'], data['storeName']),
                    'line3': u'', 
                    'line2': u'',
                    "postcode": '00000', 
                    "phone_number": pyPhone,
                    'notes': u'',
                    'state': u'',
                    'search_text': u'',
                    'title': u'',  
                    'id': None,          
                }
                
                self.checkout_session.ship_to_new_address(addressData)
                
                self.request.session['logisticMap'] = json.dumps(data)
               
                return redirect("checkout:payment-method")

            elif self.apiName == ECPAY_LOGISTIC:
                # data 
                errorMsg = None
                try:
                    self.tradeNo = request.POST.get('AllPayLogisticsID')
                    self.tradeAmt = request.POST.get('GoodsAmount')

                    self.merchantTradeNo = request.POST.get('MerchantTradeNo')
                    index = self.merchantTradeNo.find('BASKETID')
                    self.basketId = int(self.merchantTradeNo[index + 8:])

                    self.merchantID = request.POST.get('MerchantID')
                    self.rtnCode = request.POST.get('RtnCode')
                    self.rtnMsg = request.POST.get('RtnMsg')
                    
                    self.updateStatusDate = request.POST.get('UpdateStatusDate')
                    self.receiverName = request.POST.get('ReceiverName')
                    self.receiverCellPhone = request.POST.get(
                                                'ReceiverCellPhone')
                    self.receiverEmail = request.POST.get('ReceiverEmail')
                    self.logisticsType = request.POST.get('LogisticsType')
                    self.logisticsSubType = request.POST.get('LogisticsSubType')

                    self.paymentNo = request.POST.get('CVSPaymentNo')
                    self.validationNo = request.POST.get('CVSValidationNo')
                    
                    # error
                    if self.rtnMsg == 'ERROR':
                        msg = _(
                            "There was an error in the payment. "
                            "Please try again.")
                        messages.error(self.request, msg)
                        logger.error(
                            '%s API ERROR: %s' % (self.apiName, request.POST))
                        return redirect('checkout:payment-details')

                    # record
                    logisticTrade = ECPayLogisticTrade.objects.create(
                        tradeNo=self.tradeNo,
                        tradeAmt=int(self.tradeAmt),
                        merchantId=self.merchantID,
                        merchantTradeNo=self.merchantTradeNo,
                        rtnCode=self.rtnCode,
                        rtnMsg=self.rtnMsg,
                        tradeDate=self.updateStatusDate.replace('/', '-'),
                        updateStatusDate= \
                            self.updateStatusDate.replace('/', '-'),
                        receiverName=self.receiverName,
                        receiverCellPhone=self.receiverCellPhone,
                        receiverEmail=self.receiverEmail,
                        logisticsType=self.logisticsType,
                        logisticsSubType=self.logisticsSubType,
                        paymentNo=self.paymentNo,
                        validationNo=self.validationNo,
                    )

                    if self.request.session.get('paymentData'):
                        paymentData = json.loads(
                            self.request.session.get('paymentData'))
                        self.apiName = paymentData['apiName']
                        self.tradeNo = "%s|%s" % (
                            paymentData['tradeNo'], self.tradeNo)
                except Exception as e:
                    errorMsg = str(e)
                    logger.error(errorMsg)
                finally:
                    # delete session data
                    if self.request.session.get('paymentData'):
                        del self.request.session['paymentData']
                    if self.request.session.get('logisticMap'):
                        del self.request.session['logisticMap']
                    if errorMsg:
                        msg = _(
                            "There was an error in the payment. "
                            "Please try again.")
                        messages.error(self.request, msg)
                        return redirect('checkout:payment-details')
            elif self.apiName == ATM:
                self.tradeNo = '-'
                self.tradeAmt = int(request.POST.get('total'))
                self.basketId = int(request.POST.get('basketId'))

            if self.request.session.get('logisticMap'):
                self.request.basket = self.loadFrozenBasket(self.basketId)
                submission = super(
                    SuccessResponseView2, self).build_submission(**kwargs)
                form = buildAPIForm(
                    submission, 
                    self.request, 
                    ECPAY_LOGISTIC,
                    extra={"isCollection": "N",},
                )

                paymentData = {
                    'apiName': self.apiName,
                    'tradeNo': self.tradeNo,
                }
                self.request.session['paymentData'] = json.dumps(paymentData)

                ctx = {
                    'form': form,
                }
                return render(
                    request, 'shop/payment/LogisticRedirect.html', ctx)
            
            # ECPAY_LOGISTIC 不需要 loadFrozenBasket
            if not self.apiName in [ECPAY_LOGISTIC, ]:
                errorMsg = _(
                    "A problem occurred communicating with %(apiName)s "
                    "- please try again later"
                ) % {"apiName": self.apiName}
                self._basket = self.loadFrozenBasket(self.basketId)
                if not self._basket:
                    messages.error(self.request, errorMsg)
                    return redirect('checkout:payment-details')
            else:
                self._basket = request.basket
             
            user = request.user
            return self.handle_place_order_submission(request)
        except Exception as e:
            # return msg
            logger.error(str(e))

            # raise
            if settings.SERVER_MODE == DEVELOPMENT:
                raise
            
            # if payment success
            msg1 = _("There was an except error in the payment."
                     " We have notified the administrator to deal with refund.")

            # if payment not success
            msg2 = _("There was an error in the payment. Please try again.")

            # call admin
            if self.apiName == ALLPAY:
                if not self.rtnMsg == 'ERROR':
                    if not AllPayTrade.objects.filter(
                        merchantTradeNo=self.merchantTradeNo).exists():
                        logger.error(
                            '%s API ERROR(this need to refund):\n%s\n%s' % (
                                self.apiName, request.POST, str(e)))
                        messages.error(self.request, msg1)
                    else:
                        logger.error(str(e))
                        order = Order.objects.get(basket=request.basket)
                        url = "%s?order_id=%d" % (
                            reverse('checkout:thank-you'), order.id)
                        return redirect(url)
                else:
                    logger.error(
                        '%s API ERROR:\n%s\n%s' % (
                            self.apiName, request.POST, str(e)))
                    messages.error(self.request, msg2)
            elif self.apiName == ECPAY:
                if not self.rtnMsg == 'ERROR':
                    if not ECPayTrade.objects.filter(
                        merchantTradeNo=self.merchantTradeNo).exists():
                        logger.error(
                            '%s API ERROR(this need to refund):\n%s\n%s' % (
                                self.apiName, request.POST, str(e)))
                        messages.error(self.request, msg1)
                    else:
                        logger.error(str(e))
                        order = Order.objects.get(basket=request.basket)
                        url = "%s?order_id=%d" % (
                            reverse('checkout:thank-you'), order.id)
                        return redirect(url)
                else:
                    logger.error(
                        '%s API ERROR:\n%s\n%s' % (
                            self.apiName, request.POST, str(e)))
                    messages.error(self.request, msg2)
            elif self.apiName == SPGATEWAY:
                if self.status == 'SUCCESS':
                    if not SpgatewayTrade.objects.filter(
                        merchantTradeNo=self.merchantTradeNo).exists():
                        logger.error(
                            '%s API ERROR(this need to refund):\n%s\n%s' % (
                                self.apiName, request.POST, str(e)))
                        messages.error(self.request, msg1)
                    else:
                        logger.error(str(e))
                        order = Order.objects.get(basket=request.basket)
                        url = "%s?order_id=%d" % (
                            reverse('checkout:thank-you'), order.id)
                        return redirect(url)
                else:
                    logger.error(
                        '%s API ERROR:\n%s\n%s' % (
                            self.apiName, request.POST, str(e)))
                    messages.error(self.request, msg2)
            

            return redirect('checkout:payment-details')
        
        
    def build_submission(self, **kwargs):
        kwargs['basket'] = self._basket
        
        submission = super(
            SuccessResponseView2, self).build_submission(**kwargs)
        
        
        submission['payment_kwargs']['tradeNo'] = self.tradeNo
        submission['payment_kwargs']['tradeAmt'] = self.tradeAmt
        submission['payment_kwargs']['basketId'] = self.basketId
        return submission

    
    def handle_payment(self, order_number, total, **kwargs):
        tradeAmt = int(kwargs['tradeAmt'])
        tradeNo = kwargs['tradeNo']
        source_type, is_created = SourceType.objects.get_or_create(
            name=self.apiName)
        source = Source(source_type=source_type,
                        currency='TWD',
                        amount_allocated=tradeAmt,
                        amount_debited=tradeAmt,
                        reference=tradeNo) 
        
        self.add_payment_source(source)
        self.add_payment_event('Settled', tradeAmt,
                               reference=tradeNo)

class RefundView(generic.View):

    http_method_names = ['get', 'post',]
    
    def getSource(self, order):
        if Source.objects.filter(order=order, source_type__name=ECPAY_LOGISTIC).exists():
            # ecpay logistic
            isEcpayLogistic = True
        else:
            # other
            isEcpayLogistic = False
            
        source = Source.objects.get(order=order)
        
        return (source, isEcpayLogistic)

    def get(self, request, *args, **kwargs):
        orderNumber = kwargs.get('orderNumber')
        order = get_object_or_404(Order, number=orderNumber)
        source, isEcpayLogistic = self.getSource(order)
        
        if isEcpayLogistic:
            # ecpay logistic
            tradeNo = source.reference.split("|")
            if len(tradeNo) == 2: # tradeNo|tradeNo
                tradeNo = tradeNo[1]
            else:
                tradeNo = tradeNo[0]

            trade = ECPayLogisticTrade.objects.filter(
                        tradeNo=tradeNo).order_by('-id').first()
        else:
            trade = None
            
        info, created = RawHTML.objects.get_or_create(
            name='Logistic Return',
            defaults={
                'body': _('<p>Your refund request has been sent and we will process it as soon as possible.</p>'),
            }
        )
        
        return render(
            request, 
            "shop/payment/Refund.html",
            {
                'order': order,
                'trade': trade,
                'info': info,
            }
        )

    def post(self, request, *args, **kwargs):
        orderNumber = kwargs.get('orderNumber')
        order = get_object_or_404(Order, number=orderNumber)
        source, isEcpayLogistic = self.getSource(order)
        
        if isEcpayLogistic:
            tradeNo = source.reference.split("|")
            if len(tradeNo) == 2: # tradeNo|tradeNo
                tradeNo = tradeNo[1]
            else:
                tradeNo = tradeNo[0]
                
            trade = ECPayLogisticTrade.objects.filter(
                        tradeNo=tradeNo).order_by('-id').first()
           
            data = {
                'MerchantID': trade.merchantId,
                'ServerReplyURL': reverse("ecpayLogisticReply"),
                'GoodsAmount': trade.tradeAmt,
                #'GoodsName': "Reference ",
                'ServiceType': 4, # 退貨不付款
                'SenderName': trade.receiverName,
                'SenderPhone': trade.receiverCellPhone,
                'AllPayLogisticsID': trade.tradeNo,
            }

            apiUrls = {
                "FAMI": {
                    "stage": ("https://logistics-stage.ecpay.com.tw/"
                             "express/ReturnCVS"),
                    "production": ("https://logistics-stage.ecpay.com.tw/"
                                   "express/ReturnCVS"), 
                },
                "UNIMART": {
                    "stage": ("https://logistics-stage.ecpay.com.tw/"
                              "express/ReturnUniMartCVS"),
                    "production": ("https://logistics.ecpay.com.tw/"
                                   "express/ReturnUniMartCVS"), 
                },
                "HILIFE": {
                    "stage": ("https://logistics-stage.ecpay.com.tw/"
                              "express/ReturnHiLifeCVS"),
                    "production": ("https://logistics.ecpay.com.tw/"
                                   "express/ReturnHiLifeCVS"), 
                },
            }

            if settings.ECPAY_SANDBOX_MODE:
                apiUrl = apiUrls[trade.logisticsSubType]["stage"]
            else:
                apiUrl = apiUrls[trade.logisticsSubType]["production"]

            data["CheckMacValue"] = createCheckValue(data)

            res = requests.post(apiUrl, data=data)

            if not "ErrorMessage" in res.text:
                messages.success(self.request, u"成功送出退貨訊息")
                rtnMerchantTradeNo = res.text.split("|")[0]
                rtnOrderNo = res.text.split("|")[1]

                newLogisticTrade = ECPayLogisticTrade.objects.create(
                    tradeNo=trade.tradeNo,
                    tradeAmt=int(trade.tradeAmt),
                    merchantId=trade.merchantId,
                    merchantTradeNo=trade.merchantTradeNo,
                    rtnCode=100,
                    rtnMsg=u"已送出退貨資料",
                    tradeDate=trade.tradeDate,
                    updateStatusDate=timezone.now(),
                    receiverName=trade.receiverName,
                    receiverCellPhone=trade.receiverCellPhone,
                    receiverEmail=trade.receiverEmail,
                    logisticsType=trade.logisticsType,
                    logisticsSubType=trade.logisticsSubType,
                    paymentNo=trade.paymentNo,
                    validationNo=trade.validationNo,

                    rtnMerchantTradeNo=rtnMerchantTradeNo,
                    rtnOrderNo=rtnOrderNo,
                )

                order.status = settings.OSCAR_ORDER_RETURN_STATUS
                order.save()
                return redirect(reverse('refund', kwargs={'orderNumber': order.number}))
            else:
                messages.error(self.request, u"送出退貨訊息失敗")
                return redirect('customer:order-list')
        else:
            order.status = settings.OSCAR_ORDER_RETURN_STATUS
            order.save()
            return redirect(reverse('refund', kwargs={'orderNumber': order.number}))


class ECPayServerReplyView(generic.View):

    http_method_names = ['get', 'post',]

    @csrf_exempt
    def dispatch(self, request, *args, **kwargs):
        return super(ECPayServerReplyView, self).dispatch(
            request, *args, **kwargs)

    def get(self, request, *args, **kwargs):
        logger.warn("GET: %s" % (request.GET))
        return render("|ErrorMessage")

    def post(self, request, *args, **kwargs):
        logger.warn("POST: %s" % (request.POST))
        # data 
        try:
            self.tradeNo = request.POST.get('AllPayLogisticsID')
            self.tradeAmt = request.POST.get('GoodsAmount')

            self.merchantTradeNo = request.POST.get('MerchantTradeNo')

            self.merchantID = request.POST.get('MerchantID')
            self.rtnCode = request.POST.get('RtnCode')
            self.rtnMsg = request.POST.get('RtnMsg')
            
            self.updateStatusDate = request.POST.get('UpdateStatusDate')
            self.receiverName = request.POST.get('ReceiverName')
            self.receiverCellPhone = request.POST.get('ReceiverCellPhone')
            self.receiverEmail = request.POST.get('ReceiverEmail')
            self.logisticsType = request.POST.get('LogisticsType')
            self.logisticsSubType = request.POST.get('LogisticsSubType')

            self.rtnMerchantTradeNo = request.POST.get('RtnMerchantTradeNo')

            self.paymentNo = request.POST.get('CVSPaymentNo')
            self.validationNo = request.POST.get('CVSValidationNo')

            if self.rtnMerchantTradeNo:
                logisticTrade = ECPayLogisticTrade.objects.filter(
                    tradeNo=self.tradeNo).order_by('-id').first()

                newLogisticTrade = ECPayLogisticTrade.objects.create(
                    tradeNo=self.tradeNo,
                    tradeAmt=int(self.tradeAmt),
                    merchantId=self.merchantID,
                    merchantTradeNo=self.merchantTradeNo,
                    rtnCode=self.rtnCode,
                    rtnMsg=self.rtnMsg,
                    tradeDate=logisticTrade.tradeDate,
                    updateStatusDate=self.updateStatusDate.replace('/', '-'),
                    receiverName=self.receiverName,
                    receiverCellPhone=self.receiverCellPhone,
                    receiverEmail=self.receiverEmail,
                    logisticsType=self.logisticsType,
                    logisticsSubType=self.logisticsSubType,
                    paymentNo=self.paymentNo,
                    validationNo=self.validationNo,
                    rtnMerchantTradeNo=self.rtnMerchantTradeNo,
                )
                
                return HttpResponse("1|OK")
            else:
                index = self.merchantTradeNo.find('BASKETID')
                self.basketId = int(self.merchantTradeNo[index + 8:])

                order = Order.objects.get(basket=self.basketId)
                
                site = get_current_site(request)
                siteName = site.name
                
                if self.rtnCode in ['2030', '3024',]:
                    order.status = settings.OSCAR_ORDER_PROCESSING_STATUS
                elif self.rtnCode in ['2063', '3018', '2073',]:
                    order.status = settings.OSCAR_ORDER_ARRIVE_STATUS
                elif self.rtnCode in ['2067', '3022']:
                    order.status = settings.OSCAR_ORDER_COMPLETE_STATUS
                elif self.rtnCode in ['2074', '3020']:
                    order.status = settings.OSCAR_ORDER_RETURN_STATUS
                else:
                    return HttpResponse("|ErrorMessage")
                order.save()
                
                sendMessageToAdmin(
                    _("[%(site)s]Logistic order status change") % {
                        'site': siteName,
                    },
                    _("Logistic order %(orderNumber)s"
                      " has changed to %(status)s") % {
                        'orderNumber': order.number, 
                        'status': order.status,
                    },
                )
                
                hasSettings = hasattr(
                    settings, 'OSCAR_PROCESSING_SEND_EMAIL_STATUS')
                if (hasSettings and 
                    order.status == \
                        settings.OSCAR_PROCESSING_SEND_EMAIL_STATUS):
                    sendOrderEmail(
                        order, 
                        'shop/customer/emails/BeingProcessedSubject.txt', 
                        'shop/customer/emails/BeingProcessedBody.html', 
                        'shop/customer/emails/BeingProcessedBody.txt'
                    )
                hasSettings = hasattr(
                    settings, 'OSCAR_ARRIVE_SEND_EMAIL_STATUS')
                if (hasSettings and 
                    order.status == settings.OSCAR_ARRIVE_SEND_EMAIL_STATUS):
                    sendOrderEmail(
                        order, 
                        'shop/customer/emails/ArriveSubject.txt', 
                        'shop/customer/emails/ArriveBody.html', 
                        'shop/customer/emails/ArriveBody.txt'
                    )

                logisticTrade = ECPayLogisticTrade.objects.filter(
                    tradeNo=self.tradeNo).order_by('-id').first()

                newLogisticTrade = ECPayLogisticTrade.objects.create(
                    tradeNo=self.tradeNo,
                    tradeAmt=int(self.tradeAmt),
                    merchantId=self.merchantID,
                    merchantTradeNo=self.merchantTradeNo,
                    rtnCode=self.rtnCode,
                    rtnMsg=self.rtnMsg,
                    tradeDate=logisticTrade.tradeDate,
                    updateStatusDate=self.updateStatusDate.replace('/', '-'),
                    receiverName=self.receiverName,
                    receiverCellPhone=self.receiverCellPhone,
                    receiverEmail=self.receiverEmail,
                    logisticsType=self.logisticsType,
                    logisticsSubType=self.logisticsSubType,
                    paymentNo=self.paymentNo,
                    validationNo=self.validationNo,
                )

                return HttpResponse("1|OK")
        except Exception as e:
            logger.error(str(e))
            return HttpResponse("|ErrorMessage")
            
class ATMFillView(generic.View):

    def getOrder(self, request, *args, **kwargs):
        orderNumber = kwargs.get('orderNumber')
        order = get_object_or_404(Order, number=orderNumber)
        return order

    def get(self, request, *args, **kwargs):
        order = self.getOrder(request, *args, **kwargs)
        info = getATMInfo()
        return render(
            request, 
            "shop/payment/FillAccount.html",
            {
                'order': order,
                'ATMInfo': info,
            }
        )
        
    def post(self, request, *args, **kwargs):
        order = self.getOrder(request, *args, **kwargs)
        code = request.POST.get('account', None)
        info = getATMInfo()
        
        order.setATMCode(code)
        
        return render(
            request, 
            "shop/payment/FillAccount.html",
            {
                'order': order,
                'ATMInfo': info,
            }
        )
        
import os
import datetime
import hashlib
import textwrap

from io import BytesIO

from django.http import FileResponse

from PIL import Image
from PIL import ImageFont
from PIL import ImageDraw 

MAX_LENGTH = 60
        
class InvoiceView(generic.View):

    def get(self, request, *args, **kwargs):
        orderNumber = kwargs.get('orderNumber')
        order = get_object_or_404(Order, number=orderNumber)
        
        if order.user:
            if not order.user == request.user:
                raise Http404()
        
        # data
        if hasattr(settings, "IUNO_SHOP_INVOICE_LOGO"):
            logo = settings.IUNO_SHOP_INVOICE_LOGO 
        else:
            logo = "Logo.png" 
            
        items = []
        for line in order.basket.all_lines():
            product = line.product
            productName = product.get_title()
            
            items.append((
                line.quantity,
                productName,
                float(line.price_excl_tax) * line.quantity,
            ))
            
        if float(order.shipping_excl_tax) > 0:
            items.append((
                "",
                str(_("Shipping total")),
                float(order.shipping_excl_tax),
            ))
        
        if order.user:
            email = order.user.email
        else:
            email = order.guest_email
        
        if hasattr(settings, "IUNO_SHOP_INVOICE_FROM"):
            fromContent = settings.IUNO_SHOP_INVOICE_FROM
        else:
            fromContent = settings.OSCAR_SHOP_NAME
        orderNumber = order.number
        date = order.date_placed.strftime("%d %b %Y")
        dollarName = order.currency
        total = order.total_excl_tax
        
        if Source.objects.filter(order=order).exists():
            source = Source.objects.get(order=order)
            pay = source.source_type.name
        else:
            pay = str(_("Cash"))

        # https://stackoverflow.com/questions/16008670/how-to-hash-a-string-into-8-digits
        invoice = "IVBF%s" % (int(hashlib.sha1(orderNumber.encode()).hexdigest(), 16) % (10 ** 8))

        try:
            font = ImageFont.truetype("msjh.ttf", 16) # win7 and centos
        except Exception as e:
            try:
                font = ImageFont.truetype("msjh.ttc", 16) # win10
            except Exception as e:
                raise RuntimeError("msjh.ttf or msjh.ttc should be installed.")


        # background
        # 分辨率是150像素/英寸時，A4紙的尺寸的圖像的像素是1240×1754。
        background = Image.new('RGB', (1240, 1754), color=(255, 255, 255))
        draw = ImageDraw.Draw(background)

        # logo
        if os.path.exists(logo):
            logo = Image.open(logo)
            background.paste(logo, (200, 100))

        # background content
        color = (0, 0, 0) # 黑
        draw.text((200, 200), str(_("To:")), color, font=font)
        draw.text((600, 200), str(_("From:")), color, font=font)
        draw.text((800, 100), str(_("Date:")), color, font=font)
        draw.text((800, 130), str(_("Invoice No:")), color, font=font)
        draw.text((800, 160), str(_("Order No:")), color, font=font)

        # order item background
        shape = [
            (200, 400), 
            (1100, 600 + (30 * sum([len(textwrap.wrap(item[1], width=MAX_LENGTH)) for item in items])))
        ]
        color = (230, 230, 230) # 白灰
        draw.rectangle(shape, fill=color) 

        # To
        color = (87, 155, 255) # 水藍
        draw.text((200, 230), email, color, font=font)
        width, height = draw.textsize(email, font=font)
        draw.line((200, 230 + height - 2, 200 + width, 230 + height - 2), fill=color)

        # From
        color = (135, 135, 135) # 灰
        draw.text((600, 230), fromContent, color, font=font)

        # date
        draw.text((950, 100), date, color, font=font)

        # invoice
        draw.text((950, 130), invoice, color, font=font)

        # Order No
        draw.text((950, 160), orderNumber, color, font=font)

        # Qty
        color = (0, 0, 0) # 黑
        draw.text((250, 450), str(_("Qty")), color, font=font)

        # Description
        draw.text((350, 450), str(_("Description")), color, font=font)

        # Amount
        draw.text((950, 450), str(_("Amount")), color, font=font)

        # items
        color = (135, 135, 135) # 灰
        totalHeight = 0
        for item in items:
            description = textwrap.wrap(item[1], width=MAX_LENGTH)
            
            # Qty
            draw.text((250, 500 + totalHeight), str(item[0]), color, font=font)

            # Description
            draw.text((350, 500 + totalHeight), "\n".join(description), color, font=font)

            # Amount
            draw.text((950, 500 + totalHeight), "$%.2f" % item[2], color, font=font)
            
            totalHeight += 30 * len(description)
            

        # Invoice Total: USD $100.00
        color = (0, 0, 0) # 黑
        draw.text(
            (810, 520 + totalHeight), 
            str(_("Invoice Total: %s $%.2f")) % (dollarName, total), 
            color, font=font
        )

        # Paid via PayPal
        color = (135, 135, 135) # 灰
        draw.text(
            (810, 550 + totalHeight), 
            str(_("Paid via %s")) % pay, 
            color, font=font
        )

        # All amounts shown on this invoice are in US dollars.
        draw.text(
            (420, 620 + totalHeight), 
            str(_("All amounts shown on this invoice are in %s.")) % dollarName, 
            color, font=font
        )

        # response
        imagefile = BytesIO() 
        background.save(imagefile, "PDF", resolution=100.0)
        response = HttpResponse(imagefile.getvalue(), content_type='application/pdf')
        response['Content-Disposition'] = ("filename=%s" % ("Invoice%s.pdf" % orderNumber,))
        
        return response
        





class ECPayTransactionListView(generic.ListView):
    model = ECPayTrade
    template_name = 'shop/payment/dashboard/ECPay.html'
    context_object_name = 'transactions'

    def get_queryset(self):
        return super(
            ECPayTransactionListView, self).get_queryset().order_by('-id')

class AllPayTransactionListView(generic.ListView):
    model = AllPayTrade
    template_name = 'shop/payment/dashboard/AllPay.html'
    context_object_name = 'transactions'

    def get_queryset(self):
        return super(
            AllPayTransactionListView, self).get_queryset().order_by('-id')

class SpgatewayTransactionListView(generic.ListView):
    model = SpgatewayTrade
    template_name = 'shop/payment/dashboard/Spgateway.html'
    context_object_name = 'transactions'

    def get_queryset(self):
        return super(
            SpgatewayTransactionListView, self).get_queryset().order_by('-id')

class CathaybkCreditTransactionListView(generic.ListView):
    model = CathaybkCreditTrade
    template_name = 'shop/payment/dashboard/CathayCredit.html'
    context_object_name = 'transactions'

    def get_queryset(self):
        return super(
            CathaybkCreditTransactionListView, self
            ).get_queryset().order_by('-id')

class ECPayLogisticTransactionListView(generic.ListView):
    model = ECPayLogisticTrade
    template_name = 'shop/payment/dashboard/ECPayLogistic.html'
    context_object_name = 'transactions'

    def get_queryset(self):
        return super(
            ECPayLogisticTransactionListView, self
            ).get_queryset().order_by('-id')
    
# https://code.nuwainfo.com/trac/phantasos/ticket/589#ticket
from oscar.apps.dashboard.catalogue import tables
from oscar.apps.dashboard.catalogue import views as catalogueViews
from oscar.core.loading import get_model
from django_tables2 import A, TemplateColumn
from django.utils.translation import ugettext_lazy as _

Product = get_model('catalogue', 'Product')

class ProductTable(tables.ProductTable):
    categories = TemplateColumn(
        verbose_name=_('Categories'),
        accessor=A('categories'),
        template_name='shop/dashboard/catalogue/ProductRowCategory.html',
        orderable=False,
    )

    class Meta(tables.ProductTable.Meta):
        model = Product
        fields = ('upc', 'date_updated')
        sequence = ('title', 'upc', 'image', 'product_class', 
                    'categories', 'variants',
                    'stock_records', '...', 'date_updated', 'actions')
        order_by = '-date_updated'
        
catalogueViews.ProductListView.table_class = ProductTable


        
    


