#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: AttachSettingsTest.py 11172 2018-07-16 09:31:00Z Lavender $
#
# Copyright (c) 2020 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-07-16 18:31:00 +0900 (週一, 16 七月 2018) $
# $Revision: 11172 $

import os
import json
import platform
import zipfile
import time
import sys
import pkg_resources
import subprocess
import psutil
import hashlib

import requests

from django.urls import reverse
from django.conf import settings
from django.utils import timezone
from selenium import webdriver
from selenium.webdriver.chrome.options import Options as ChromeOptions

from Iuno.Version import IUNO_VERSION
from Iuppiter.Logging import createLogger
from Iuno.Util import update
from Iuno import attachSettings, DEVELOPMENT

logger = createLogger(__name__)

TEST_DB = 'db_test.sqlite'

def importSettings(settingsDict, settings):  
    settingsDict.update({k: getattr(settings, k) for k in dir(settings)})
    testSettings = {
        "SERVER_MODE": DEVELOPMENT,
        "DEBUG": True,
        "DATABASES": {
            'default': {
                'ENGINE': 'django.db.backends.sqlite3',
                'NAME': TEST_DB,
            }
        },
        "SITE_ID": 1,
    }
    attachSettings(settingsDict)
    settingsDict.update(testSettings)
   
CHROME_VERSION_PATH = os.path.join("chromedriver.txt")

def downloadChromedriver():
    needDownload = False
  
    if platform.system() in ['Linux', 'Darwin']:
        if not os.path.exists("chromedriver"):
            needDownload = True
    else:
        if not os.path.exists("chromedriver.exe"):
            needDownload = True
            
    chromeVersion = requests.get("https://chromedriver.storage.googleapis.com/LATEST_RELEASE").text
    
    if os.path.exists(CHROME_VERSION_PATH):
        with open(CHROME_VERSION_PATH, "r") as f:
            currentVersion = f.read()
    else:
        currentVersion = ""
        
    if currentVersion == chromeVersion:
        needDownload = False
    else:
        # 只要版本不一樣，就重新下載
        needDownload = True
    
    if needDownload:
        with open(CHROME_VERSION_PATH, "w") as f:
            f.write(chromeVersion)
            
        if platform.system() == 'Linux':
            driverZip = requests.get(
                "https://chromedriver.storage.googleapis.com/%s/chromedriver_linux64.zip" % chromeVersion).content
        elif platform.system() == 'Darwin':
            driverZip = requests.get(
                "https://chromedriver.storage.googleapis.com/%s/chromedriver_mac64.zip" % chromeVersion).content
        else:
            driverZip = requests.get(
                "https://chromedriver.storage.googleapis.com/%s/chromedriver_win32.zip" % chromeVersion).content
            
        zipPath = os.path.join("chromedriver.zip")
                   
        with open(zipPath, "wb") as f:
            f.write(driverZip)
            
        with zipfile.ZipFile(zipPath, 'r') as zip:
            zip.extractall(".")
            
        if platform.system() in ['Linux', 'Darwin']:
            os.system("chmod +x chromedriver")
            
        os.remove("chromedriver.zip")

class RemoteTestCaseMixin(object):

    TEST_URL = None
    TEST_SETTINGS = None
    
    def runTest(self):
        try:
            remoteURL = reverse("remoteTest")
        except Exception as e:
            # 如 OscarDemoTest 所 run 的環境沒有 urls.py，因此不能用 reverse
            remoteURL = "/remote/test/"
        result = requests.post(
            "%s%s" % (self.TEST_URL, remoteURL), data={"password": "25025529", "settings": self.TEST_SETTINGS})
        allData = json.loads(result.text)
        version = allData.get('IUNO_VERSION', "0.0.0.0")
        localAppVersion = getattr(settings, "APP_VERSION", None)
        remoteAppVersion = allData.get("APP_VERSION", "0.0.0.0")
        if localAppVersion:
            if [int(i) for i in localAppVersion.split(".")] > [int(i) for i in remoteAppVersion.split(".")]:
                logger.warning(
                    "Local app version is %s, but remote app version is %s." % (localAppVersion, remoteAppVersion))
        else:
            if [int(i) for i in IUNO_VERSION.split(".")] > [int(i) for i in version.split(".")]:
                logger.warning("Iuno version is %s, but latest Iuno version is %s." % (IUNO_VERSION, version))
        logger.info(allData.get('result', None))
        
        errorCount = 0
        
        for testcase in allData.get('track', []):
            if testcase.get('error', []):
                classname = testcase.get('classname', None)
                name = testcase.get('name', None)
                time = testcase.get('time', None)
                msg = '''
======================================================================
ERROR [%ss]: %s (%s)
----------------------------------------------------------------------
                ''' % (time, name, classname)
                logger.info(msg)
                for error in testcase.get('error', []):
                    logger.info("Message: %s" % error.get("message", None))
                    logger.info("Type: %s" % error.get("type", None))
                    logger.info(error.get("traceback", None))
                    errorCount += 1
        
        # has error
        self.assertEqual(0, errorCount)
        
SCHEMA_FIX = '''
    def __enter__(self):
        # Some SQLite schema alterations need foreign key constraints to be
        # disabled. Enforce it here for the duration of the transaction.
        self.connection.disable_constraint_checking()
        self.connection.cursor().execute('PRAGMA legacy_alter_table = ON')
        return super().__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super().__exit__(exc_type, exc_value, traceback)
        self.connection.cursor().execute('PRAGMA legacy_alter_table = OFF')
        self.connection.enable_constraint_checking()
'''

SCHEMA_FIX_MD5_1_11 = "08fc81f9012c4a7f94782edc030885e7"
SCHEMA_ORIGIN_MD5_DJANGO_1_11 = "c27ea4f5a250178b0a417b719c40c3a8"
SCHEMA_ORIGIN_DJANGO_1_11 = '''
    def __enter__(self):
        with self.connection.cursor() as c:
            # Some SQLite schema alterations need foreign key constraints to be
            # disabled. This is the default in SQLite but can be changed with a
            # build flag and might change in future, so can't be relied upon.
            # We enforce it here for the duration of the transaction.
            c.execute('PRAGMA foreign_keys')
            self._initial_pragma_fk = c.fetchone()[0]
            c.execute('PRAGMA foreign_keys = 0')
        return super(DatabaseSchemaEditor, self).__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super(DatabaseSchemaEditor, self).__exit__(exc_type, exc_value, traceback)
        with self.connection.cursor() as c:
            # Restore initial FK setting - PRAGMA values can't be parametrized
            c.execute('PRAGMA foreign_keys = %s' % int(self._initial_pragma_fk))
'''

SCHEMA_FIX_MD5_2_1 = "de8e3a3770fdb73b3f4478e0410ceaf9"
SCHEMA_ORIGIN_MD5_DJANGO_2_1 = "280dba9301b0a827962786c34456ca54"
SCHEMA_ORIGIN_DJANGO_2_1 = '''
    def __enter__(self):
        # Some SQLite schema alterations need foreign key constraints to be
        # disabled. Enforce it here for the duration of the schema edition.
        if not self.connection.disable_constraint_checking():
            raise NotSupportedError(
                'SQLite schema editor cannot be used while foreign key '
                'constraint checks are enabled. Make sure to disable them '
                'before entering a transaction.atomic() context because '
                'SQLite3 does not support disabling them in the middle of '
                'a multi-statement transaction.'
            )
        self.connection.cursor().execute('PRAGMA legacy_alter_table = ON')
        return super().__enter__()

    def __exit__(self, exc_type, exc_value, traceback):
        super().__exit__(exc_type, exc_value, traceback)
        self.connection.cursor().execute('PRAGMA legacy_alter_table = OFF')
        self.connection.enable_constraint_checking()
'''

def fixSchema(schemaPath):
    import django 
    try:
        with open(schemaPath, 'r') as f:
            content = f.read()
            
        if not django.VERSION >= (2, 0):
            SCHEMA_FIX_MD5 = SCHEMA_FIX_MD5_1_11
            SCHEMA_ORIGIN = SCHEMA_ORIGIN_DJANGO_1_11 
            SCHEMA_ORIGIN_MD5 = SCHEMA_ORIGIN_MD5_DJANGO_1_11
        else:
            SCHEMA_FIX_MD5 = SCHEMA_FIX_MD5_2_1
            SCHEMA_ORIGIN = SCHEMA_ORIGIN_DJANGO_2_1
            SCHEMA_ORIGIN_MD5 = SCHEMA_ORIGIN_MD5_DJANGO_2_1
        
        if hashlib.md5(content.encode()).hexdigest() == SCHEMA_ORIGIN_MD5:
            content = content.replace(SCHEMA_ORIGIN, SCHEMA_FIX)
              
        with open(schemaPath, 'w') as f:
            f.write(content)
            
        with open(schemaPath, 'r') as f:
            content = f.read()
    except Exception as e:
        pass
        
def revertSchema(schemaPath):
    import django 
    try:
        with open(schemaPath, 'r') as f:
            content = f.read()
            
        if not django.VERSION >= (2, 0):
            SCHEMA_FIX_MD5 = SCHEMA_FIX_MD5_1_11
            SCHEMA_ORIGIN = SCHEMA_ORIGIN_DJANGO_1_11 
            SCHEMA_ORIGIN_MD5 = SCHEMA_ORIGIN_MD5_DJANGO_1_11
        else:
            SCHEMA_FIX_MD5 = SCHEMA_FIX_MD5_2_1
            SCHEMA_ORIGIN = SCHEMA_ORIGIN_DJANGO_2_1
            SCHEMA_ORIGIN_MD5 = SCHEMA_ORIGIN_MD5_DJANGO_2_1
            
        if hashlib.md5(content.encode()).hexdigest() == SCHEMA_FIX_MD5:
            content = content.replace(SCHEMA_FIX, SCHEMA_ORIGIN)
            
        with open(schemaPath, 'w') as f:
            f.write(content)
    except Exception as e:
        pass

def kill(pid):
    p = psutil.Process(pid)
    for cp in p.children(recursive=True):
        cp.kill()
    p.kill()

        
class SeleniumMixin(object):

    INITIAL_DATA = None
    TEST_SETTINGS = None
    HEADLESS = True
    PORT = "9000"
    live_server_url = "http://127.0.0.1:%s" % PORT
    CMS_LOADDATA = False
    LOADDATA_WITH_PARAMETER = ''

    @staticmethod
    def _setUpClass(cls):
        try:
            cls.schemaPath = pkg_resources.resource_filename(
                'django.db.backends.sqlite3', 'schema.py')
        except Exception as e:
            cls.schemaPath = None
        if cls.schemaPath:
            fixSchema(cls.schemaPath)
        cls.pid = cls._runServer(cls)
        
        # Chrome
        downloadChromedriver()
        options = ChromeOptions()
        if cls.HEADLESS:
            options.add_argument("--headless")
        options.add_argument('--disable-gpu')
        options.add_argument('--no-sandbox')
        
        if platform.system() == 'Linux' or platform.system() == 'Darwin':
            cls.browser = webdriver.Chrome(executable_path="./chromedriver", chrome_options=options)
        else:
            cls.browser = webdriver.Chrome(executable_path="./chromedriver.exe", chrome_options=options)
        
        cls.browser.implicitly_wait(20)

    @staticmethod
    def _tearDownClass(cls):
        if cls.schemaPath:
            revertSchema(cls.schemaPath)
        cls.browser.quit()
        kill(cls.pid)
      
    # 因為 TestCase 不會寄信，因此改用自己 runServer
    # https://docs.djangoproject.com/en/1.8/topics/testing/tools/#email-services
    @staticmethod
    def _runServer(cls):
        env = os.environ.copy()
        envPython = sys.executable
        
        if os.path.exists(TEST_DB):
            os.remove(TEST_DB)
        
        os.system('"%s" manage.py migrate --settings %s' % (envPython, cls.TEST_SETTINGS))


        if cls.CMS_LOADDATA:
            os.system('"%s" manage.py cms_loaddata %s %s --settings %s' % (envPython, cls.INITIAL_DATA, cls.LOADDATA_WITH_PARAMETER, cls.TEST_SETTINGS))
        else:
            os.system('"%s" manage.py loaddata %s %s --settings %s' % (envPython, cls.INITIAL_DATA, cls.LOADDATA_WITH_PARAMETER, cls.TEST_SETTINGS))

       
        if cls.TEST_SETTINGS:
            p = subprocess.Popen(
                ('"%s" manage.py runserver %s --noreload --settings %s') % (
                    envPython, cls.PORT, cls.TEST_SETTINGS), 
                env=env, shell=True, cwd=".")
        else:
            p = subprocess.Popen(
                ('"%s" manage.py runserver %s --noreload') % (
                    envPython, cls.PORT), 
                env=env, shell=True, cwd=".")
       
        time.sleep(20)
       
        i = 0
        while i < 30: # 每 10 秒確認一次，試 5 次
            try:
                response = requests.get("http://127.0.0.1:%s/admin/login/" % cls.PORT)
                if response.status_code == 200:
                    return p.pid
            except Exception as e:
                print(str(e))
            finally:
                time.sleep(10)
                i += 1
           
        raise RuntimeError("Run server time out.")
    