#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: build_boilerplate.py 9978 2017-05-20 15:24:37Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2017-05-20 23:24:37 +0800 (週六, 20 五月 2017) $
# $Revision: 9978 $

import os
import sys
import re
import logging
import hashlib
import json

from lxml import etree
from lxml.html.soupparser import parse
from django.core.management.base import BaseCommand, CommandError
from django.template.loader import get_template

from Zephyrus.boilerplate.management.commands import (
    TEMPLATES_DIR_PATH, STATIC_DIR_PATH)

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
handler = logging.StreamHandler()
handler.setFormatter(
    logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
logger.addHandler(handler)

# inside
TEXT_TAGS = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'button',]
VISUAL_TAGS = ['em', 'strong', 'b', 'small',]

ALL_INSIDE_TAGS = TEXT_TAGS + VISUAL_TAGS

# outside
SIBLING_TAGS = ['img',]
ADD_PLACEHOLDER_TO_OUTSIDE_TAGS = ['ul', 'a', 'table', 'ol',] 

ALL_OUTSIDE_TAGS = SIBLING_TAGS + ADD_PLACEHOLDER_TO_OUTSIDE_TAGS

# special
BLOCK_TAGS = ['div', 'span',]

# comment
COMMENT = "The template was modified by Nuwainfo.com"
LEGEL = "\n%s\n" % get_template('boilerplate/txt/Legel.txt').render()

# cms
CMS_HEAD_FRONT_TAG = "{% load cms_tags menu_tags sekizai_tags %}" + \
                     "{% load staticfiles %}"
CMS_HEAD_BACK_TAG = "{% render_block 'css' %}"

CMS_BODY_FRONT_TAG = "{% cms_toolbar %}"
CMS_BODY_BACK_TAG = "{% render_block 'js' %}"

CMS_PAGE_ATTRS_COMMENT = '''
<!--
CMS page attribute
Slug: {% page_attribute "slug" %}
Changed_date: {% page_attribute "changed_date" %}
Changed_by: {% page_attribute "changed_by" %}
-->
'''

CMS_PAGE_ATTRS_COMMENT_MD5 = hashlib.md5(CMS_PAGE_ATTRS_COMMENT).hexdigest()

class Command(BaseCommand):

    help = 'Add placeholder to templates'
    args = '<destination folder>'

    def handle(self, *args, **options):
        if len(args) < 1:
            path = TEMPLATES_DIR_PATH
        else:
            path = args[0]

        logger.info('Processing...')

        staticDict = self.buildFileDict('static')
        staticDict['js/FixTemplate.js'] = \
            hashlib.md5('js/FixTemplate.js').hexdigest()
        templateDict = self.buildFileDict('templates')
        fileDict = dict(staticDict)
        fileDict.update(templateDict)
       
        for root, dirs, files in os.walk(path):
            for file in files:
                name = file
                file = os.path.join(root, file)
                logger.info(file)

                # start preprocess
                commentDict = self.findComment(file)
                commentDict[CMS_PAGE_ATTRS_COMMENT_MD5] = CMS_PAGE_ATTRS_COMMENT
                # end preprocess

                # start process lxml etree
                with open(file) as f:
                    if not f.name.endswith(".html"):
                        continue

                    page = parse(f)

                    tags = ALL_INSIDE_TAGS + ALL_OUTSIDE_TAGS + BLOCK_TAGS
                    tags = self.clearTags(tags)

                    for tag in tags:
                        self.buildPlaceholder(page, tag, name)
                    
                    pageData = self.addCMSTags(page)
                    commentDict[pageData[0]] = pageData[1]

                    self.replaceKeyword(page, fileDict)
                    idDict = self.buildImgIdDict(page, name)

                    page.write(file, method='html')
                # end process lxml etree

                # start endprocess
                self.addLegelComment(file)
                self.fixStaticTemplate(file, fileDict)
                self.fixComment(file, commentDict)
                self.fixImgComment(file, idDict)
                # end endprocess

                logger.info(file + " is OK!")

        logger.info('ALL OK!')

    #---------------------------------Main--------------------------------------

    def clearTags(self, tagList):
        tags = []
        for tag in tagList:
            if tag in tags:
                continue
            tags.append(tag)
        return tags

    def buildPlaceholder(self, page, tag, name):
        index = name.index('.html')
        name = name[0:index]
        for i, ele in  enumerate(page.iter(tag=tag), 1):
            id = "%s-%s-%d" % (name, ele.tag, i)

            if 'id' in ele.items():
                id = ele.items()['id']

            if tag in ALL_INSIDE_TAGS:
                self.addInsidePlaceholder(id, ele)
            elif tag in ALL_OUTSIDE_TAGS:
                self.addOutsidePlaceholder(id, ele)
            elif tag in BLOCK_TAGS:
                self.addDivPlaceholder(id, ele)
            else:
                pass

    def buildFileDict(self, path):
        fileDict = {}
        
        if path == 'static':
            for root, dirs, files in os.walk(STATIC_DIR_PATH):
                for f in files:
                    index = len(STATIC_DIR_PATH) + 1
                    filePath = os.path.join(root, f)[index:].replace('\\', '/')
                    fileDict[filePath] = hashlib.md5(filePath).hexdigest()
        elif path == 'templates':
            for root, dirs, files in os.walk(TEMPLATES_DIR_PATH):
                for f in files:
                    index = len(TEMPLATES_DIR_PATH) + 1
                    filePath = os.path.join(root, f)[index:].replace('\\', '/')
                    fileDict[filePath] = hashlib.md5(filePath).hexdigest()
        else:
            for root, dirs, files in os.walk(path):
                for f in files:
                    index = len(path) + 1
                    filePath = os.path.join(root, f)[index:].replace('\\', '/')
                    fileDict[filePath] = hashlib.md5(filePath).hexdigest()
        """
        for root, dirs, files in os.walk(path):
            for f in files:
                # [7:] 是用來去掉 static/
                # [10:] 是用來去掉 templates/
                if path == 'static':
                    filePath = os.path.join(root, f)[7:].replace('\\', '/')
                elif path == 'templates':
                    filePath = os.path.join(root, f)[10:].replace('\\', '/')
                else:
                    filePath = os.path.join(root, f).replace('\\', '/')
                fileDict[filePath] = hashlib.md5(filePath).hexdigest()
        """
        return fileDict



    #----------------------------lxml etree ele---------------------------------

    def addInsidePlaceholder(self, id, ele):
        if ele.text:
            if "{% placeholder " in ele.text:
                return
        else:
            ele.text = ''

        if ele.getchildren():
            if ele[-1].tail is None:
                ele[-1].tail = ''
            ele[-1].tail = ele[-1].tail + "{% endplaceholder %}"
            ele.text = "{% placeholder " + "'%s' " % id + "or %}\n " + ele.text
        else:
            ele.text = "{% placeholder " + "'%s' " % id + "or %}\n " + \
                        ele.text + "{% endplaceholder %}"

    def addOutsidePlaceholder(self, id, ele):
        if ele.tail:
            if ele.tail == "{% endplaceholder %}":
                return
            ele.tail = "{% endplaceholder %}" + ele.tail
        else:
            ele.tail = "{% endplaceholder %}"

        previous = ele.getprevious()
        parent = ele.getparent()

        comment = hashlib.md5(id).hexdigest()
        if previous is None:
            if parent.text is None:
                parent.text = ''
            if ele.tag == 'img':
                parent.text = comment + \
                    parent.text + "{% placeholder " + "'%s' " % id + "or %}"
            else:
                parent.text =  \
                    parent.text + "{% placeholder " + "'%s' " % id + "or %}"
        else:
            if previous.tail is None:
                previous.tail = ''
            if ele.tag == 'img':
                previous.tail = previous.tail + comment + \
                                "{% placeholder " + "'%s' " % id + "or %}"
            else:
                previous.tail = previous.tail + \
                                "{% placeholder " + "'%s' " % id + "or %}"

    def addDivPlaceholder(self, id, ele):
        context = etree.tostring(ele, pretty_print=True)

        if not context.startswith("<div>{% placeholder "):
            if not ele.getchildren():
                if ele.text is not None:
                    
                    if ele.text.strip() is '':
                        return

                    self.addInsidePlaceholder(id, ele)
                else:
                    ele.text = ''

    def addCMSTags(self, page):
        for tag in page.iter(tag='head'):
            head = tag
        for tag in page.iter(tag='body'):
            body = tag
        
        titleContent = None
        createTitle = True
        for tag in page.iter(tag='title'):
            titleContent = tag.text
            tag.text = "{% page_attribute 'page_title' %}"
            createTitle = False

        description = None
        createMetaDescription = True
        for tag in page.iter(tag='meta'):
            for name, value in tag.items():
                if name == 'name' and value == 'description':
                    description = tag.get('content')
                    tag.set(
                        "content", "{% page_attribute 'meta_description' %}")
                    createMetaDescription = False
                    
        data = {
            "title": titleContent,
            "description": description,
        }
        cmsPageData = ("<!--CMS_PAGE_DATA%sEND_CMS_PAGE_DATA-->" % 
                       json.dumps(data))
        dataMd5 = hashlib.md5(cmsPageData).hexdigest()

        comment = etree.Element('meta')
        comment.set("name", "comment")
        comment.set("content", COMMENT)

        jq = etree.Element('script')
        jq.set("src", 
            "https://ajax.googleapis.com/ajax/libs/jquery/3.1.1/jquery.min.js")

        addJ = etree.Element('script')
        addJ.set('src', 'js/FixTemplate.js')

        if head.getchildren():
            head.insert(0, addJ)
            head.insert(0, jq)
            head.insert(0, comment)
            if createMetaDescription:
                metaDescription = etree.Element('meta')
                metaDescription.set("name", "description")
                metaDescription.set(
                    "content", "{% page_attribute 'meta_description' %}")
                head.insert(0, metaDescription)
            if createTitle:
                title = etree.Element('title')
                title.text = "{% page_attribute 'page_title' %}"
                head.insert(0, title)
        else:
            if createTitle:
                title = etree.Element('title')
                title.text = "{% page_attribute 'page_title' %}"
                head.append(title)
            if createMetaDescription:
                metaDescription = etree.Element('meta')
                metaDescription.set("name", "description")
                metaDescription.set(
                    "content", "{% page_attribute 'meta_description' %}")
                head.append(metaDescription)
            head.append(comment)
            head.append(jq)
            head.append(addJ)
            
        head[-1].tail = CMS_HEAD_BACK_TAG

        if head.text is None:
            head.text = ''
        head.text = CMS_HEAD_FRONT_TAG + \
                    CMS_PAGE_ATTRS_COMMENT_MD5 + "\n" + \
                    dataMd5 + \
                    head.text

        if body.getchildren():
            if body[-1].tail is None:
                body[-1].tail = ''
            body[-1].tail = body[-1].tail + CMS_BODY_BACK_TAG
            body.text = CMS_BODY_FRONT_TAG + body.text
        else:
            body.text = CMS_BODY_FRONT_TAG + body.text + \
                        CMS_BODY_BACK_TAG
        
        return (dataMd5, cmsPageData)

    def replaceKeyword(self, page, fileDict):
        for tag in page.iter():
            if tag.tag == 'script' or tag.tag == 'style':
                if tag.text:  
                    for path in fileDict:
                        result = os.path.splitext(path)[-1]
                        if not "." in result:
                            continue

                        if ("./" + path) in tag.text:
                            tag.text = tag.text.replace(
                                ("./" + path), fileDict[path])
                        if path in tag.text:
                            tag.text = tag.text.replace(path, fileDict[path])

            for name, value in tag.items():
                for path in fileDict:
                    value = value.replace('\\', '/')
                    # 考慮到有時 attr 的 value 可能不只 static 路徑，
                    # 像是attr="return func('static')"等等，所以才用現在的方式

                    # 因為上述原因，會導致有些會有以下狀況
                    # "./images/g2.jpg"" 會變 "./{% static 'images/g2.jpg' %}"
                    # 所有暫且先這樣
                    if ("./" + path) in value or value == ("./" + path):
                        newValue = value.replace(("./" + path), fileDict[path])
                        tag.set(name, newValue)
                        value = newValue
                    if path in value:
                        if fileDict[path] in value:
                            continue
                        newValue = value.replace(path, fileDict[path])
                        tag.set(name, newValue)

    def buildImgIdDict(self, page, name):
        idDict = {}

        index = name.index('.html')
        name = name[0:index]
        for i, ele in  enumerate(page.iter(tag='img'), 1):
            id = "%s-%s-%d" % (name, ele.tag, i)

            if 'id' in ele.items():
                id = ele.items()['id']

            idDict[hashlib.md5(id).hexdigest()] = id
        return idDict


    #--------------------------------file---------------------------------------
    def addLegelComment(self, filePath):
        f = open(filePath, "r")
        contents = f.readlines()
        f.close()

        contents.insert(1, "<!--" + LEGEL + "-->\n")

        f = open(filePath, "w")
        f.writelines(contents)
        f.close()

    def findComment(self, filePath):
        commentDict = {}
        with open(filePath, 'r') as f:
            content = f.read()
            commentList = re.findall(r"\<!--((.|\n)*?)--\>", content)
            
            for comment in commentList:
                code = hashlib.md5(comment[0]).hexdigest()
                content = content.replace(
                    "<!--" + comment[0] + "-->", "<!--" + code + "-->")
                commentDict[code] = comment[0]
            
        with open(filePath, 'w') as f:
            f.write(content)
        return commentDict

    def fixComment(self, filePath, commentDict):
        with open(filePath, 'r') as f:
            content = f.read()
            for key in commentDict:
                content = content.replace(key, commentDict[key])
        with open(filePath, 'w') as f:
            f.write(content)

    def fixImgComment(self, filePath, idDict):
        with open(filePath, 'r') as f:
            content = f.read()
            for key in idDict:
                comment = "<!--CMS:" + idDict[key] + "-->"
                content = content.replace(key, comment)
        with open(filePath, 'w') as f:
            f.write(content)

    def fixStaticTemplate(self, filePath, fileDict):
        with open(filePath, 'r') as f:
            content = f.read()

            for path in fileDict:
                if '.html' in path:
                    if path == 'index.html' or path == 'Index.html':
                        content = content.replace(fileDict[path], '/')
                    index = path.index(".html")
                    href = path[0:index]
                    templatePath = "/" + href
                    content = content.replace(fileDict[path], templatePath)
                else:
                    staticPath = "{% static '" + path + "' %}"
                    content = content.replace(fileDict[path], staticPath)

        with open(filePath, 'w') as f:
            f.write(content)
