#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime

from django.db import models
from django.db.models import F, Q
from django.db import transaction
from django.core import exceptions
from django.contrib.auth import get_user_model
from django.conf import settings
from django.utils import timezone
from django.core.validators import MaxValueValidator, MinValueValidator
from django.template import TemplateDoesNotExist, engines
from django.utils.translation import gettext_lazy as _
from solo.models import SingletonModel
from django.template.loader import get_template
from oscar.core.loading import get_model

User = get_user_model()
Source = get_model('payment', 'Source')
UserAddress = get_model('address', 'UserAddress')
Order = get_model('order', 'Order')
Basket = get_model('basket', 'Basket')
ShippingAddress = get_model('order', 'ShippingAddress')

ATM = 'ATM'

TEXT = "text"
INTEGER = "integer"
BOOLEAN = "boolean"
FLOAT = "float"
RICHTEXT = "richtext"
DATE = "date"
DATETIME = "datetime"
OPTION = "option"
MULTI_OPTION = "multi_option"
ENTITY = "entity"
FILE = "file"
IMAGE = "image"
TYPE_CHOICES = (
    (TEXT, _("Text")),
    (INTEGER, _("Integer")),
    (BOOLEAN, _("True / False")),
    (FLOAT, _("Float")),
    (RICHTEXT, _("Rich Text")),
    (DATE, _("Date")),
    (DATETIME, _("Datetime")),
    (OPTION, _("Option")),
    #(MULTI_OPTION, _("Multi Option")),
    #(ENTITY, _("Entity")),
    (FILE, _("File")),
    (IMAGE, _("Image")),
)

class OptionType(models.Model):
    option = models.ForeignKey(
        'catalogue.Option',
        on_delete=models.CASCADE,
        verbose_name=_("Option")
    )
    
    type = models.CharField(
        choices=TYPE_CHOICES, default=TYPE_CHOICES[0][0],
        max_length=20, verbose_name=_("Type"))
    optionGroup = models.ForeignKey(
        'catalogue.AttributeOptionGroup',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        related_name='productAttributes',
        verbose_name=_("Option Group"),
        help_text=_(
            'Select an option group if using type "Option" or "Multi Option"'))
            
class OrdererAddress(models.Model):
    user = models.ForeignKey(
        User,
        related_name='orderer',
        verbose_name=_("Orderer"),
        on_delete=models.CASCADE,
    )
    
    address = models.ForeignKey(
        'address.UserAddress',
        on_delete=models.CASCADE,
        related_name='ordererAddress',
        verbose_name=_("Orderer Address"),
    )
    
@property
def ordererAddress(self):
    if OrdererAddress.objects.filter(user=self).exists():
        return OrdererAddress.objects.get(user=self).address
    else:
        return None
        
User.ordererAddress = ordererAddress
    
class OrderOrdererAddress(models.Model):
    order = models.ForeignKey(
        Order,
        related_name='order',
        verbose_name=_("Order"),
        on_delete=models.CASCADE,
    )
    
    address = models.ForeignKey(
        ShippingAddress,
        on_delete=models.CASCADE,
        related_name='ordererAddress',
        verbose_name=_("Orderer Address"),
    )
 
@property
def ordererAddress2(self):
    if OrderOrdererAddress.objects.filter(order=self).exists():
        return OrderOrdererAddress.objects.get(order=self).address
    else:
        return None

Order.ordererAddress = ordererAddress2

# https://code.nuwainfo.com/trac/Mercurius/ticket/3296#ticket
@property
def useATM(self):
    
    if Source.objects.filter(order=self, source_type__name=ATM).exists():
        source = Source.objects.get(order=self, source_type__name=ATM)
        return source
    else:
        return None
        
@property
def useATMPaymentSuccess(self):
    source = self.useATM
    if source:
        if not source.reference == '-':
            return True
        else:
            return False
    else:
        return False
        
@property
def useATMPaymentExpired(self):
    expiredHours = settings.IUNO_SHOP_ATM_EXPIRED_HOURS
    expiredHours = datetime.timedelta(hours=expiredHours)
    now = timezone.localtime(timezone.now())
    
    if now - self.date_placed > expiredHours:
        return True
    return False
        
def setATMCode(self, code):
    source = self.useATM
    if source:
        source.reference = code
        source.save()

Order.useATM = useATM
Order.useATMPaymentSuccess = useATMPaymentSuccess
Order.useATMPaymentExpired = useATMPaymentExpired
Order.setATMCode = setATMCode    
    
@property
def isOrdererAddress(self):
    orderAddress, created = OrdererAddress.objects.get_or_create(
        user=self.user,
        defaults={
            'address': self,
        }
    )
    if orderAddress.address == self:
        return True
    else:
        return False
        
UserAddress.isOrdererAddress = isOrdererAddress

class Bonus(models.Model):
    user = models.ForeignKey(
        User,
        verbose_name=_("User"),
        on_delete=models.CASCADE,
    )
    
    quantity = models.PositiveIntegerField(
        verbose_name=_("Quantity"), default=0)
        
@property
def bonus(self):
    b, created = Bonus.objects.get_or_create(user=self)
    return b.quantity
    
User.bonus = bonus

class BonusRecordManager(models.Manager):
    @transaction.atomic
    def recordUsingBonus(self, record, adjust=False):
        bonusRecords = BonusRecord.objects.filter(
            user=record.user.id, createdTime__lt=record.createdTime
            ).exclude(id=record.id).order_by('createdTime') # old to new
            
        use = record.use
        
        if record.adjust < 0: # 調整紅利為負，表示扣除
            use = use - record.adjust
        
        for r in bonusRecords:
            if use == 0: # 已記錄完畢
                break
            else:
                canUsingBonus = r.canUsingBonus
                if canUsingBonus > 0:
                    if canUsingBonus >= use:
                        r.usedBonus = r.usedBonus + use
                        use = use - use # 0
                    else:
                        use = use - canUsingBonus
                        r.usedBonus = r.usedBonus + canUsingBonus # canUsingBonus will be 0
                        
                    r.save()
                
        if use != 0:
            if adjust:
                # 用於紀錄對不上時(如使用 admin 自行修改 bonus 而沒有被 record 處理過的)，這部分用於 migrate 時，
                # 調整當筆紀錄之前的紀錄並扣除已使用的 bonus
                # 屬於紀錄補正用
                r = BonusRecord.objects.create(
                    user_id=int(record.user.id),
                    adjust=use,
                    usedBonus=use,
                    # note=_("System adjust") 如果先 migrate 19 版還沒有 note，會出問題，因此暫不自動設定
                )
                r.createdTime = record.createdTime
                r.save()
            else:
                raise RuntimeError("Not enough bonus to deduct: %s" % record)
        
    
    @transaction.atomic
    def createBonusRecord(self, user, order, use=0, receive=0, adjust=0, note=None):
        if not use == 0 or not receive == 0:
            record = BonusRecord.objects.create(
                user=user,
                order=order,
                use=use,
                receive=receive,
                adjust=adjust,
                note=note,
            )
            
            # recordUsingBonus
            self.recordUsingBonus(record)
            return record
   
class BonusRecord(models.Model):
    user = models.ForeignKey(
        User,
        verbose_name=_("User"),
        on_delete=models.CASCADE,
    )
    order = models.ForeignKey(
        Order,
        verbose_name=_("Order"),
        null=True,
        on_delete=models.CASCADE,
    )
    
    
    use = models.PositiveIntegerField(
        verbose_name=_("Use"), default=0)
    receive = models.PositiveIntegerField(
        verbose_name=_("Receive"), default=0)
    adjust = models.IntegerField(
        verbose_name=_("Adjust"), default=0)
        
    usedBonus = models.IntegerField(
        verbose_name=_("Used bonus"), default=0)
        
    expiredBonus = models.IntegerField(
        verbose_name=_("Expired bonus"), default=0)
        
    createdTime = models.DateTimeField(_("Created Time"), auto_now_add=True)
    
    expiredTime = models.DateTimeField(_("Expired Time"), null=True)
    expireTimeSetting = models.DateTimeField(_("Expired Time Setting"), null=True)
    
    note = models.TextField(_("Note"), null=True)
    
    objects = BonusRecordManager()
    
    @property
    def canUsingBonus(self):
        return self.receive + self.adjust - self.usedBonus - self.expiredBonus
    
    @property
    def systemExpireTime(self):
        config = BonusSettings.objects.get()
        return self.createdTime + datetime.timedelta(days=config.expiryDays)
            
    @property
    def expectedExpireTime(self):
        if not self.expiredTime:
            config = BonusSettings.objects.get()
            if config.expiryDays >= 0:
                if not self.expireTimeSetting:
                    # 動態產生
                    return self.createdTime + datetime.timedelta(days=config.expiryDays)
                else:
                    # 固定時間
                    return self.expireTimeSetting
            else:
                # 沒有設定
                return None
        else:
            # 已經過期
            return self.expiredTime
            
    def updateExpiredBonus(self):
        if self.expectedExpireTime:
            # 有過期設定，還沒更新過期
            config = BonusSettings.objects.get()
            b = Bonus.objects.get(user=self.user)
            now = timezone.localtime(timezone.now())
            if config.expiryDays >= 0:
                if self.expectedExpireTime <= now and not self.expiredTime: # 還沒紀錄過過期的紀錄
                    canUsingBonus = self.canUsingBonus
                    
                    self.expiredBonus = canUsingBonus
                    self.expiredTime = self.expectedExpireTime
                    self.save()
                    
                    if b.quantity - self.expiredBonus >= 0:
                        b.quantity = b.quantity - self.expiredBonus
                    else:
                        # 過期點數沒得扣除，因此設為 0，原因出自於 database 
                        # 自行更改 bonus 值或者在此bonus過期機制版本
                        # 更新前有 bug 時修改過 bonus 所造成的
                        b.quantity = 0
                    b.save()           
    
    def __str__(self):
        return "<BonusRecord: user: %s order: %s createdTime: %s>" % (self.user, self.order, self.createdTime)
        
    
@property
def hasReceiveBonus(self):
    return BonusRecord.objects.filter(order=self).exists()
    
def calcReceiveBouns(total, shipping):
    config = BonusSettings.objects.get()
    
    if shipping:
        consumption = (
            int(total) - config.consumption - 
            int(shipping))
    else:
        consumption = (
            int(total) - config.consumption)
            
    receive = 0
    if consumption >= 0:
        # config.percent / 100 = X %
        receive = int(consumption * (config.percent / 100))
        
    return receive
    
@property
def canReceiveBonus(self):
    return calcReceiveBouns(self.total_excl_tax, self.shipping_excl_tax)
    
Order.hasReceiveBonus = hasReceiveBonus
Order.canReceiveBonus = canReceiveBonus
    
@property
def bonusRecords(self):
    return BonusRecord.objects.filter(user=self).order_by('-createdTime')
    
User.bonusRecords = bonusRecords    
    
class BonusSettings(SingletonModel):
    consumption = models.PositiveIntegerField(
        verbose_name=_("Consumption exceed(0 = no limit)"), default=0)
    percent = models.FloatField(
        verbose_name=_("Percent of consumption turn into bouns(10 = 10%)"), default=0.0,
        validators=[MinValueValidator(0.0),],
    )
    discount = models.FloatField(
        verbose_name=_("Discount per bonus"), default=0.0, 
        validators=[MinValueValidator(0.0),],
    )
    limit = models.IntegerField(
        verbose_name=_("Dividend use cap(-1 is no limit)"), default=-1,
        validators=[MinValueValidator(-1),],
    )
    
    receiveStatus = models.CharField(
        choices=settings.OSCAR_ORDER_STATUS, 
        default=settings.OSCAR_ORDER_COMPLETE_STATUS,
        max_length=50, verbose_name=_("Receive status"))
    
    includeRange = models.ForeignKey(
        'offer.Range',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        verbose_name=_("Include range(default is all products)"),
        related_name='bonusIncludeEange'
    )
    excludeRange = models.ForeignKey(
        'offer.Range',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        verbose_name=_("Exclude range(default is no limit)"),
        related_name='bonusExcludeRange'
    )
    
    expiryDays = models.IntegerField(
        verbose_name=_("Expiry days(-1 is no limit)"), default=-1, validators=[MinValueValidator(-1),],)
    
# https://code.nuwainfo.com/trac/Mercurius/ticket/3301#ticket
CommunicationEventType = get_model('customer', 'CommunicationEventType')
RawHTML = get_model('promotions', 'RawHTML')

def getATMInfo():
    info, created = RawHTML.objects.get_or_create(
        name='ATM Info',
        defaults={
            'body': '<p>XXX</p><p>XXXXXXXXXXXXX</p>',
        }
    )
    
    return info
    
originGetMessages = CommunicationEventType.get_messages

def get_messages(self, ctx=None):
    code = self.code.lower()
    
    if ctx is None:
        ctx = {}
        
    if code == 'order_placed':
        info = getATMInfo()
        ctx['ATMInfo'] = info
        
    return originGetMessages(self, ctx=ctx)
        
CommunicationEventType.get_messages = get_messages

class TradeSession(models.Model):
    basket = models.ForeignKey(
        Basket,
        verbose_name=_("Basket"),
        on_delete=models.CASCADE,
    )
    sessionKey = models.CharField(max_length=255)
    
class ECPayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)
    
    tradeDate = models.DateTimeField()
    paymentDate = models.DateTimeField()
    
    tradeAmt = models.IntegerField()
    payAmt = models.IntegerField()
         
    rtnCode = models.CharField(max_length=255)
    rtnMsg = models.CharField(max_length=255)
    paymentType = models.CharField(max_length=255)
    simulatePaid = models.CharField(max_length=255)
    paymentTypeChargeFee = models.CharField(max_length=255)

class AllPayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)
    
    tradeDate = models.DateTimeField()
    paymentDate = models.DateTimeField()
    
    tradeAmt = models.IntegerField()
    payAmt = models.IntegerField()
         
    rtnCode = models.CharField(max_length=255)
    rtnMsg = models.CharField(max_length=255)
    paymentType = models.CharField(max_length=255)
    simulatePaid = models.CharField(max_length=255)
    paymentTypeChargeFee = models.CharField(max_length=255)

class SpgatewayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)   
    paymentDate = models.DateTimeField()
    paymentType = models.CharField(max_length=255) 
    tradeAmt = models.IntegerField()
    rtnMsg = models.CharField(max_length=255)
    
    paymentMethod = models.CharField(max_length=255, null=True)
    rtnCode = models.CharField(max_length=255, null=True)
    
class CathaybkCreditTrade(models.Model):
    merchantId = models.CharField(max_length=255) # STOREID
    merchantTradeNo = models.CharField(max_length=255) # ORDERNUMBER
    tradeNo = models.CharField(max_length=255) # AUTHCODE
    
    paymentDate = models.DateTimeField() # AUTHTIME
    
    tradeAmt = models.IntegerField() # AMOUNT
         
    rtnCode = models.CharField(max_length=255) # AUTHSTATUS
    rtnMsg = models.CharField(max_length=255) # AUTHMSG
    cardNo = models.CharField(max_length=255) # CARDNO

class ECPayLogisticTrade(models.Model):
    # B2C & C2C
    merchantId = models.CharField(max_length=255) # MerchantID
    merchantTradeNo = models.CharField(max_length=255) # MerchantTradeNo
    tradeNo = models.CharField(max_length=255) # AllPayLogisticsID
    
    tradeDate = models.DateTimeField()
    updateStatusDate = models.DateTimeField() # UpdateStatusDate
    
    tradeAmt = models.IntegerField() # GoodsAmount
         
    rtnCode = models.CharField(max_length=255) # RtnCode
    rtnMsg = models.CharField(max_length=255) # RtnMsg

    receiverName = models.CharField(max_length=255) # ReceiverName
    receiverCellPhone = models.CharField(max_length=255) # ReceiverCellPhone
    receiverEmail = models.CharField(max_length=255, null=True) # ReceiverEmail

    logisticsType = models.CharField(max_length=255) # LogisticsType
    logisticsSubType = models.CharField(max_length=255) # LogisticsSubType

    # return
    rtnMerchantTradeNo = models.CharField(max_length=255, null=True)
    rtnOrderNo = models.CharField(max_length=255, null=True)

    # CVS C2C
    paymentNo = models.CharField(max_length=255, null=True) # CVSPaymentNo 
    validationNo = models.CharField(max_length=255, null=True) # CVSValidationNo 
    
