#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: forms.py 10641 2017-12-12 18:08:04Z David $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-12-13 02:08:04 +0800 (Wed, 13 Dec 2017) $
# $Revision: 10641 $

from django import forms
from django.forms.models import save_instance
from django.core.urlresolvers import reverse
from django.utils.translation import ugettext as _

from Iuppiter.extension.Widgets import ReadOnlyWidget

from Theophrastus.integration.models import App, GroupMappingTable, Group
from Theophrastus.integration.service import IntegrationService

integrationSrv = IntegrationService.getInstance()

class AppAdminForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        super(AppAdminForm, self).__init__(*args, **kwargs)
        obj = kwargs.get('instance', None)
        if obj:
            # Widgets.
            self.fields['user'].widget = ReadOnlyWidget(
                originalValue=obj.user.id,
                displayValue=obj.user.username,
                link=reverse('admin:auth_user_change', args=[obj.user.id])
            )

    class Meta:
        model = App
        fields = '__all__'

class GroupMappingTableForm(forms.ModelForm):

    def __init__(self, *args, **kwargs):
        self.request = kwargs.pop('request', None)
        super(GroupMappingTableForm, self).__init__(*args, **kwargs)
        user = self.request.user

        self._removeCurrentGroup = False
        # Widgets.
        apps = integrationSrv.findApps(user=user)
        # App.objects.none()
        self.fields['app'] = forms.ModelChoiceField(queryset=apps)

        self.fields['group'] = forms.CharField(
            required=False,
            help_text=_('Please enter your group name that you want matched.'))

        self.fields['current'].required = False

    condition = forms.CharField(
        widget=forms.TextInput(attrs={'size': 80}),
        help_text=_('Please enter a reqular expression or a keyword.'))

    conditionType = forms.TypedChoiceField(
        choices=GroupMappingTable.CONDITION_TYPES_CHOICES,
        widget=forms.RadioSelect, coerce=str)

    def clean_group(self):
        data = self.data
        group = data['group']
        if self.data['choiceGroup'] == '0': # Use a new group.
            if group.strip() == '':
                raise forms.ValidationError('Need a group name.')
        elif self.data['choiceGroup'] == '1': # Use current group.
            if group.strip() != '':
                return ''
        return group

    def clean_current(self):
        data = self.data
        current = data['current']
        if data['choiceGroup'] == '0': # Use a new group.
            self._removeCurrentGroup = True
        elif self.data['choiceGroup'] == '1': # Use current group.
            if current == '':
                raise forms.ValidationError('Need a current group.')

        if self._removeCurrentGroup:
            return None
        return Group.objects.get(id=current)

    def save(self, commit=True):
        """
            Saves this ``form``'s cleaned_data into model instance
            ``self.instance``.

            If commit=True, then the changes to ``instance`` will be saved to the
            database. Returns ``instance``.
            """
        if self.instance.pk is None:
            fail_message = 'created'
        else:
            fail_message = 'changed'

        if self._removeCurrentGroup:
            self.instance.current = None
        return save_instance(self, self.instance, self._meta.fields,
                             fail_message, commit, construct=False)
    save.alters_data = True

from Theophrastus.admin import NewsletterTinyMCEForm
from Theophrastus.models import MailingList

class NewsletterIntegrationForm(NewsletterTinyMCEForm):
    def __init__(self, *args, **kwargs):
        super(NewsletterIntegrationForm, self).__init__(*args, **kwargs)
        ids = []
        for g in integrationSrv.findAllGroups():
            ns = g.newsletters.values('id')
            ids.extend([g['id'] for g in ns])
        self.fields['mailing_list'].queryset = MailingList.objects.exclude(
            newsletter__id__in=ids)
