#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: cms_plugins.py 11884 2020-01-20 08:51:15Z Trista $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Trista $
# $Date: 2020-01-20 17:51:15 +0900 (週一, 20 一月 2020) $
# $Revision: 11884 $


import requests


from django.contrib import admin
from django.utils.translation import ugettext as _
from django.conf import settings
from django.template.loader import select_template

from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool

from collections import namedtuple

from Zephyrus.map import settings as mapSettings
from Zephyrus.map.models import Map, MapPosition


Position = namedtuple('Position', 
    ['region', 'name', 'tel', 'email', 'address', 'coordinate']
)


class PositionInline(admin.StackedInline):
    model = MapPosition
    extra = 0


    def get_fieldsets(self, request, obj=None):
        fields = (
            ('region', 'name', 'tel', 'email', 'address')
        )

        fieldsets = (
            (None, {
                'fields': fields
            }),
            (None, {
                'description': _('<p>Coordinate format: lat,lng (string type) \
                 ex: 24,121 (means 24°0′N 121°0′E)</p>'),
                'fields': ('coordinate',),
            }),
        )
        return fieldsets



class MapPlugin(CMSPluginBase):
    model = Map
    module = _("Map")
    name = _("Map")
    inlines = (PositionInline, )
    render_template = mapSettings.MAP_DEFAULT_TEMPLATE

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            mapSettings.MAP_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):

        positionList = []
        
        for position in instance.positions.all():
            spot = Position(
                position.region, 
                position.name,
                position.tel,
                position.email,
                position.address,
                position.coordinate,
            )
        
            positionList.append(spot)
            
        context.update({
            'positionList': positionList,
            'apiKey': instance.apiKey,
        })

        return context




plugin_pool.register_plugin(MapPlugin)
