#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 11234 2018-09-29 12:58:04Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2018-09-29 21:58:04 +0900 (週六, 29 九月 2018) $
# $Revision: 11234 $

from functools import wraps
from django import forms
from django.db import models
from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from cms.models import CMSPlugin
from aldryn_categories.models import Category
from aldryn_newsblog.models import Article
from aldryn_newsblog import models as newsblogModels
from taggit.models import Tag

from Zephyrus.blog_extension import settings as _settings

METHODS = [
    ('or',  _('OR')),
    ('and',  _('AND')),
]

class FilterPlugin(CMSPlugin):
    limit = models.PositiveIntegerField(_('Limit'), default=6)

    filterMethod = models.CharField(
        _('Filter Method'), max_length=3, choices=METHODS, default='or')

    tag = models.ManyToManyField(
        Tag,
        verbose_name=_('Tag'),
        default=None,
        blank=True,
        null=True,
        help_text=_(
            "If you check 'Not use tags to filter article'. "
            "It will no effect.")
    )

    catagory = models.ForeignKey(
        Category,
        verbose_name=_('Catagory'),
        on_delete=models.CASCADE,
        related_name='+',
        default=None,
        blank=True,
        null=True,
        help_text=_(
            "If you check 'Not use catagories to filter articles'. "
            "It will no effect.")
    )

    allTag = models.BooleanField(
        _('Not use tags to filter articles'), 
        default=True,
        help_text=_(
            'If you check this. It will not filter articles with tags.')
    )
    allCatagory = models.BooleanField(
        _('Not use catagories to filter articles'), 
        default=True,
        help_text=_(
            'If you check this. It will not filter articles with catagories.')
    )

    renderTemplate = models.CharField(
        _('Render template'), max_length=150,
        choices=_settings.ZEPHYRUS_BLOG_MAYLIKE_TEMPLATES,
        default=_settings.ZEPHYRUS_BLOG_MAYLIKE_DEFAULT_TEMPLATE,
    )

class DatePluginConfig(CMSPlugin):
    renderTemplate = models.CharField(
        _('Render template'), max_length=150,
        choices=_settings.ZEPHYRUS_BLOG_DATE_TEMPLATES,
        default=_settings.ZEPHYRUS_BLOG_DATE_DEFAULT_TEMPLATE,
    )

class ViewCount(models.Model):
    url = models.CharField(max_length=400, editable=False)
    count = models.PositiveIntegerField(default=0)


# modify slug field formfield
def formfieldDecorator(f):
    @wraps(f)
    def wrapper(*args, **kwds):
        return f(form_class=forms.CharField ,*args, **kwds)
    return wrapper

for field in Article.translations.related.related_model._meta.fields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

for field in Category.translations.related.related_model._meta.fields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

for field in Tag._meta.fields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

# https://code.nuwainfo.com/trac/Mercurius/ticket/3134#comment:1

from aldryn_newsblog.utils import utilities
from aldryn_newsblog.utils.utilities import get_cleaned_bits, get_field_value

def get_plugin_index_data(base_plugin, request):
    # copy from  aldryn_newsblog.utils.utilities.get_plugin_index_data

    text_bits = []

    instance, plugin_type = base_plugin.get_plugin_instance()

    if instance is None:
        # this is an empty plugin
        return text_bits

    search_fields = getattr(instance, 'search_fields', [])

    if hasattr(instance, 'search_fulltext'):
        # check if the plugin instance has search enabled
        search_contents = instance.search_fulltext
    elif hasattr(base_plugin, 'search_fulltext'):
        # now check in the base plugin instance (CMSPlugin)
        search_contents = base_plugin.search_fulltext
    elif hasattr(plugin_type, 'search_fulltext'):
        # last check in the plugin class (CMSPluginBase)
        search_contents = plugin_type.search_fulltext
    else:
        # disabled if there's search fields defined,
        # otherwise it's enabled.
        search_contents = not bool(search_fields)
    
    if search_contents:
        if hasattr(instance, 'render_plugin'):
            plugin_contents = instance.render_plugin(
                context=RequestContext(request))

            if plugin_contents:
                text_bits = get_cleaned_bits(plugin_contents)
    else:
        values = (get_field_value(instance, field) for field in search_fields)

        for value in values:
            cleaned_bits = get_cleaned_bits(value or '')
            text_bits.extend(cleaned_bits)
    return text_bits

utilities.get_plugin_index_data = get_plugin_index_data
newsblogModels.get_plugin_index_data = get_plugin_index_data


