#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 11904 2020-02-05 13:38:19Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-02-05 21:38:19 +0800 (週三, 05 二月 2020) $
# $Revision: 11904 $

import urllib
import re
import six
import unicodedata

from django import forms
from django.contrib import admin
from django.conf import settings
from django.contrib.redirects.models import Redirect
from django.contrib.sites.shortcuts import get_current_site
from django.utils.translation import ugettext_lazy as _
from django.db import IntegrityError, transaction
from aldryn_newsblog.admin import ArticleAdmin, ArticleAdminForm
from aldryn_newsblog.models import Article
from aldryn_categories.admin import CategoryAdmin
from aldryn_categories.models import Category
from djangocms_text_ckeditor.models import Text
from taggit import models
from awesome_slugify import Slugify

from Iuppiter.DjangoUtil import DJANGO_VERSION

if DJANGO_VERSION >= 20000:
    from django.urls import reverse_lazy
else:
    from django.core.urlresolvers import reverse_lazy

from Zephyrus.blog_extension import views

unicodeSlugify = Slugify(translate=None)

# from aldryn_translation_tools import models
# models.slugify = unicodeSlugify # doesn't work

# TAG
models.default_slugify = unicodeSlugify

def slugify(self, tag, i=None):
    slug = models.default_slugify(tag)
    if i is not None:
        slug += "_%d" % i
    return slug

def save(self, *args, **kwargs):
    if self._state.adding and not self.slug:
        self.slug = self.slugify(self.name)
        from django.db import router
        using = kwargs.get("using") or router.db_for_write(
            type(self), instance=self)
        # Make sure we write to the same db for all attempted writes,
        # with a multi-master setup, theoretically we could try to
        # write and rollback on different DBs
        kwargs["using"] = using
        # Be oportunistic and try to save the tag, this should work for
        # most cases ;)
        try:
            with transaction.atomic(using=using):
                res = super(models.TagBase, self).save(*args, **kwargs)
            return res
        except IntegrityError:
            pass
        # Now try to find existing slugs with similar names
        slugs = set(
            self.__class__._default_manager
            .filter(slug__startswith=self.slug)
            .values_list('slug', flat=True)
        )
        i = 1
        while True:
            slug = self.slugify(self.name, i)
            if slug not in slugs:
                self.slug = slug
                # We purposely ignore concurrecny issues here for now.
                # (That is, till we found a nice solution...)
                return super(models.TagBase, self).save(*args, **kwargs)
            i += 1
    else:
        if self.slug:
            self.slug = self.slugify(self.slug)
        return super(models.TagBase, self).save(*args, **kwargs)

models.TagBase.slugify = slugify
models.TagBase.save = save

# Article
class ArticleAdminForm2(ArticleAdminForm):
    oldSlug = forms.CharField(
        label=_('slug'),
        required=False, max_length=255, 
        help_text=_(
            'Used in the URL. If changed, the URL will change. '
            'Clear it to have it re-created automatically.'),
        widget=forms.HiddenInput(),
    )

    class Meta:
        model = Article
        fields = [
            'app_config',
            'categories',
            'featured_image',
            'is_featured',
            'is_published',
            'lead_in',
            'meta_description',
            'meta_keywords',
            'meta_title',
            'owner',
            'related',
            'slug',
            'tags',
            'title',
        ]

class ArticleAdmin2(ArticleAdmin):
    form = ArticleAdminForm2

    fieldsets = (
        (None, {
            'fields': (
                'title',
                'author',
                'publishing_date',
                'is_published',
                'is_featured',
                'featured_image',
            )
        }),
        (_('Meta Options'), {
            'classes': ('collapse',),
            'fields': (
                'slug',
                'oldSlug',
                'meta_title',
                'meta_description',
                'meta_keywords',
            )
        }),
        (_('Advanced Settings'), {
            'classes': ('collapse',),
            'fields': (
                'tags',
                'categories',
                'related',
                'owner',
                'app_config',
            )
        }),
    )
   
    def get_form(self, request, obj=None, **kwargs):
        form = super(ArticleAdmin2, self).get_form(request, obj, **kwargs)
        if obj:
            form.base_fields['oldSlug'].initial = obj.slug
        else:
            form.base_fields['oldSlug'].initial = ''
        return form

    def response_add(self, request, obj, post_url_continue=None):
        body = _("double click here to edit article content")
        textPlugin = Text.objects.create(
            placeholder=obj.content,
            language=obj.language_code,
            body=body, plugin_type='TextPlugin')

        slug = request.POST.get('slug')
        if slug:
            obj.slug = unicodeSlugify(slug)
        else:
            obj.slug = unicodeSlugify(obj.title)
        obj.save()
        
        return super(ArticleAdmin2, self).response_add(
            request, obj, post_url_continue=None)

    def response_change(self, request, obj):
        slug = request.POST.get('slug')
        if slug:
            obj.slug = unicodeSlugify(slug)
        else:
            obj.slug = unicodeSlugify(obj.title)
        obj.save()

        oldslug = request.POST.get('oldSlug')
        if not oldslug == obj.slug:
            site = get_current_site(request)
            oldUrl = reverse_lazy(
                'aldryn_newsblog:article-detail', kwargs={'slug': oldslug,})
            newUrl = reverse_lazy(
                'aldryn_newsblog:article-detail', kwargs={'slug': obj.slug,})

            Redirect.objects.create(
                site=site,
                old_path=oldUrl,
                new_path=newUrl,
            )
            
        
        return super(ArticleAdmin2, self).response_change(request, obj)

admin.site.unregister(Article)
admin.site.register(Article, ArticleAdmin2)

# Category
class CategoryAdmin2(CategoryAdmin):

    def response_add(self, request, obj, post_url_continue=None):
        slug = request.POST.get('slug')
        if slug:
            obj.slug = unicodeSlugify(slug)
        else:
            obj.slug = unicodeSlugify(obj.name)
        obj.save()
        return super(CategoryAdmin2, self).response_add(
            request, obj, post_url_continue=None)

    def response_change(self, request, obj):
        slug = request.POST.get('slug')
        if slug:
            obj.slug = unicodeSlugify(slug)
        else:
            obj.slug = unicodeSlugify(obj.name)
        obj.save()
        return super(CategoryAdmin2, self).response_change(request, obj)

admin.site.unregister(Category)
admin.site.register(Category, CategoryAdmin2)


