import datetime

from babel.numbers import format_currency
from django.contrib.sites.models import Site
from django.conf import settings
from django.test import RequestFactory
from django.contrib import auth
from django.core.management.base import BaseCommand, CommandError
from django.contrib.sessions.backends.base import SessionBase
from django.utils.translation import get_language, to_locale
from django.contrib.messages.storage import default_storage

from oscar.core.loading import get_model, get_class

from Iuppiter.Logging import createLogger

logger = createLogger(__name__)

Line = get_model('basket', 'line')
Basket = get_model('basket', 'basket')
Product = get_model('catalogue', 'product')
OrderLine = get_model('order', 'line')
Address = get_model('order', 'ShippingAddress')
Order = get_model('order', 'Order')
Source = get_model('payment', 'Source')
SourceType = get_model('payment', 'SourceType')

Selector = get_class('partner.strategy', 'Selector')
Repository = get_class('shipping.repository', 'Repository')
OrderTotalCalculator = get_class(
    'checkout.calculators', 'OrderTotalCalculator')
Applicator = get_class('offer.utils', 'Applicator')
OrderCreator = get_class('order.utils', 'OrderCreator')
OrderNumberGenerator = get_class('order.utils', 'OrderNumberGenerator')

class Command(BaseCommand):
    help = (
        "Create a local version of one of Oscar's app so it can "
        "be customised")

    def add_arguments(self, parser):
        parser.add_argument('basket', help='Basket ID of Frozen')
        parser.add_argument('address', help='Address ID for order')
        parser.add_argument('shipping', help='Shipping code for order')
        parser.add_argument('-d', '--dry-run', action='store_true', help='not create order')
        parser.add_argument('-p', '--payment-type', help='payment type.ex: ecpay spgateway')
        parser.add_argument('-t', '--trade-no', help='trade no of payment')
        
    def formatCurrency(self, total, currency):
        currencyStr = format_currency(
            total, currency=currency, format='##0', currency_digits=False,
            locale=to_locale(get_language() or settings.LANGUAGE_CODE))
        
        total = int(currencyStr)
        return total
        
        
    def handle(self, *args, **options):
        # required
        dryRun = options['dry_run'] 
        basket = Basket.objects.get(id=options['basket'])
        address = Address.objects.get(id=options['address'])
        code = options['shipping'] 
        
        paymentType = options.get('payment_type', None)
        if paymentType:
            tradeNo = options.get('trade_no', None)
            if not tradeNo:
                raise ValueError("Put input --trade-no(-t) when you use --payment-type(-p).")
        
        # make fake request
        requestFactory = RequestFactory()
        request = requestFactory.get("/")
        request.session = SessionBase()
        request._messages = default_storage(request)
        request.user = basket.owner
        request.basket = basket
        selector = Selector()
        strategy = selector.strategy(request=request, user=request.user)
        basket.strategy = strategy
        
        # shipping 
        methods = Repository().get_shipping_methods(
            basket=basket, user=request.user,
            shipping_addr=address, request=request)
        shippingMethod = None
        for method in methods:
            if method.code == code:
                shippingMethod = method
        if not shippingMethod:
            raise ValueError("No shipping code name %s." % code)
            
        # set offer
        Applicator().apply(request.basket, request.user, request)
            
        # calculate total
        shippingCharge = shippingMethod.calculate(basket)
        total = OrderTotalCalculator(request).calculate(basket, shippingCharge)
            
        # check order data
        orderData = {
            'number': OrderNumberGenerator().order_number(basket),
            'site': Site.objects.get(id=settings.SITE_ID),
            'basket': basket,
            'user': basket.owner,
            'billing_address': None,
            
            'currency': total.currency,
            'total_incl_tax': self.formatCurrency(total.incl_tax, total.currency),
            'total_excl_tax': self.formatCurrency(total.excl_tax, total.currency),
            'shipping_incl_tax': 0,
            'shipping_excl_tax': 0,
            
            'shipping_address': address,
            'shipping_method': shippingMethod.name,
            'shipping_code': shippingMethod.code,
            
            'status': '等待處理中',
            'guest_email': '',
            'date_placed': datetime.datetime.now(),
        }
        
        # print data
        printData = []
        for k, v in orderData.items():
            printData.append("%s: %s" % (k, v))
            
        logger.info("\n%s" % "\n".join(printData))
        
        # create order
        if not dryRun:
            order = OrderCreator().place_order(
                user=orderData['user'],
                order_number=orderData['number'],
                basket=basket,
                shipping_address=address,
                shipping_method=shippingMethod,
                shipping_charge=shippingCharge,
                total=total,
                billing_address=None,
                status=None,
                request=request,
            )
            basket.submit()
            logger.info("Order Created: %s" % order)
            
            if paymentType:
                sourceType, created = SourceType.objects.get_or_create(name=paymentType)
                source = Source.objects.create(
                    source_type=sourceType,
                    currency=total.currency,
                    amount_allocated=total.incl_tax,
                    amount_debited=total.incl_tax,
                    reference=tradeNo,
                    order=order,
                ) 
                logger.info("Order Payment Created: %s" % source)
            
