#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 12918 2021-05-27 07:26:58Z Lavender $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2021-05-27 15:26:58 +0800 (週四, 27 五月 2021) $
# $Revision: 12918 $

from django.contrib import admin
from django.contrib.admin.views.main import ChangeList
from django.urls import reverse
from django.utils.translation import ugettext as _
from django.utils.safestring import mark_safe

from emencia.django.newsletter.models import Newsletter

from Theophrastus.statistics.admin import NewsletterStatisticsAdmin
from Theophrastus.integration.service import IntegrationService
from Theophrastus.integration.forms import AppAdminForm, GroupMappingTableForm
from Theophrastus.integration.forms import NewsletterIntegrationForm
from Theophrastus.integration.models import (
    App, Group, GroupMappingTable
)

integrationSrv = IntegrationService.getInstance()

class AppChangeList(ChangeList):
    def __init__(self, request, *args, **kwargs):
        self._request = request
        self._user = request.user
        super(AppChangeList, self).__init__(request, *args, **kwargs)

    def get_queryset(self, request):

        if self._user.is_authenticated and self._user.is_staff:
            user = self._user
            queryset = super(AppChangeList, self).get_queryset(request)
            queryset = queryset.filter(user=user)
            return queryset
        else:
            raise RuntimeError('Need authenticating.')

class AppAdmin(admin.ModelAdmin):

    list_display = ('goGroup', 'getLastUpdatedTime', 'formatedCreatedTime',
                    'groupCount', 'goEdit',)

    readonly_fields = ('formatedCreatedTime',)

    search_fields = ('name', 'createdTime',)

    fieldsets = (
        (None, {
            'fields': ('name', 'user',)
        }),
    )

    actions_on_top = False

    actions_on_bottom = False

    form = AppAdminForm

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        return AppChangeList

    def getLastUpdatedTime(self, obj):
        """
        Get last updated time by newsletters.
        """
        allTimes = []
        for g in obj.groups.all():
            result = g.newsletters.order_by('-modification_date')
            if len(result) > 0:
                allTimes.append(result[0].modification_date)
        if allTimes:
            lastTime = max(allTimes)
            return lastTime.strftime("%Y-%m-%d %H:%M:%S")
        else:
            return '-'
    getLastUpdatedTime.short_description = _('Last updated time')
    getLastUpdatedTime.allow_tags = True
    getLastUpdatedTime.admin_order_field = 'createdTime'

    def goEdit(self, obj):
        link = reverse('admin:integration_app_change', args=[obj.id])
        return mark_safe('<a href="%s">%s</a>' % (link, _('Edit')))
    goEdit.short_description = _('Edit')
    goEdit.allow_tags = True

    def goGroup(self, obj):
        link = reverse('admin:integration_group_changelist')
        return mark_safe('<a href="%s?app=%d">%s</a>' % (link, obj.id, obj.name))
    goGroup.short_description = _('Name')
    goGroup.allow_tags = True

    def formatedCreatedTime(self, obj):
        return obj.createdTime.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreatedTime.short_description = _('Created time')

    def groupCount(self, app):
        """
        Display group count.

        @param app App instance.
        """
        return app.groups.count()
    groupCount.short_description = _('Group count')

admin.site.register(App, AppAdmin)

class GroupChangeList(ChangeList):
    def __init__(self, request, *args, **kwargs):
        self._app = request.GET.get('app', None)
        self._user = request.user
        super(GroupChangeList, self).__init__(request, *args, **kwargs)

    def findIdsByUser(self):
        ids = []
        for a in integrationSrv.findApps(user=self._user):
            groups = a.groups.values('id')
            ids.extend([g['id'] for g in groups])
        return ids

    def getApps(self,):
        if not self._app is None:
            if 'app' in self.params:
                del self.params['app']

            _app = integrationSrv.getApp(id=self._app, user=self._user)
            return [q['id'] for q in _app.groups.values('id')]

        return []

    def getQuerySet(self, request=None, original=False):
        apps = self.getApps()

        if original:
            queryset = integrationSrv.findGroups(id__in=self.findIdsByUser())
            if apps:
                queryset = queryset.filter(id__in=apps)
            return queryset

        queryset = super(GroupChangeList, self).get_queryset(request)
        if apps:
            if 'app' in self.params:
               del self.params['app']
            _queryset = None
            try:
                _app = integrationSrv.getApp(id=self._app, user=self._user)
                _queryset = _app.groups.all()
            except:
                return Group.objects.none()

            if self.params:
                ids = [q['id'] for q in _queryset.values('id')]
                queryset = super(GroupChangeList, self).get_queryset(request)
            return queryset.filter(id__in=apps)
        else:
            return queryset.filter(id__in=self.findIdsByUser())

    def get_queryset(self, request):

        return self.getQuerySet(request=request)

class GroupAdmin(admin.ModelAdmin):

    list_display = ('goNewsletter', 'getLastUpdatedTime', 'formatedCreatedTime',
                    'newsletterCount', 'displayHistory', 'displayStatistics',
                    'displaySubscribers')

    readonly_fields = ('formatedCreatedTime',)

    search_fields = ('name',)

    actions_on_top = False

    actions_on_bottom = False

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        return GroupChangeList

    def goNewsletter(self, obj):
        """
        Get last updated time by newsletters.
        """
        link = reverse('admin:newsletter_newsletter_changelist')
        return mark_safe('<a href="%s?group=%d">%s</a>' % (link, obj.id, obj.name))
    goNewsletter.short_description = _('name')
    goNewsletter.allow_tags = True

    def getLastUpdatedTime(self, obj):
        result = obj.newsletters.order_by('-modification_date')
        if len(result) > 0:
            return result[0].modification_date.strftime("%Y-%m-%d %H:%M:%S")
        else:
            return '-'
    getLastUpdatedTime.short_description = _('Last updated time')
    getLastUpdatedTime.allow_tags = True

    def displayHistory(self, obj):
        return mark_safe('<a href="%s">%s</a>' % (
            reverse(
                'Theophrastus.admin.integration.views.getGroupHistory', 
                kwargs={'groupId': obj.id}
            ), _('View historic')
        ))
    displayHistory.short_description = _('History')
    displayHistory.allow_tags = True

    def displayStatistics(self, obj):
        return mark_safe(obj.getStatisticsLink())
    displayStatistics.short_description = _('Statistics')
    displayStatistics.allow_tags = True

    def displaySubscribers(self, obj):
        return mark_safe(obj.getSubscriberLink())
    displaySubscribers.short_description = _('Subscribers')
    displaySubscribers.allow_tags = True

    def formatedCreatedTime(self, obj):
        return obj.createdTime.strftime("%Y-%m-%d %H:%M:%S")
    formatedCreatedTime.short_description = _('Created time')

    def newsletterCount(self, group):
        """
        Display newsletter count.

        @param group Group instance.
        """
        return group.newsletters.count()
    newsletterCount.short_description = _('Newsletter count')

admin.site.register(Group, GroupAdmin)

class GroupMappingTableAdmin(admin.ModelAdmin):
    form = GroupMappingTableForm

    def get_form(self, request, obj=None, **kwargs):
        ModelForm = super(GroupMappingTableAdmin,
                          self).get_form(request, obj, **kwargs)
        class ModelFormMetaClass(ModelForm):
            def __new__(cls, *args, **kwargs):
                kwargs['request'] = request
                _m = ModelForm(*args, **kwargs)
                return _m
        return ModelFormMetaClass

admin.site.register(GroupMappingTable, GroupMappingTableAdmin)

from emencia.django.newsletter.models import MailingList
from emencia.django.newsletter.admin import MailingListAdmin

class MailingListChangeList(ChangeList):
    def __init__(self, request, *args, **kwargs):
        super(MailingListChangeList, self).__init__(request, *args, **kwargs)

    def getQuerySet(self, request=None, original=False):
        if original:
            queryset = MailingList.objects.all()
            return queryset
        else:
            queryset = super(MailingListChangeList, self).get_queryset(request)

            ids = []
            for _a in integrationSrv.findAllApps():
                groups = _a.groups.values('id')
                ids.extend([g['id'] for g in groups])
            return queryset.exclude(newsletter__group__id__in=ids)

    def get_queryset(self, request):
        return self.getQuerySet(request=request)

class MailingListIntegrationAdmin(MailingListAdmin):
    list_filter = ('creation_date', 'modification_date', )
    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        return MailingListChangeList

admin.site.unregister(MailingList)
admin.site.register(MailingList, MailingListIntegrationAdmin)

class AppNewsletterFilter(admin.SimpleListFilter):
    # Human-readable title which will be displayed in the
    # right admin sidebar just above the filter options.
    title = _('App')

    # Parameter for the filter that will be used in the URL query.
    parameter_name = 'decade'

    def lookups(self, request, model_admin):
        """
        Returns a list of tuples. The first element in each
        tuple is the coded value for the option that will
        appear in the URL query. The second element is the
        human-readable name for the option that will appear
        in the right sidebar.
        """
        appTuple = []
        apps = App.objects.all()
        for app in apps:
            appName = app.name
            appId = app.id
            appTuple.append((appId, appName,))
        return appTuple

    def queryset(self, request, queryset):
        """
        Returns the filtered queryset based on the value
        provided in the query string and retrievable via
        `self.value()`.
        """
        # Compare the requested value (either '80s' or '90s')
        # to decide how to filter the queryset.
        appId = self.value()
        if appId:
            appNewsletterList = AppNewsletter.objects.filter(app_id=appId)
            appNewsletterList = appNewsletterList.values_list(
                'newsletter', flat=True)
            return Newsletter.objects.filter(id__in=appNewsletterList)
        else:
            return Newsletter.objects.all()

class NewsletterIntegrationChangeList(ChangeList):
    def __init__(self, request, *args, **kwargs):
        self._user = request.user
        self._group = request.GET.get('group', None)
        super(NewsletterIntegrationChangeList,
              self).__init__(request, *args, **kwargs)

    def getAllAppIds(self):
        _ids = []
        for a in integrationSrv.findAllApps():
            for g in a.groups.all():
                ids = g.newsletters.values('id')
                _ids.extend([g['id'] for g in ids])
        return _ids

    def getIdsByUser(self):
        _ids = []
        for a in integrationSrv.findApps(user=self._user):
            for g in a.groups.all():
                ids = g.newsletters.values('id')
                _ids.extend([g['id'] for g in ids])
        return _ids

    def getIdsByGroup(self):
        if not self._group is None:
            if 'group' in self.params:
                del self.params['group']

            _group = Group.objects.get(id=self._group)
            return [q['id'] for q in _group.newsletters.values('id')]
        return []

    def getQuerySet(self, request=None, original=False):

        ids = self.getIdsByGroup()

        if original:
            if ids:
                return Newsletter.objects.filter(id__in=ids)
            else:
                return Newsletter.objects.exclude(id__in=self.getAllAppIds())
        else:
            queryset = super(
                NewsletterIntegrationChangeList, self).get_queryset(request)
            if ids:
                return queryset.filter(id__in=ids)
            else:
                return queryset.exclude(id__in=self.getAllAppIds())
            

    def get_queryset(self, request):
        return self.getQuerySet(request=request, original=False)

class NewsletterIntegrationAdmin(NewsletterStatisticsAdmin):
    
    list_filter = (
        'status', 'sending_date', 'creation_date', 'modification_date',
    )
    form = NewsletterIntegrationForm
    class Media:
        js = [
            'js/intergation/Newsletter.js',
        ]

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        return NewsletterIntegrationChangeList

admin.site.unregister(Newsletter)
admin.site.register(Newsletter, NewsletterIntegrationAdmin)