#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 12366 2020-06-19 06:41:16Z Andy $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $
# $Date: 2020-06-19 14:41:16 +0800 (週五, 19 六月 2020) $
# $Revision: 12366 $

import os
import re
import base64
import json

from random import randint, choice

from datetime import datetime

from django.core.files.base import ContentFile

from Iuppiter.Service import BaseService

from Iuppiter import debug

from emencia.django.newsletter.models import Contact, Newsletter, \
                                             MailingList, Attachment

from Theophrastus.integration import models
from Theophrastus.Utility import _slugify

from Theophrastus.cloud.service import CloudService

WORDS = [chr(i) for i in range(ord('a'), ord('a') + 26)]

cloudSrv = CloudService.getInstance()

class IntegrationService(BaseService):

    def __init__(self, *args, **kws):
        """
        Constructor.
        """
        super(IntegrationService, self).__init__(models)

    def _getTime(self):
        """
        Get current time.
        """
        return datetime.strftime(datetime.now(), '%Y%m%d%H%M%S')

    def _getRandomWords(self, length=5):
        """
        Get random words.

        @param length Word length.
        """
        words = ''
        _condition = [True, False]
        for i in range(length):
            if choice(_condition):
                words += str(randint(0, 9))
            else:
                words += choice(WORDS)
        return words

    def _getNewsletterSlug(self, content):
        """
        Get valid slug for newsletter.

        @param content The content of slug.
        """
        slug = _slugify(content)

        def _cleanSlug(slug):
            _slug = '%s%s' % (slug[:30], self._getRandomWords())
            _slug = '%s%s' % (_slug, self._getTime())
            _slug = _slug[:50]
            return _slug

        _slug = _cleanSlug(slug)
        while Newsletter.objects.filter(slug=_slug):
            _slug = _cleanSlug(slug)

        return _slug

    def createMail(self, group, emails, title, text, headerSender='',
                   headerReply='', attachments=None): # TOFIX , async=False
        """
        Create mail.
        We need create a mailing list and contacts to newsletter.
        The mailer will get this newsletter and send it after created.

        @param group Group instance.
        @param emails Send mail to?
        @param title Mail title.
        @param text Mail text.
        @param headerSender Sender for mail header.
        @param headerReply Reply for mail header.
        @return Newsletter instance.
        """
        # Create contacts.
        contacts = []
        for email in emails:
            ct, created = Contact.objects.get_or_create(email=email)
            contacts.append(ct)

        _t = datetime.strftime(datetime.now(), '%Y%m%d%H%M%S')
        mailingListName = '%d_%s_%s' % (group.id, title, _t)
        _ms = MailingList.objects.filter(name=mailingListName)
        if _ms:
            mailingListName = '%s_(%d)' % (mailingListName, len(_ms))

        mailingListDesc = 'Mailing list for %s' % (mailingListName)
        mailingList = MailingList.objects.create(
            name=mailingListName, description=mailingListDesc)

        # Remove contacts.
        removedContacts = mailingList.subscribers.exclude(email__in=contacts)
        mailingList.subscribers.remove(*removedContacts)

        # Add contacts.
        mailingList.subscribers.add(*contacts)
        mailingList.save()

        kwargs = {}
        if headerSender:
            kwargs.update({'header_sender': headerSender})

        if headerReply:
            kwargs.update({'header_reply': headerReply})

        # Create newsletter.
        newsletter = Newsletter.objects.create(
            title=title, subject=title, 
            content=text, mailing_list=mailingList,
            status=Newsletter.DRAFT, slug=self._getNewsletterSlug(title),
            **kwargs)

        if attachments:
            for attach in attachments:
                name = attach['name']
                content = attach['content']
                mimeType = attach['mimeType']

                _name = name.split('.')
                _name.pop()
                _name = '.'.join(_name)
                _a = Attachment.objects.create(newsletter=newsletter,
                                               title=_name,
                                               contentType=mimeType)

                dirPath = os.path.abspath('static')
                if not os.path.exists(dirPath):
                    os.makedirs(dirPath)

                _a.file_attachment.save(name, ContentFile(content))

        # if not async:
            # cloudSrv.createNewsletterInformation(newsletter=newsletter,
                                                 # sync=not async)

        newsletter.status = Newsletter.WAITING
        newsletter.save()

        return newsletter

    def getOrCreateGroup(self, app, name):
        """
        Get group, if group doesn't exist we need to create a group.

        @param name Group name.
        """
        group = app.groups.filter(name=name)
        if group:
            return group[0]
        else:
            return self.createGroup(name=name)

    def getGroupByMappingTable(self, app, content):
        """
        Get group's mapping tables.

        @param app app instance or app id.
        @param content Content.
        """
        if isinstance(app, int):
            apps = self.findGroupMappingTables(app__id=app)
        else:
            apps = self.findGroupMappingTables(app=app)

        for g in apps:
            if g.conditionType == models.GroupMappingTable.KEYWORD:
                # Keyword.
                _content = content.lower()
                if _content.find(g.condition.lower()) != -1:
                    if g.current:
                        return g.current
                    else:
                        return self.getOrCreateGroup(app, g.group)
            else:
                # Regular expression.
                pattern = re.compile(r'%s' % g.condition,
                                     re.IGNORECASE | re.UNICODE)
                result = pattern.search(content)
                if result and result.group():
                    if g.current:
                        return g.current
                    else:
                        return self.getOrCreateGroup(app, g.group)

        # Not have any matched groups.
        group = self.getOrCreateGroup(app, content)
        return group
