#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Util.py 10963 2018-04-03 06:49:42Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-04-03 14:49:42 +0800 (Tue, 03 Apr 2018) $
# $Revision: 10963 $

"""
Utilities.
"""

import os
import sys
import contextlib
import six

def extendUnique(l1, l2, strict=True, referenceKey=None, addBefore=False):
    """
    Combining two lists/tuples and removing duplicates, with or without
    removing duplicates in original list.

    @param l1 List or tuple 1.
    @param l2 List or tuple 2.
    @param strict True if result shall be without duplicates, False if only
                  duplicates in l2 shall be removed.
    @param referenceKey Reference key to insert l2.
    @param addBefore Insert l2 before reference key or not.
    @return Combinated list/tuple.
    """
    if strict:
        lookup = OrderedDict((k, True) for k in l1)
    else:
        lookup = dict((k, True) for k in l1)

    lookup2 = OrderedDict((k, True) for k in l2)
    filtered = type(l1)([key for key in list(lookup2.keys()) if key not in lookup])

    if strict:
        uniqued = list(lookup.keys())
        if type(uniqued) != type(l1):
            l1 = type(l1)(uniqued)
        else:
            l1 = uniqued
            
    if referenceKey:
        l1 = list(l1)
        l2 = list(l2)
        keyIndex = l1.index(referenceKey)
        if addBefore is True:
            l1[keyIndex:keyIndex] = filtered
            return l1
        else:
            l1[keyIndex + 1:keyIndex + 1] = filtered
            return l1
    else:      
        return l1 + filtered

def escapeUrlFragment(url):
    """
    Escape URL's fragment.

    "#!" used for AJAX, it's a generate content dynamically.
    Google use "?_escaped_fragment_=" to replace "#!" in URL, it can get the
    static HTML content, so we referred google.
    From:
    https://developers.google.com/webmasters/ajax-crawling/docs/specification

    @param url URL.
    """
    from six.moves.urllib_parse import urlparse, quote
    from Iuppiter.Encoding import utf8

    def getNewQueryStr(queryStr):

        newArgs = []
        for s in queryStr.split('&'):
            v = s.split('=')
            newArgs.append('='.join([quote(utf8(v)) for v in v]))

        return quote('&').join(newArgs)

    parseData = urlparse(url)
    if '#!' in url and parseData.fragment.startswith('!'):
        newUrl = ''
        # The hash fragment becomes part of the query parameters.
        key = parseData.fragment[1:]
        newKeys = []
#        from Iuppiter import debug
#        debug.breakpoint()
        for s in key.split('/'):
            newKeys.append(getNewQueryStr(s))

        keys = '/'.join(newKeys)

        # The hash fragment is indicated in the query parameters by preceding
        # it with _escaped_fragment_=
        # Some characters are escaped when the hash fragment becomes part of
        # the query parameters. These characters are listed below.
        queryStr = '_escaped_fragment_=%s' % keys
        # All other parts of the URL (host, port, path, existing query
        # parameters, and so on) remain unchanged.
        newUrl = '%s://%s%s' % (parseData.scheme, parseData.netloc,
                                parseData.path)
        if parseData.query:
            queryStr = '%s&%s' % (parseData.query, queryStr)
        newUrl = '%s?%s' % (newUrl, queryStr)
        url = newUrl
    return url

def path(unixPath):
    """
    Convert paths to the platform-specific separator.

    @param unixPath Path string using unix-like expression.
    @note Reference: wxPython demo.
    """
    st = os.path.join(*unixPath.split('/'))
    # HACK: on Linux, a leading / gets lost...
    if unixPath.startswith('/'):
        st = '/' + st
    return st

@contextlib.contextmanager
def cd(path):
    """
    With supported chdir.
    http://lateral.netmanagers.com.ar/weblog/posts/BB963.html

    @param path Target directory to chdir.
    """
    oldDir = os.getcwd()
    os.chdir(path)
    try:
        yield
    finally:
        os.chdir(oldDir)

import glob

def rglob(base, pattern, quote=False):
    """
    Recursive glob.

    @param base Base directory.
    @param pattern glob pattern.
    @param quote Quote the result path or not.
    @return List of matched files.
    """
    return ['"%s"' % v if quote else v
            for f in [glob.glob(os.path.join(d[0], pattern))
                      for d in os.walk(base)] if f
            for v in f]

import zipfile

# In python 2.7, we can use shutil.make_archive to create an archive file:
# https://docs.python.org/2/library/shutil.html#shutil.make_archive
def archiveZip(src, zipPath, compress=True, fullPath=False):
    """
    Archive files into a zip file.

    @param src Source file or folder.
    @param zipPath Zip file path.
    @param compress Compress or just store.
    @param fullPath True if you want to preserve the src's full path structure
                    you passed.
    """
    zipPath = os.path.abspath(zipPath)

    handle = zipfile.ZipFile(zipPath, "w",
        zipfile.ZIP_DEFLATED if compress else zipfile.ZIP_STORED)

    def archive(zip, dir, relative=""):
        for p in os.listdir(dir):
            p2 = os.path.join(dir, p)
            if p2 == zipPath: # No self include.
                continue

            if os.path.isdir(p2):
                archive(zip, p2, relative + p + "/")
            else:
                zip.write(p2, relative + p)

    try:
        if os.path.isdir(src):
            archive(handle, src,
                    src.replace('\\', '/') + '/' if fullPath else "")
        else:
            p = src.replace('\\', '/') if fullPath else os.path.split(src)[-1]
            handle.write(src, p)
    finally:
        handle.close()

def extractZip(zipPath, target):
    """
    Extract zip file to target folder.

    @param zipPath Zip file path.
    @param target Target folder.
    """
    handle = zipfile.ZipFile(zipPath, "r")

    # NOTE: There is an extractall method in zipfile handle.
    # http://docs.python.org/library/zipfile.html#zipfile.ZipFile.extractall
    try:
        # Extract to a target directory.
        for p in handle.namelist():
            if not p.endswith('/'):
                f = os.path.join(target, p)
                dir = os.path.dirname(f)
                if not os.path.exists(dir):
                    os.makedirs(dir)
                with open(f, 'wb') as fo:
                    fo.write(handle.read(p))
    finally:
        handle.close()

import subprocess
import shlex
import platform

def run(cmd, args='', stdin=None, stdout=subprocess.PIPE, stderr=None,
        shell=False, cwd=None, env=None,
        printCmd=False, printOut=False, hide=True, expectedReturnCode=0):
    """
    Run command with args.

    @param cmd Command.
    @param args Arguments string.
    @param stdin subprocess.Popen stdin argument.
    @param stdout subprocess.Popen stdout argument.
    @param stderr subprocess.Popen stderr argument.
    @param shell subprocess.Popen shell argument.
    @param cwd subprocess.Popen cwd argument.
    @param env subprocess.Popen env argument.
    @param printCmd Print full command that will run.
    @param printOut Print the stdout result of command.
    @param hide Hide running console.
    @param expectedReturnCode The expected return code of subprocess.Popen.
                              If Popen object's return code is not this value,
                              subprocess.CalledProcessError will be raised.
                              None if you don't want any check for return code.
    @return Command output.
    """
    if printCmd:
        print((cmd, args))
  
    # https://docs.python.org/2/library/subprocess.html#subprocess.Popen
    if os.name == 'nt' and shell:
        cmdlist = shlex.split(cmd) + shlex.split(args)
    else:
        cmdlist = [cmd] + shlex.split(args)
        
    startupinfo = None

    if hide:
        if os.name == 'nt':
            startupinfo = subprocess.STARTUPINFO()
            startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW

    # If application is wrapped by py2exe or run by pythonw, there is a
    # strange bug on windows platform, please see:
    # http://www.py2exe.org/index.cgi/Py2ExeSubprocessInteractions
    if hasattr(sys, 'frozen') and sys.frozen:
        if stderr is None:
            if hasattr(sys.stderr, 'fileno'):
                stderr = sys.stderr
            elif (hasattr(sys.stderr, '_file') and
                  hasattr(sys.stderr._file, 'fileno')):
                stderr = sys.stderr._file
            else:
                # Give up and point child stderr at nul.
                stderr = file('nul', 'a')
        if stdin is None:
            # Point stdin at nul.
            stdin = file('nul', 'r')

    p = subprocess.Popen(cmdlist, stdin=stdin, stdout=stdout, stderr=stderr,
                         shell=shell, cwd=cwd, env=env,
                         startupinfo=startupinfo)
    out = p.communicate()[0]
    if expectedReturnCode and expectedReturnCode != p.returncode:
        raise subprocess.CalledProcessError(p.returncode, ' '.join(cmdlist))

    if printOut:
        print(out)

    return out

from distutils.file_util import copy_file
from distutils.dir_util import mkpath

def copyTree(src, dst, before=lambda s, d: True, after=lambda s, d: None,
             ignore=lambda src, names: [n for n in names if n.startswith('.')]):
    """
    Copy tree structure from file system.
    Reference: distutils.dir_util.copy_tree

    @param src Source.
    @param dst Destination.
    @param before Preprocessor.
    @param after Postprocessor.
    @return Copied files.
    """
    names = os.listdir(src)
    if ignore:
        excludes = ignore(src, names)
    else:
        excludes = []

    mkpath(dst)
    outputs = []

    for n in names:
        if n in excludes:
            continue

        srcName = os.path.join(src, n)
        dstName = os.path.join(dst, n)

        if os.path.isdir(srcName):
            if before(srcName, dstName):
                outputs.extend(copyTree(srcName, dstName, before, after))
                after(srcName, dstName)
        else:
            if before(srcName, dstName):
                copy_file(srcName, dstName, before, after)
                after(srcName, dstName)
                outputs.append(dstName)

    return outputs

def classForName(qulifiedName):
    """
    Get class or module by given name.

    @param qulifiedName Qualified class name (xx.zzz.yy).
    @return Class or module.
    @note If any error occurs, it will raise ImportError exception.
    """
    if not isinstance(qulifiedName, str):
        qulifiedName = str(qulifiedName)

    if qulifiedName.find('.') != -1:
        modules = qulifiedName.split(".")
        module = __import__(".".join(modules[:-1]), globals(), locals(),
                            [modules[-1]])
        try:
            return getattr(module, modules[-1])
        except:
            raise ImportError('Unable to import %s.' % qulifiedName)
    else:
        return __import__(qulifiedName)

class IsolationRef(object):
    """
    A isolation reference to any object (module, class...etc.)
    You can get attribute from it just like original object, but every
    modification is isolated in this reference (do not affect original object).
    For example:
        iwx = IsolationRef(wx)
        iwx.MessageBox          # OK.
        iwx.MessageBox = lambda: *args, **kws: None   # OK, but wx.MessageBox
                                                      # doesn't change.
        iwx.MessageBox(None, 'caption', 'message', 0) # Pass.
    """

    def __init__(self, target):
        """
        Constructor.

        @param target The reference target.
        """
        object.__setattr__(self, '_IsolationRef__target', target)
        object.__setattr__(self, '_IsolationRef__cache', {})

    def __getattribute__(self, name):
        c = object.__getattribute__(self, '_IsolationRef__cache')
        if name in c:
            return c[name]
        else:
            v = getattr(object.__getattribute__(self, '_IsolationRef__target'),
                        name)
            c[name] = v
            return v

    def __setattr__(self, name, value):
        c = object.__getattribute__(self, '_IsolationRef__cache')
        if name in c:
            c[name] = value
        else:
            # Get into cache.
            getattr(self, name)
            c[name] = value

import re

unescapeRE = re.compile("&#?\w+;")

from six.moves import html_entities

# http://effbot.org/zone/re-sub.htm#unescape-html
def unescapeML(text):
    """
    Removes HTML or XML character references and entities from a text string.

    @param text The HTML (or XML) source text.
    @return The plain text, as a UNICODE string, if necessary.
    """
    def fixup(m):
        text = m.group(0)
        if text[:2] == "&#":
            # character reference
            try:
                if text[:3] == "&#x":
                    return chr(int(text[3:-1], 16))
                else:
                    return chr(int(text[2:-1]))
            except ValueError:
                pass
        else:
            # named entity
            try:
                text = chr(html_entities.name2codepoint[text[1:-1]])
            except KeyError:
                pass
        return text # leave as is
    return unescapeRE.sub(fixup, text)

class LRU(object):
    """
    http://code.activestate.com/recipes/252524/ (r3)
    (note we moved Node class inside LRU class.)

    Implementation of a length-limited O(1) LRU queue.
    Built for and used by PyPE:

    http://pype.sourceforge.net
    Copyright 2003 Josiah Carlson.
    """
    class Node(object):
        __slots__ = ['prev', 'next', 'me']

        def __init__(self, prev, me):
            self.prev = prev
            self.me = me
            self.next = None

    def __init__(self, count, pairs=[]):
        self.count = max(count, 1)
        self.d = {}
        self.first = None
        self.last = None
        for key, value in pairs:
            self[key] = value

    def __contains__(self, obj):
        return obj in self.d

    def __getitem__(self, obj):
        a = self.d[obj].me
        self[a[0]] = a[1]
        return a[1]

    def __setitem__(self, obj, val):
        if obj in self.d:
            del self[obj]
        nobj = LRU.Node(self.last, (obj, val))
        if self.first is None:
            self.first = nobj
        if self.last:
            self.last.next = nobj
        self.last = nobj
        self.d[obj] = nobj
        if len(self.d) > self.count:
            if self.first == self.last:
                self.first = None
                self.last = None
                return
            a = self.first
            a.next.prev = None
            self.first = a.next
            a.next = None
            del self.d[a.me[0]]
            del a

    def __delitem__(self, obj):
        nobj = self.d[obj]
        if nobj.prev:
            nobj.prev.next = nobj.next
        else:
            self.first = nobj.next
        if nobj.next:
            nobj.next.prev = nobj.prev
        else:
            self.last = nobj.prev
        del self.d[obj]

    def __iter__(self):
        cur = self.first
        while cur != None:
            cur2 = cur.next
            yield cur.me[1]
            cur = cur2

    def iteritems(self):
        cur = self.first
        while cur != None:
            cur2 = cur.next
            yield cur.me
            cur = cur2

    def iterkeys(self):
        return iter(self.d)

    def itervalues(self):
        for i, j in list(self.items()):
            yield j

    def keys(self):
        return list(self.d.keys())

try:
    # Since Python 3.2.
    from functools import lru_cache as lruCache
except ImportError:
    from Iuppiter.Decorator import decorator

    def lruCache(maxsize=100):
        """
        LRU cache decorator, to wrap a function with a memoizing callable that
        saves up to the maxsize most recent calls. It can save time when an
        expensive or I/O bound function is periodically called with the same
        arguments.

        @param maxsize Max cache size.
        """
        @decorator
        def _t2(func):
            lru = LRU(maxsize)
            def _t(*args, **kws):
                v = args
                if kws:
                    v += tuple(sorted(kws.items()))

                if v in lru:
                    return lru[v]
                else:
                    r = func(*args, **kws)
                    lru[v] = r
                    return r
            return _t
        return _t2

def preferredSort(values, preferred, reverse=False, inPlace=True):
    """
    Sort an sequence by given preferred sequence.

    For example:
        preferredSort(["Apple", "Banana", "Cherry", "Peach"],
                      ["Banna", "Peach", "NOT IN"])
        => ["Banana", "Peach", "Apple", "Cherry"]

    @param values Value sequence.
    @param preferred Preferred sequence.
    @param reverse True if you want to reverse result.
    @param inPlace In place sort or not.
    @return Sorted sequence.
    """
    scores = dict([(v, j) for j, v in enumerate(preferred)])
    default = max(len(values), len(preferred))

    if inPlace:
        if reverse:
            values.reverse()
        values.sort(key=lambda v: scores.get(v, default), reverse=reverse)
        return values
    else:
        if reverse:
            values = reversed(values)
        return sorted(values, key=lambda v: scores.get(v, default),
                      reverse=reverse)

try:
    # Since Python 2.7.
    from collections import OrderedDict
    try:
        # Since Python 3.0.
        from configparser import RawConfigParser, ConfigParser, SafeConfigParser
    except ImportError:
        from ConfigParser import RawConfigParser, ConfigParser, SafeConfigParser
except ImportError:
    from UserDict import DictMixin

    # http://code.activestate.com/recipes/576693/
    # Copied from ordereddict 1.1 (http://pypi.python.org/pypi/ordereddict)
    # MIT License.

    # Please note that this class is not thread-safe, it is different with
    # collecitons.OrderedDict.
    class OrderedDict(dict, DictMixin):

        def __init__(self, *args, **kwds):
            if len(args) > 1:
                raise TypeError('expected at most 1 arguments, got %d' %
                                len(args))
            try:
                self.__end
            except AttributeError:
                self.clear()
            self.update(*args, **kwds)

        def clear(self):
            self.__end = end = []
            end += [None, end, end]       # sentinel node for doubly linked list
            self.__map = {}               # key --> [key, prev, next]
            dict.clear(self)

        def __setitem__(self, key, value):
            if key not in self:
                end = self.__end
                curr = end[1]
                curr[2] = end[1] = self.__map[key] = [key, curr, end]
            dict.__setitem__(self, key, value)

        def __delitem__(self, key):
            dict.__delitem__(self, key)
            key, prev, next = self.__map.pop(key)
            prev[2] = next
            next[1] = prev

        def __iter__(self):
            end = self.__end
            curr = end[2]
            while curr is not end:
                yield curr[0]
                curr = curr[2]

        def __reversed__(self):
            end = self.__end
            curr = end[1]
            while curr is not end:
                yield curr[0]
                curr = curr[1]

        def popitem(self, last=True):
            if not self:
                raise KeyError('dictionary is empty')
            if last:
                key = next(reversed(self))
            else:
                key = next(iter(self))
            value = self.pop(key)
            return key, value

        def __reduce__(self):
            items = [[k, self[k]] for k in self]
            tmp = self.__map, self.__end
            del self.__map, self.__end
            inst_dict = vars(self).copy()
            self.__map, self.__end = tmp
            if inst_dict:
                return (self.__class__, (items,), inst_dict)
            return self.__class__, (items,)

        def keys(self):
            return list(self)

        setdefault = DictMixin.setdefault
        update = DictMixin.update
        pop = DictMixin.pop
        values = DictMixin.values
        items = DictMixin.items
        iterkeys = DictMixin.iterkeys
        itervalues = DictMixin.itervalues
        iteritems = DictMixin.iteritems

        def __repr__(self):
            if not self:
                return '%s()' % (self.__class__.__name__,)
            return '%s(%r)' % (self.__class__.__name__, list(self.items()))

        def copy(self):
            return self.__class__(self)

        @classmethod
        def fromkeys(cls, iterable, value=None):
            d = cls()
            for key in iterable:
                d[key] = value
            return d

        def __eq__(self, other):
            if isinstance(other, OrderedDict):
                if len(self) != len(other):
                    return False
                for p, q in  zip(list(self.items()), list(other.items())):
                    if p != q:
                        return False
                return True
            return dict.__eq__(self, other)

        def __ne__(self, other):
            return not self == other

    # Backport ConfigParser with OrderedDict (not thread-safe).
    if six.PY3:
        import configparser as _ConfigParser
    else:
        import ConfigParser as _ConfigParser

    class RawConfigParser(_ConfigParser.RawConfigParser):
        def __init__(self, defaults=None, dict_type=OrderedDict):
            _ConfigParser.RawConfigParser.__init__(self, defaults, dict_type)

    class ConfigParser(_ConfigParser.ConfigParser):
        def __init__(self, defaults=None, dict_type=OrderedDict):
            _ConfigParser.ConfigParser.__init__(self, defaults, dict_type)

    class SafeConfigParser(_ConfigParser.SafeConfigParser):
        def __init__(self, defaults=None, dict_type=OrderedDict):
            _ConfigParser.SafeConfigParser.__init__(self, defaults, dict_type)

# Generic cprint and colored functions.
try:
    import colorama
    import termcolor

    colorama.init()

    cprint = termcolor.cprint
    colored = termcolor.colored
except ImportError:
    def cprint(text, *args, **kws):
        print(text)

    colored = lambda text, *args, **kws: text


def buildZipByDir(dirName, zipFileName=None,
                  additionalFiles=None, outputName=None):
    """
    Build zip that using a directory to replace a zip file.

    @param dirName Directory will be zipped.
    @param zipFileName Zip filename that will be replaced.
    @param additionalFiles A dictionary that indicates additional
                           files' names & contents.
    @return Zip file path that is built.
    """

    zipFileName = os.path.abspath(zipFileName)
    dirName = os.path.abspath(dirName)

    replacingFileNames = []
    for (dirPath, dirNames, fileNames) in os.walk(dirName):
        for name in fileNames:
            replacingFileNames.append(
                os.path.relpath(os.path.join(
                    dirPath, name), dirName).replace('\\', '/'))

    replaceDirPath = os.path.dirname(dirName)
    if not os.path.exists(replaceDirPath):
        os.mkdir(replaceDirPath)

    # generate a temp file
    if outputName:
        if not os.path.exists(outputName):
            tmpFd = open(outputName, 'w')
        else:
            tmpFd = open(outputName, 'r')
        tmpFileName = outputName
        tmpFd.close()
    else:
        import tempfile
        tmpFd, tmpFileName = tempfile.mkstemp()
        os.close(tmpFd)

    if not additionalFiles:
        additionalFiles = {}

    with zipfile.ZipFile(tmpFileName, 'w', 
            compression=zipfile.ZIP_DEFLATED) as zipOut:
        if zipFileName:
            with zipfile.ZipFile(zipFileName, 'r') as zipIn:
                zipOut.comment = zipIn.comment  # preserve the comment
                for item in zipIn.infolist():
                    if item.filename.endswith('/'):  # directory
                        continue
                    if item.filename not in replacingFileNames \
                            and item.filename not in additionalFiles:
                        zipOut.writestr(item.filename,
                                        zipIn.read(item.filename))

        for filename in replacingFileNames:
            if filename not in additionalFiles:
                zipOut.write(os.path.join(dirName, filename),
                             arcname=filename)

        if additionalFiles:
            for key, val in additionalFiles.items():
                zipOut.writestr(key, val)

    return tmpFileName
