#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 9427 2015-09-15 08:01:29Z Judy $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Judy $ (last)
# $Date: 2013-05-13 18:59:38$
# $Revision: 9427 $

from apscheduler.scheduler import Scheduler as _Scheduler

class Scheduler(object):

    def __init__(self, config={}, **options):
        """
        Constructor.

        @param config You can config your settings through a dictionary object.
                      Please see "configure" method.
        @param options keyword arguments.
        """
        self.sched = _Scheduler(gconfig=config, **options)

    def configure(self, gconfig={}, **options):
        """
        Reconfigures the scheduler with the given options. Can only be done
        when the scheduler isn't running.

        Scheduler configuration options:

        misfire_grace_time: Maximum time in seconds for the job execution to be
                            allowed to delay before it is considered a
                            misfire.(Default is 1)
        coalesce: Roll several pending executions of jobs into one.
                  (Default is False)
        daemonic: Controls whether the scheduler thread is daemonic or not.
                  (Default is True)
                  If set to False, then the scheduler must be shut down
                  explicitly when the program is about to finish, or it will
                  prevent the program from terminating.
                  If set to True, the scheduler will automatically terminate
                  with the application, but may cause an exception to be raised
                  on exit. Jobs are always executed in non-daemonic threads.
        threadpool: (built-in)Instance of a PEP 3148 compliant thread pool or
                    a dot-notation (x.y.z:varname) reference to one.
        threadpool.core_threads: Maximum number of persistent threads in the
                                 pool.(Default is 0)
        threadpool.max_threads: Maximum number of total threads in the
                                pool(Default is 20)
        threadpool.keepalive: Seconds to keep non-core worker threads waiting
                              for new tasks.(Default is 1)
        jobstores.X.class: Class of the jobstore named X
                           (specified as module.name:classname)
        jobstores.X.Y: Constructor option Y of jobstore X.

        sample:
            config = {
                'daemonic': False,
                'threadpool.max_threads': 30
            }
            dispatch = Dispatcher(config)
        """
        self.sched.configure(self, gconfig=gconfig, **options)

    def addDateJob(self, func, date, args=None, kwargs=None, **options):
        """
        Schedules a job to be completed on a specific date and time.
        """
        return self.sched.add_date_job(func, date,
                                       args=args, kwargs=kwargs, **options)

    def addIntervalJob(self, func, weeks=0, days=0, hours=0, minutes=0,
                       seconds=0, startDate=None, args=None, kwargs=None,
                       **options):
        """
        Schedules a job to be completed on specified intervals.
        """
        return self.sched.add_interval_job(
            func, weeks=weeks, days=days, hours=hours, minutes=minutes,
            seconds=seconds, start_date=startDate, args=args, kwargs=kwargs,
            **options)

    def addCronJob(self, func, year=None, month=None, day=None, week=None,
                   dayOfWeek=None, hour=None, minute=None, second=None,
                   startDate=None, args=None, kwargs=None, **options):
        """
        Schedules a job to be completed on times that match the given
        expressions.
        """
        return self.sched.add_cron_job(
            func, year=year, month=month, day=day, week=week,
            day_of_week=dayOfWeek, hour=hour, minute=minute, second=second,
            start_date=startDate, args=args, wargs=kwargs, **options)

    def addJobstore(self, jobstore, alias, quiet=False):
        """
        Adds a job store to this scheduler.

        :param jobstore: job store to be added
        :param alias: alias for the job store
        :param quiet: True to suppress scheduler thread wakeup
        :type jobstore: instance of
            :class:`~apscheduler.jobstores.base.JobStore`
        :type alias: str
        """
        self.sched.add_jobstore(self, jobstore, alias, quiet=quiet)

    def start(self):
        """
        Start scheduler.
        """
        self.sched.start()

    def stop(self, wait=True, shutdownThreadpool=True, closeJobstores=True):
        """
        Stop scheduler.
        """
        self.sched.shutdown(wait=wait, shutdown_threadpool=shutdownThreadpool,
                            close_jobstores=closeJobstores)

