#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: FunctionTest.py 9427 2015-09-15 08:01:29Z Judy $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Judy $ (last)
# $Date: 2015-09-15 16:01:29 +0800 (Tue, 15 Sep 2015) $
# $Revision: 9427 $
"""
Test all function.
"""

import unittest

from Iuppiter import Function

class EnhanceTest(unittest.TestCase):
    """
    Test case for enhance function.
    """

    def _testSignature(self, f, f1):
        """
        Test the signature of two callable is the same or not.

        @param f Callable 1.
        @param f1 Callable 2.
        """

        import inspect

        name = f1.__name__
        args, varargs, varkws, defaults = inspect.getargspec(f1)

        self.assertEqual(name, f.__name__)
        self.assertEqual(args, inspect.getargspec(f)[0])
        self.assertEqual(varargs, inspect.getargspec(f)[1])
        self.assertEqual(varkws, inspect.getargspec(f)[2])
        self.assertEqual(defaults, inspect.getargspec(f)[3])

    def testEnhance(self):
        """
        Test for enhance.
        """
        def f(i, j, k=3):
            return i + j + k

        def e1(f, i, j, k):
            return 100 + f(i, j, k)

        f1 = Function.enhance(f, e1)
        self.assertEqual(f1(1, 2), 100 + 1 + 2 + 3)
        self._testSignature(f, f1)

        def e2(f, *args, **kws):
            return 200 + f(*args, **kws)

        f1 = Function.enhance(f, e2)
        self.assertEqual(f1(1, 2, 10), 200 + 1 + 2 + 10)
        self._testSignature(f, f1)

        # Enhance twice.
        f1 = Function.enhance(Function.enhance(f, e1), e2)
        self.assertEqual(f1(1, 2, 10), 200 + (100 + 1 + 2 + 10))
        self._testSignature(f, f1)

        def em1(m, self, a, b):
            return 10 + m(self, a, b)

        def ecm1(cm, cls):
            return "emc " + cls.__name__

        class A:
            def __init__(self):
                self.z = 2

            def m(self, a, b):
                return (a * b) / self.z

            @classmethod
            def cm(cls):
                return cls

        A.m2 = Function.enhance(A.m, em1)
        self.assertEqual(A().m2(10, 2), 10 + ((10 * 2) / 2))
        self._testSignature(A.m, A.m2)

        a = A()
        f1 = Function.enhance(a.m, em1)
        self.assertEqual(f1(10, 2), 10 + ((10 * 2) / 2))
        self._testSignature(A.m, f1)

        A.cm2 = Function.enhance(A.cm, ecm1)
        self.assertEqual(A.cm2(), "emc A")
        self._testSignature(A.cm, A.cm2)

    def testEnhance2(self):
        """
        Test for enhance function for functor.
        """
        class Functor(object):
            def __init__(self, i):
                self.i = i

            def __call__(self, wrapped):
                return wrapped() + self.i

        def f():
            return 10

        c = Functor(100)

        f2 = Function.enhance(f, c)

        self.assertEqual(f2(), 100 + 10)

if __name__ == '__main__':
    unittest.main()
