#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 9644 2016-04-17 10:12:36Z Bear $
#
# Copyright (c) 2020 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Bear $ (Last)
# $Date: 2016-04-17 18:12:36 +0800 (週日, 17 四月 2016) $
# $Revision: 9644 $

import os

from Iuppiter.Util import cd, run

TREEIO_REPOSITORY_URL = "https://github.com/nuwainfo/treeio"
TREEIO_BRANCH = "portal"

def setup(settingsDir):
    """
    Setup treeio to project directory and make Yggdrasil workable.
    
    @param settingsDir Project settings directory.
    """
    import warnings
    import shutil
    
    projectDir = os.path.abspath(os.path.join(settingsDir, '..'))
    treeioDir = os.path.join(projectDir, 'treeio')
    settingsModule = os.path.join(settingsDir, 'settings.py')
    wsgiModule = os.path.join(settingsDir, 'wsgi.py')
    treeioSettings = os.path.join(treeioDir, 'settings.py')
    treeioSettingsBak = os.path.join(treeioDir, '_settings.py')
    treeioWsgi = os.path.join(treeioDir, 'wsgi.py')
    treeioWsgiBak = os.path.join(treeioDir, '_wsgi.py')
    
    import site
    import sys
    
    site.addsitedir(treeioDir)

    if treeioDir not in sys.path:
        sys.path.insert(0, treeioDir)    
        
    if not os.path.exists(settingsModule):
        raise RuntimeError('There is no settings.py inside project directory.')
        
    if 'VIRTUAL_ENV' not in os.environ:
        raise RuntimeError('You must run inside virtualenv.')        

    def update():
        def cp(src, dst):
            if os.name == 'posix': # In Linux/Unix
                run('cp -rf %s %s' % (src, dst), 
                    shell=True, printCmd=True, printOut=True)
            elif os.name == 'nt': # In Windows
                run('copy /Y %s %s' % (src, dst), 
                    shell=True, printCmd=True, printOut=True)
            else:
                shutil.copy2(src, dst)        

            if not os.path.exists(dst):
                if os.name == 'posix': # In Linux/Unix
                    os.system('cp -rf %s %s' % (src, dst))
                elif os.name == 'nt': # In Windows
                    os.system('copy /Y %s %s' % (src, dst))
                else:
                    shutil.copy(src, dst)
            
            import hashlib
            
            with open(dst, 'rb') as f:
                print('%s md5: %s' % (
                    os.path.abspath(dst), hashlib.md5(f.read()).hexdigest()))
            
        with cd(treeioDir):
        
            run('pip install --upgrade -r requirements.pip', 
                shell=True, printCmd=True, printOut=True)
            
            djangoDir = run('python -c "import django; import os; '
                            'print os.path.dirname(django.__file__)"', 
                            shell=True, printCmd=True, printOut=True)
            djangoDir = djangoDir.strip()
            
            run('python related_fields_patch.py "%s"' % djangoDir, 
                shell=True, printCmd=True, printOut=True)

            # Initialize settings.            
            shutil.move(treeioSettings, treeioSettingsBak)        
            cp(settingsModule, treeioSettings)              
            
            # Initialize wsgi.            
            shutil.move(treeioWsgi, treeioWsgiBak)        
            cp(wsgiModule, treeioWsgi)
                
            # See issue #121. (https://github.com/treeio/treeio/issues/121)
            run('python manage.py collectstatic --noinput', 
                shell=True, printCmd=True, printOut=True)            
        
        
    if os.path.exists(treeioSettings):
        # The project is initialized, update its code.
        with cd(treeioDir):
            # We must reset settings.py and wsgi.py.
            os.remove(treeioSettings)
            os.remove(treeioWsgi)
        
            run("git checkout -- .", shell=True, printCmd=True, printOut=True)
            run("git pull origin %s" % TREEIO_BRANCH, 
                shell=True, printCmd=True, printOut=True)
                
        update()
        
        return
            
    with cd(projectDir):
        clone = "git clone"
        if TREEIO_BRANCH:
            clone = "%s -b %s" % (clone, TREEIO_BRANCH)
        clone = "%s %s" % (clone, TREEIO_REPOSITORY_URL)
        
        # http://stackoverflow.com/questions/5377960/
        # whats-the-best-practice-to-git-clone-into-an-existing-folder
        # We only support this situation for Linux because of docker needs, 
        # please see: 
        # https://code.nuwainfo.com/trac/pandora/ticket/19
        # https://code.nuwainfo.com/trac/mercurius/ticket/2644
        if os.path.exists(treeioDir) and not sys.platform.startswith('win'):
            clone = '%s _treeio' % clone
            run(clone, shell=True, printCmd=True, printOut=True)
            run('mv _treeio/.git %s/.git' % treeioDir, 
                shell=True, printCmd=True, printOut=True)
            run('cp -rf _treeio/* %s/' % treeioDir, 
                shell=True, printCmd=True, printOut=True)                
            run('rm -rf _treeio', shell=True, printCmd=True, printOut=True)
        else:
            run(clone, shell=True, printCmd=True, printOut=True)

        staticDir = os.path.join(treeioDir, 'static')   
         
        if sys.platform.startswith('win'):
            run('mklink /D static "%s"' % staticDir, 
                shell=True, printCmd=True, printOut=True)
        else:
            run('ln -s "%s" static' % staticDir, 
                shell=True, printCmd=True, printOut=True)            
                    
    with cd(treeioDir):
        
        update()
                
        # python manage.py installdb
        run('python manage.py syncdb --all --noinput', 
            shell=True, printCmd=True, printOut=True)
        run('python manage.py migrate --all --fake --noinput --no-initial-data',
            shell=True, printCmd=True, printOut=True)
                
        run('python manage.py loaddata data.json', 
            shell=True, printCmd=True, printOut=True)                
        
def attachSettings(settingsDict):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    """
    from Iuno import require, warnWithout
       
    import imp
    
    # Load treeio's default settings module.
    # Propagate DEBUG to environment variable to make treeio's settings work.
    if 'DEBUG' not in os.environ:
        os.environ['DEBUG'] = str(settingsDict['DEBUG'])
        
    projectDir = os.path.abspath(os.path.dirname(settingsDict["__file__"]))
    f, filename, description = imp.find_module('_settings', [projectDir])
    _settings = imp.load_module('_settings', f, filename, description)
    
    _settingsDict = dict([(k, v) for k, v in _settings.__dict__.items() 
                         if not (k.startswith('__') and k.endswith('__'))])
    
    # We will attach our own database settings latter.
    _settingsDict.pop('DATABASES', None) 
    # This value should follow the SERVER_MODE.
    _settingsDict.pop('DEBUG', None) 
    _settingsDict.pop('TEMPLATE_DEBUG', None) 
    
    # Admins.
    _settingsDict.pop('ADMINS', None)
    _settingsDict.pop('MANAGERS', None)
    _settingsDict.pop('ALLOWED_HOSTS', None)
    
    # i18n.
    _settingsDict.pop('TIME_ZONE', None)
    _settingsDict.pop('LANGUAGE_CODE', None)
    
    # Secret key.
    _settingsDict.pop('SECRET_KEY', None)
    
    # SSL.
    if 'HTTPS' in os.environ and (
        os.environ['HTTPS'].lower() == 'on' or 
        os.environ['HTTPS'].lower() == 'yes'):
        _settingsDict['SECURE_PROXY_SSL_HEADER'] = (
            'HTTP_X_FORWARDED_PROTOCOL', 'https')
        _settingsDict['HARDTREE_SUBSCRIPTION_SSL_ENABLED'] = True
        _settingsDict['SESSION_COOKIE_SECURE'] = True
        _settingsDict['CSRF_COOKIE_SECURE'] = True
    
    # Email settings should be cleared.
    require('EMAIL_SERVER', settingsDict)
    require('EMAIL_USERNAME', settingsDict)
    require('EMAIL_PASSWORD', settingsDict)
    require('EMAIL_FROM', settingsDict)
    warnWithout('DEFAULT_SIGNATURE', settingsDict, _settingsDict)
    
    # Use email settings below as Django's email settings to avoid duplicate.
    settingsDict['DEFAULT_FROM_EMAIL'] = settingsDict['EMAIL_USERNAME']
    settingsDict['EMAIL_HOST'] = settingsDict['EMAIL_SERVER']
    settingsDict['EMAIL_HOST_USER'] = settingsDict['EMAIL_USERNAME']        
    settingsDict['EMAIL_HOST_PASSWORD'] = settingsDict['EMAIL_PASSWORD']
    
    _settingsDict.pop('EMAIL_SERVER', None)
    _settingsDict.pop('IMAP_SERVER', None)
    _settingsDict.pop('EMAIL_USERNAME', None)
    _settingsDict.pop('EMAIL_PASSWORD', None)
    _settingsDict.pop('EMAIL_FROM', None)        
    _settingsDict.pop('DEFAULT_SIGNATURE', None)       
    
    _settingsDict.pop('DEFAULT_FROM_EMAIL', None)
    _settingsDict.pop('EMAIL_HOST', None)
    _settingsDict.pop('EMAIL_HOST_USER', None)
    _settingsDict.pop('EMAIL_HOST_PASSWORD', None)
        
    settingsDict.update(_settingsDict)
    
    # Correct the sqlite3 database file path to treeio's directory.
    # This is because the current directory will be parent of treeio's 
    # directory since django's project structure change.
    defaultDb = settingsDict['DATABASES']['default']
    if (settingsDict.get('DATABASE_ENGINE', '').startswith('sqlite') or 
        defaultDb.get('ENGINE', '').startswith('sqlite')):
        if 'DATABASE_NAME' in settingsDict:
            dbfile = settingsDict['DATABASE_NAME']
        else:
            dbfile = defaultDb['NAME']
        
        # If path is specified, we won't change it.
        if os.path.sep not in dbfile:            
            dbfile = os.path.join(projectDir, dbfile)
            settingsDict['DATABASE_NAME'] = dbfile
            defaultDb['NAME'] = dbfile
            
            