#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11168 2018-07-15 02:38:28Z Lavender $
#
# Copyright (c) 2018 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-07-15 11:38:28 +0900 (週日, 15 七月 2018) $
# $Revision: 11168 $

import os
import sys
import warnings

from Iuno import warnWithout, update, require
from Iuppiter.DjangoUtil import (extendInstalledApps, 
                                 extendMiddlewareClasses, 
                                 extendTemplateContextProcessors,
                                 extendTemplateDirs, 
                                 extendDatabaseDefaults, 
                                 extendAuthenticationBackend)

from oscar import get_core_apps
from oscar import OSCAR_MAIN_TEMPLATE_DIR

from django.utils.translation import ugettext_lazy as _

CKEDITOR_CONFIGS = {
    'default': {
        'skin': 'moono',
        # 'skin': 'office2013',
        'toolbar_Basic': [
            ['Source', '-', 'Bold', 'Italic']
        ],
        'toolbar_YourCustomToolbarConfig': [
            {'name': 'document', 'items': [
                'Source', '-', 'Save', 'NewPage', 
                'Preview', 'Print', '-', 'Templates']},
            {'name': 'clipboard', 'items': [
                'Cut', 'Copy', 'Paste', 'PasteText', 
                'PasteFromWord', '-', 'Undo', 'Redo']},
            {'name': 'editing', 'items': ['Find', 'Replace', '-', 'SelectAll']},
            {'name': 'forms',
             'items': ['Form', 'Checkbox', 'Radio', 'TextField', 
                       'Textarea', 'Select', 'Button', 'ImageButton',
                       'HiddenField']},
            '/',
            {'name': 'basicstyles',
             'items': ['Bold', 'Italic', 'Underline', 'Strike', 
                       'Subscript', 'Superscript', '-', 'RemoveFormat']},
            {'name': 'paragraph',
             'items': ['NumberedList', 'BulletedList', '-', 'Outdent', 
                       'Indent', '-', 'Blockquote', 'CreateDiv', '-',
                       'JustifyLeft', 'JustifyCenter', 'JustifyRight', 
                       'JustifyBlock', '-', 'BidiLtr', 'BidiRtl',
                       'Language']},
            {'name': 'links', 'items': ['Link', 'Unlink', 'Anchor']},
            {'name': 'insert',
             'items': ['Image', 'Flash', 'Table', 'HorizontalRule', 
                       'Smiley', 'SpecialChar', 'PageBreak', 'Iframe']},
            '/',
            {'name': 'styles', 'items': [
                'Styles', 'Format', 'Font', 'FontSize']},
            {'name': 'colors', 'items': ['TextColor', 'BGColor']},
            {'name': 'tools', 'items': ['Maximize', 'ShowBlocks']},
            {'name': 'about', 'items': ['About']},
            '/',  # put this to force next toolbar on new line
            {'name': 'yourcustomtools', 'items': [
                # put the name of your editor.ui.addButton here
                'Preview',
                'Maximize',

            ]},
        ],
        'toolbar': 'YourCustomToolbarConfig',  # put selected toolbar config here
        # 'toolbarGroups': [{ 'name': 'document', 'groups': [ 'mode', 'document', 'doctools' ] }],
        # 'height': 291,
        # 'width': '100%',
        # 'filebrowserWindowHeight': 725,
        # 'filebrowserWindowWidth': 940,
        # 'toolbarCanCollapse': True,
        # 'mathJaxLib': '//cdn.mathjax.org/mathjax/2.2-latest/MathJax.js?config=TeX-AMS_HTML',
        'tabSpaces': 4,
        'extraPlugins': ','.join([
            'uploadimage', # the upload image feature
            # your extra plugins here
            'div',
            'autolink',
            'autoembed',
            'embedsemantic',
            'autogrow',
            # 'devtools',
            'widget',
            'lineutils',
            'clipboard',
            'dialog',
            'dialogui',
            'elementspath'
        ]),
    }
}

def attachSettings(settingsDict, enablePaypal=False):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    @param enablePaypal enable pay with paypal or not
    """

    #=========================================================================
    # Django plugins.

    extraApps = []
    
    if 'IUNO_SHOP_EXTRA_APPS' in settingsDict:
        extraApps = settingsDict['IUNO_SHOP_EXTRA_APPS']

    ADDING_INSTALLED_APPS = (
        'django.contrib.sites',
        'django.contrib.flatpages',
        'widget_tweaks',
        'Iuno.shop',
        'ckeditor',
        'ckeditor_uploader',
    ) + tuple(get_core_apps(extraApps))
    
    ADDING_MIDDLEWARE_CLASSES = (
        'oscar.apps.basket.middleware.BasketMiddleware',
        'django.contrib.flatpages.middleware.FlatpageFallbackMiddleware',
    )

    installed = settingsDict['INSTALLED_APPS']
    keys = [k for k in installed if k.startswith('django.contrib')]
    extendInstalledApps(settingsDict, ADDING_INSTALLED_APPS,
                        key=list(keys)[-1])

    extendMiddlewareClasses(settingsDict, ADDING_MIDDLEWARE_CLASSES,
                            key='django.middleware.security.SecurityMiddleware')
    #=========================================================================
    
    #=========================================================================
    # Email.
    emailDefaults = {
        'ACCOUNT_EMAIL_CONFIRMATION_REQUIRED' : True,
    }

    update(settingsDict, emailDefaults)
    settingsDict['OSCAR_FROM_EMAIL'] = settingsDict['DEFAULT_FROM_EMAIL']
    #=========================================================================
    
    #=========================================================================
    # authentication backends.
    authenticationBackends = [            
        'django.contrib.auth.backends.ModelBackend',
        'oscar.apps.customer.auth_backends.EmailBackend',
    ]

    authenticationBackends = tuple(authenticationBackends)
    
    extendAuthenticationBackend(settingsDict, authenticationBackends)
    #=========================================================================
    
    #=========================================================================
    #  Django templates.
    #  Context processors
    ADDING_TEMPLATES_CONTEXT_PROCESSORS = (
        'oscar.apps.search.context_processors.search_form',
        'oscar.apps.promotions.context_processors.promotions',
        'oscar.apps.checkout.context_processors.checkout',
        'oscar.apps.customer.notifications.context_processors.notifications',
        'oscar.core.context_processors.metadata',                       
    )
    
    extendTemplateContextProcessors(settingsDict, 
                                    ADDING_TEMPLATES_CONTEXT_PROCESSORS)

    # Template tags
    ADDING_TEMPLATES_DIRS = [
        os.path.join(settingsDict['BASE_DIR'], 'templates'),
        OSCAR_MAIN_TEMPLATE_DIR,
    ]
    extendTemplateDirs(settingsDict, ADDING_TEMPLATES_DIRS)
    #=========================================================================
    # Django database default values.
    ADDING_DATABASES_DEFAULTS = {
        'ATOMIC_REQUESTS': True
    }
    extendDatabaseDefaults(settingsDict, ADDING_DATABASES_DEFAULTS)
    #=========================================================================
    # Django haystack connections.
    ADDING_HAYSTACK_CONNECTIONS = {
        'HAYSTACK_CONNECTIONS' : {
            'default': {
                'ENGINE': 'haystack.backends.simple_backend.SimpleEngine',
            },
        }
    }
    update(settingsDict, ADDING_HAYSTACK_CONNECTIONS)
    #=========================================================================

    #=========================================================================
    # Static files setting for Iuno.shop
    location = lambda x: os.path.join(
        os.path.dirname(os.path.realpath(settingsDict['__file__'])), x)

    if settingsDict['DEBUG']:
        addStaticRoot = {
            'STATIC_ROOT': location('static')
        }
        update(settingsDict, addStaticRoot)
    elif settingsDict['DEBUG'] == False:
        ADD_STATICFILES_FINDERS = {
            'STATICFILES_FINDERS': [
                'django.contrib.staticfiles.finders.FileSystemFinder',
                'django.contrib.staticfiles.finders.AppDirectoriesFinder',
            ]
        }
        update(settingsDict, ADD_STATICFILES_FINDERS)

    #=========================================================================
    if 'OSCAR_DEFAULT_CURRENCY' not in settingsDict:
        settingsDict['OSCAR_DEFAULT_CURRENCY'] = 'USD'
    #=========================================================================
    # Oscar default settings 
    # __import__ 模組的用法與官方文件有落差，可能是使用時帶入參數有誤，先以目前的寫法帶過
    # see #2784
    addOscarDefaultsPackages = {}
    oscarDefaultsPackages = __import__(
            'oscar.defaults', globals(), locals(), [], -1
        ).__dict__['defaults']
    for attr in dir(oscarDefaultsPackages):
        addOscarDefaultsPackages[attr] = getattr(
            oscarDefaultsPackages, attr)
    update(settingsDict, addOscarDefaultsPackages)
    #=========================================================================

    #=========================================================================
    if enablePaypal:
        require('PAYPAL_API_USERNAME', settingsDict)
        require('PAYPAL_API_PASSWORD', settingsDict)
        require('PAYPAL_API_SIGNATURE', settingsDict)

        ADDING_PAYPAL_INSTALLED_APPS = (
            'paypal',
        )
        extendInstalledApps(settingsDict, ADDING_PAYPAL_INSTALLED_APPS,
                        key=list(keys)[-1])


        ADDING_OSCAR_SHOP_TAGLINE = {
            'OSCAR_SHOP_TAGLINE' : 'Paypal'
        }
        update(settingsDict, ADDING_OSCAR_SHOP_TAGLINE)

        ADDING_OSCAR_DASHBOARD_NAVIGATION = {
            'label': _('PayPal'),
            'icon': 'icon-globe',
            'children': [
                {
                    'label': _('Express transactions'),
                    'url_name': 'paypal-express-list',
                },
            ]
        }
        if 'OSCAR_DASHBOARD_NAVIGATION' in settingsDict:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'].append(ADDING_OSCAR_DASHBOARD_NAVIGATION)
        else:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'] = ADDING_OSCAR_DASHBOARD_NAVIGATION
    #=========================================================================
    
    #=========================================================================
    # Django ckeditor settings
    DJANGO_CKEDITOR_CONF = {
        'CKEDITOR_UPLOAD_PATH': "uploads/",
        'CKEDITOR_CONFIGS': CKEDITOR_CONFIGS,
    }
    update(settingsDict, DJANGO_CKEDITOR_CONF)
    #=========================================================================


def attachShopURLs(settings, urlpatterns):
    """
    Attach shop related urlpatterns.
    It must be called in urls.py by passing urlpatterns.
 
    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """

    from django.conf.urls import url, patterns, include
    from oscar.app import application as shop
    
    urlpatterns += patterns('',
        # the i18n is the default url settings of django oscar, but generally
        # we are not used.
        (r'^i18n/', include('django.conf.urls.i18n')),
        (r'^shop/', shop.urls),
    )

def attachPaypalURLs(settings, urlpatterns):
    """
    Attach paypal related urlpatterns.
    It must be called in urls.py by passing urlpatterns.
 
    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """
    from Iuno.shop import views

    from django.conf.urls import url, patterns, include
    from paypal.express.dashboard.app import application

    urlpatterns += patterns('',
        
        url(r'^shop/dashboard/paypal/express/', application.urls),
        url(r'^checkout/paypal/place-order/(?P<basket_id>\d+)/$',
            views.SuccessResponseView.as_view(), name='paypal-place-order'),
        url(r'^checkout/paypal/', include('paypal.express.urls')),

        url(r'^(?P<name>.*)/preview/', views.SuccessResponseView2.as_view()),
    )

# ------------------------------------------------------------------------------
import hashlib
import time
import datetime

from django.conf import settings
from django.template.loader import get_template
from django.template import Context
from pyallpay import AllPay

from Iuppiter.Encoding import utf8
from Iuno.shop.Util import Pay2GoMPGAPI

ECPAY_SANDBOX_SERVICE_URL = \
                    'https://payment-stage.ecpay.com.tw/Cashier/AioCheckOut/V2'
ECPAY_SERVICE_URL = 'https://payment.ecpay.com.tw/Cashier/AioCheckOut/V2'

ALLPAY_SANDBOX_SERVICE_URL = \
                    'http://payment-stage.allpay.com.tw/Cashier/AioCheckOut'
ALLPAY_SERVICE_URL = 'https://payment.allpay.com.tw/Cashier/AioCheckOut'

SPGATEWAY = 'spgateway'
ALLPAY = 'allpay'
ECPAY = 'ecpay'

def buildAPIForm(submission, request, apiName):
    scheme = request.scheme
    domain = request.META['HTTP_HOST']

    if not apiName in [SPGATEWAY, ALLPAY, ECPAY]:
        raise ValueError("Not pay api name: %s" % apiName)

    MERCHANT_TRADE_NO_LEN = 20

    total = submission['order_total'].excl_tax
    basket = submission['basket']
    
    # item name
    itemName = u''
    for index, line in enumerate(basket.all_lines()):
        product = line.product
        productName = product.get_title()
        itemName += utf8(productName)
        
        itemName += u' %d dollar X %d#' % (line.price_excl_tax, line.quantity)
    itemName += (u"shipping charge %d dollar X 1" % 
                 submission['shipping_charge'].excl_tax)
    
    # merchantTradeNo
    basketId = basket.id
    tradeNo = 'BASKETID%d' % basketId
    hashCode = hashlib.sha224(str(datetime.datetime.now())).hexdigest().upper()
    lenNo = (MERCHANT_TRADE_NO_LEN - len(tradeNo))
    merchantTradeNo = str(hashCode[0:lenNo]) + tradeNo
    

    # return url
    RETURN_URL = 'http://localhost:8000/%s/preview/' % apiName
    CLIENT_BACK_URL = 'http://localhost:8000/'
    PAYMENT_INFO_URL = 'http://localhost:8000/feedback'

    returnUrl = RETURN_URL.replace(
                           'localhost:8000', domain).replace('http', scheme)
    backUrl = CLIENT_BACK_URL.replace(
                        'localhost:8000', domain).replace('http', scheme)

    # build form data
    if apiName == SPGATEWAY:
        data = {
            'TimeStamp': time.time(),
            'RespondType': 'JSON',
            'Version': '1.2',
            'LangType': 'zh-tw',
            'MerchantID': settings.SPGATEWAY_API_MERCHANT_ID,
            'MerchantOrderNo': merchantTradeNo,
            'Amt': int(total),
            'ItemDesc': itemName.replace('#', '\n'),
            'LoginType': 0,
            'CREDIT': 1,
            'UNIONPAY': 1,
        }

        mpg = Pay2GoMPGAPI(
            hashKey=settings.SPGATEWAY_API_HASH_KEY,
            hashIv=settings.SPGATEWAY_API_HASH_IV,
            sandbox=settings.SPGATEWAY_SANDBOX_MODE,
        )

        payment = mpg.createPayment(data)
        context = Context({'spgateway': payment,})

        temp = get_template('shop/Payment/Spgateway.html')
        form = temp.render(context)
                                      
        return form

    if apiName in [ECPAY, ALLPAY]:
        data = {
            'TotalAmount': int(total), 
            'ChoosePayment': 'Credit', 
            'MerchantTradeNo': merchantTradeNo, 
            'ItemName': itemName,
            'TradeDesc' : settings.OSCAR_SHOP_NAME, 
            'Remark' : settings.OSCAR_SHOP_NAME,
        }

        ap = AllPay(data)

        if apiName == ECPAY:
            if settings.ECPAY_SANDBOX_MODE:
                ap.service_url = ECPAY_SANDBOX_SERVICE_URL
            else:
                ap.service_url = ECPAY_SERVICE_URL

            ap.url_dict['MerchantID'] = settings.ECPAY_API_MERCHANT_ID
            ap.HASH_KEY = settings.ECPAY_API_HASH_KEY
            ap.HASH_IV = settings.ECPAY_API_HASH_IV
        else:
            if settings.ALLPAY_SANDBOX_MODE:
                ap.service_url = ALLPAY_SANDBOX_SERVICE_URL
            else:
                ap.service_url = ALLPAY_SERVICE_URL

            ap.url_dict['MerchantID'] = settings.ALLPAY_API_MERCHANT_ID
            ap.HASH_KEY = settings.ALLPAY_API_HASH_KEY
            ap.HASH_IV = settings.ALLPAY_API_HASH_IV


        # modify return Url
        ap.url_dict['ReturnURL'] = returnUrl
        ap.url_dict['OrderResultURL'] = returnUrl
        ap.url_dict['ClientBackURL'] = backUrl

        dictUrl = ap.check_out()

        if apiName == ECPAY:
            form = ap.gen_check_out_form(dictUrl, 
                                    settings.ECPAY_AUTO_SEND_FORM)
            form = form.replace(u'allPay-Form', u'ecpayForm')
            form = form.replace(u'allPayForm', u'ecpayForm')
        else:
            form = ap.gen_check_out_form(dictUrl, 
                                    settings.ALLPAY_AUTO_SEND_FORM)
            form = form.replace(u'allPay-Form', u'allpayForm')
            form = form.replace(u'allPayForm', u'allpayForm')
        
        return form