#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 12836 2021-04-08 06:31:19Z Lavender $
#
# Copyright (c) 2013 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2021-04-08 14:31:19 +0800 (Thu, 08 Apr 2021) $
# $Revision: 12836 $

import logging
import traceback

from six.moves.urllib.parse import quote, urlencode
from six.moves.urllib.request import urlopen
from django.core.mail.backends.base import BaseEmailBackend
from django.core.mail.backends.smtp import EmailBackend as SmtpEmailBackend
from django.conf import settings

from Iuno.Util import update
from Iuppiter.Encoding import utf8, _unicode
from Iuppiter.Json import serializer
from Iuppiter.Logging import createLogger
from Iuppiter.DjangoUtil import extendInstalledApps
    
logger = createLogger(__name__, 'Newsletter.log')

def attachSettings(settingsLocals, backends, preferred, config):
    """
    Attach 'newsletter' related settings to settings.py.

    @param settingsLocals locals() in settings.py.
    @param backends Backend instances.
    @param preferred Preferred backends.
    @param config Configuration settings.
    """
    
    extendInstalledApps(settingsLocals, (
        'Iuno.cloud.newsletter',
    ))
    
    DEFAULT_BOINCED_EMAIL_SUBJECTS = [
        "Undelivered Mail Returned to Sender",
        "Delivery Status Notification (Failure)"
    ]
    
    DEFAULT_SETTINGS = {
        'IUNO_CLOUD_NEWSLETTER_BOINCED_EMAIL_SUBJECTS': DEFAULT_BOINCED_EMAIL_SUBJECTS,
        'IUNO_CLOUD_NEWSLETTER_BOINCED_EMAIL_SEARCH_DAYS': 1,
    }
    
    update(settingsLocals, DEFAULT_SETTINGS)

class EmailBackend(BaseEmailBackend):

    def __init__(self, host=None, port=None, username=None, password=None,
                 use_tls=None, fail_silently=False, **kwargs):
        """
        Constructor.
        """
        super(EmailBackend, self).__init__(fail_silently=fail_silently)

        if not username:
            try:
                username = settings.IUNO_CLOUD_NEWSLETTER_EMAIL_HOST_USER 
                password = settings.IUNO_CLOUD_NEWSLETTER_EMAIL_HOST_PASSWORD
                host = settings.IUNO_CLOUD_NEWSLETTER_EMAIL_HOST
                use_tls = settings.IUNO_CLOUD_NEWSLETTER_EMAIL_USE_TLS
                port = settings.IUNO_CLOUD_NEWSLETTER_EMAIL_PORT

                self.useSmtp = False
            except Exception as e:
                username = ''
                password = ''
                host = ''
                use_tls = False
                port = 443
                self.useSmtp = True

        # username input:
        # admin$templates_shop$templates_shop_mails
        if '$' in username:
            appInfo = username.split('$')
            self.username = appInfo[0]
            self.appName = appInfo[1]
            if len(appInfo) > 2:
                self.groupName = appInfo[2]
            else:
                self.groupName = ''
        else:
            self.username = username
            
        self.password = password

        if settings.DEFAULT_FROM_EMAIL:
            self.from_email = settings.DEFAULT_FROM_EMAIL

        if host is None:
            host = 'localhost'

        _port = ':%s' % str(port) if port else ''
        _domain = '%s%s/newsletters/api/' % (host, _port)

        if use_tls is True:
            _scheme = 'https'
        else:
            _scheme = 'http'

        self.domain = '%s://%s' % (_scheme, _domain)

        self.kwargs = kwargs

    def parseResult(self, content):
        """
        Parse content.

        @param content Content.
        """
        try:
            content = serializer.deserialize(content)
            if 'error' in content:
                raise RuntimeError(
                    '[parseResult]:\n'
                    '%s: %s' % (content['error'], content['message']))
            else:
                return content
        except Exception as e:
            raise

    def _getToken(self, app):
        """
        Get token.

        @param app App name.
        """

        try:
            data = urlencode({
                'user': self.username,
                'pwd': self.password,
                'app': utf8(app)
            })
            link = '%sapp/login/' % self.domain
            result = urlopen(link, data=data.encode()).read()
            result = self.parseResult(result.decode()[1:-2])
            return result['token']
        except Exception as e:
            raise

    def send_messages(self, email_messages):
        """
        Send messages.

        @param email_messages Email messages.
        """
        # filter black contact 
        from Iuno.Email import filterBlacklist
        email_messages = filterBlacklist(email_messages)
        
        try:
            if self.useSmtp:
                raise RuntimeError(
                    "The project should set IUNO_CLOUD_NEWSLETTER_XXX"
                    " in settings.py. see https://code.nuwainfo.com/"
                    "trac/Mercurius/ticket/3123#comment:3")

            if not email_messages:
                return

            for email in email_messages:
                app = email.extra_headers.get('app', '')
                if not app:
                    app = self.appName
                token = self._getToken(app)
                title = utf8(email.subject)
                
                if '$' in email.from_email and not '@' in email.from_email:
                    fromMail = self.from_email
                else:
                    fromMail = email.from_email

                if email.content_subtype == 'html':
                    emailText = email.body
                else:
                    if hasattr(email, 'alternatives') and email.alternatives:
                        # if email is multipart/alternative email
                        # [(u'\n<p xmlns="http:/...eam</p>\n', 'text/html')]
                        # see https://docs.djangoproject.com/en/1.11/topics/
                        #     email/#send-mail
                        emailText = email.alternatives[0][0]
                    else:
                        emailText = '<pre>%s</pre>' % (email.body, )

                data = {
                    'token': token,
                    'title': title,
                    'emails': ','.join(email.to),
                    'text': utf8(emailText),
                    'sender': fromMail,
                }

                replyTo = email.extra_headers.get('Reply-To', '')
                if replyTo:
                    data['reply'] = replyTo

                senderEmail = email.extra_headers.get('Sender', '')
                if senderEmail:
                    data['sender'] = senderEmail

                group = email.extra_headers.get('group', '')
                if group:
                    data['group'] = utf8(group)
                else:
                    data['group'] = self.groupName if self.groupName else title

                data = urlencode(data)

                attachData = {}
                for (n, attach) in enumerate(email.attachments):
                    nameKey = 'attachmentName%d' % (n + 1)
                    contentKey = 'attachmentContent%d' % (n + 1)
                    typeKey = 'attachmentType%d' % (n + 1)

                    name, content, cType = attach

                    if name:
                        attachData.update({
                            nameKey: quote(name),
                        })

                    if cType:
                        attachData.update({
                            typeKey: quote(cType),
                        })

                    attachData.update({
                        contentKey: quote(content),
                    })
                data = '%s&%s' % (data, urlencode(attachData))

                try:
                    link = '%sapp/mail/' % self.domain
                    result = urlopen(link, data=data.encode()).read()
                    result = self.parseResult(result[1:-2])
                except Exception as e:
                    raise
        except Exception as e:
            from Iuno.Email import ConfiguredEmailBackend
            errMsg = ("ERROR: %s\n%s\nUse ConfiguredEmailBackend to send email" % 
                      (str(e), traceback.format_exc()))
            logger.error(errMsg)
            configureEmailBackend = ConfiguredEmailBackend()
            return configureEmailBackend.send_messages(email_messages)