#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11475 2019-04-21 09:31:51Z Kevin $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Kevin $ (last)
# $Date: 2019-04-21 17:31:51 +0800 (週日, 21 四月 2019) $
# $Revision: 11475 $

import datetime

from contextlib import contextmanager
from googleapiclient.errors import HttpError
from googleapiclient.http import MediaFileUpload

from Iuno.cloud.storage.gos import path
from Iuno.cloud.storage.gos.Service import (
    ServiceSingleton, BY_ID, BY_PATH, getFile, getFiles, handleHttpError)

_open = open

def listdir(path):
    path = path.replace("\\", "/")
    return [f["name"] for f in getFiles(path, type=BY_PATH)]


def mkdir(path):
    # pre-process params
    path = path.replace("\\", "/")
    names = path.split("/")

    # split path to parent & filename
    parent = getFile("/".join(names[:-1]), type=BY_PATH).get("id")
    dirName = names[-1]

    service = ServiceSingleton.getSingleton().service
    service.files().create(body={
        "name": dirName,
        "parents": [parent],
        "mimeType": "application/vnd.google-apps.folder"
    }).execute()


from Iuno.cloud.storage.gos import path as _path

def makedirs(path):
    # pre-process params
    path = path.replace("\\", "/")
    names = path.split("/")

    # create directories iteratively
    for i, name in enumerate(names):
        dst = "/".join(names[:i + 1])
        if not _path.exists(dst):
            mkdir(dst)


def _move(fileId, folderId):
    service = ServiceSingleton.getSingleton().service

    # Retrieve the existing parents to remove
    try:
        file = service.files().get(
            fileId=fileId, fields="parents").execute()
    except HttpError as e:
        handleHttpError(e)

    previousParents = ",".join(file.get("parents"))

    # Move the file to the new folder
    service.files().update(
        fileId=fileId,
        addParents=folderId,
        removeParents=previousParents).execute()


def _rename(fileId, title):
    service = ServiceSingleton.getSingleton().service

    # Rename the file.
    service.files().update(
        fileId=fileId,
        body={"name": title},
        fields="name").execute()


def rename(src, dst):
    # pre-process params
    fileId = getFile(src, type=BY_PATH).get("id")
    src = src.replace("\\", "/")
    dst = dst.replace("\\", "/")
    srcNames = src.split("/")
    dstNames = dst.split("/")

    # check if dst and src are in the same folder
    if srcNames[:-1] != dstNames[:-1]:
        _move(fileId, getFile(
            "/".join(dstNames[:-1]), type=BY_PATH)["id"])

    if srcNames[-1] != dstNames[-1]:
        _rename(fileId, dstNames[-1])


def remove(path):
    # pre-process params
    path = path.replace("\\", "/")
    fileId = getFile(path, type=BY_PATH).get("id")

    # delete the file
    try:
        service = ServiceSingleton.getSingleton().service
        service.files().delete(fileId=fileId).execute()
    except HttpError as e:
        handleHttpError(e)


def mkfile(path):
    # pre-process params
    path = path.replace("\\", "/")
    names = path.split("/")

    # split path to parent & filename
    parent = getFile("/".join(names[:-1]), type=BY_PATH).get("id")
    file = names[-1]

    service = ServiceSingleton.getSingleton().service
    service.files().create(body={
        "name": file,
        "parents": [parent],
        "mimeType": "text/plain"
    }).execute()


def copy(oldPath, newPath):
    # pre-process params
    oldPath = oldPath.replace("\\", "/")
    newPath = newPath.replace("\\", "/")
    newPathNames = newPath.split('/')

    fileId = getFile(oldPath, type=BY_PATH).get("id")
    parentId = getFile('/'.join(newPathNames[:-1]), type=BY_PATH).get("id")

    service = ServiceSingleton.getSingleton().service
    service.files().copy(fileId=fileId,
                         body={"parents": [parentId]}).execute()


class File:
    def __init__(self, filename, mode):
        self.filename = filename.replace("\\", "/")
        self.mode = mode
        self.service = ServiceSingleton.getSingleton().service
        self.fileId = getFile(self.filename, type=BY_PATH).get("id")

    def read(self):
        content = self.service.files().get_media(fileId=self.fileId).execute()

        if self.mode == 'rb':
            return content
        elif self.mode == 'r':
            return content.decode()
        else:
            return None

    def readlines(self):
        content = self.read()
        return content.split('n')

    def write(self, s):
        if self.mode == 'w':
            tmpName = 'tmp' + str(datetime.datetime.now().microsecond)

            with _open(tmpName, 'w') as f:
                f.write(s)

            file = self.service.files().get(fileId=self.fileId).execute()

            mediaBody = MediaFileUpload(tmpName, mimetype=file['mimeType'])

            self.service.files().update(
                fileId=self.fileId,
                media_body=mediaBody).execute()
        else:
            return None

    def writelines(self, lines):
        self.write('\n'.join(lines))


@contextmanager
def open(file, mode='r', buffering=None, encoding=None,
         errors=None, newline=None, closefd=True):
    yield File(file, mode)
