#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 11929 2020-02-17 04:57:08Z Trista $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Trista $
# $Date: 2020-02-17 13:57:08 +0900 (週一, 17 二月 2020) $
# $Revision: 11929 $

import datetime
import logging
import urllib
import six
import hashlib

from unidecode import unidecode

from django.shortcuts import redirect, get_object_or_404, render
from django.views.generic import View
from django.core.mail import send_mail
from django.conf import settings
from django.template.loader import get_template
from django.template import Context
from django.template.base import Template
from django.contrib import auth
from django.core.validators import validate_email
from django.core.exceptions import ValidationError
from django.template.defaultfilters import slugify
from django.urls import reverse
from django.conf import settings

from djangocms_forms.models import FormDefinition
from djangocms_forms.utils import hashid_to_int
from djangocms_forms.forms import FormBuilder

from Zephyrus.landing import ErrorCode
from Zephyrus.landing.models import Email

from Zephyrus.blog_extension.models import ActionRecord, Action
from filer.models.filemodels import File

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
logger.addHandler(logging.StreamHandler())

def sendAndRedirect(request, email, fromUrl, redirectTo, formId=None):
    errorCode = ErrorCode.NORMAL
    
    # Deal with Form Plugin
    if 'form_id' in request.POST:
        formId = hashid_to_int(request.POST.get('form_id'))
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        form = FormBuilder(landingForm, data=request.POST, files=request.FILES)
        if not form.is_valid():
            return redirect(fromUrl)
            
        order = landingForm.fields.all()
        excludedFieldTypes = ['MathCaptcha', 'ReCaptcha']
        order = [field for field in order 
                 if field.field_type not in excludedFieldTypes]
        
        dataDict = {}
        for field in order:
            key = slugify(unidecode(field.label).replace('-', '_'))
            value = form.cleaned_data.get(key, '(no input)')
            dataDict[key] = {'value': value, 'field': field,}

        if 'fileId' in request.POST:
            fileId = request.POST.get('fileId')
            targetFile = get_object_or_404(File, pk=fileId)
            action = get_object_or_404(
                Action,
                file=targetFile,
                formdefinition_ptr_id=landingForm.id
            )
            m = hashlib.md5()
            data = f"{email}{targetFile.original_filename}"
            m.update(data.encode("utf-8"))
            token = m.hexdigest()
            actionRecord, created = ActionRecord.objects.get_or_create(
                action=action,
                email=email,
                url=fromUrl,
                defaults={'token':token, 'count':0}
            )
            if not created:
                actionRecord.count = actionRecord.count + 1
            actionRecord.save()
            link = reverse('showActionFile') + '?key=' + targetFile.sha1
            fileData = {'link': link, 'token': token}

    elif formId:
        formId = hashid_to_int(formId)
        landingForm = get_object_or_404(FormDefinition, pk=formId)
        dataDict = {}
        form = True
    else:
        landingForm = None
        form = None
   
    try:
        validate_email(email)

        # Save data to db
        try:
            if not form:
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,    
                )
                email.save()
            else:
                data = {}
                for key in dataDict:
                    value = dataDict[key]['value']
                    data[key] = value
                email = Email.objects.create(
                    email=email,
                    fromUrl=fromUrl,
                    data=data,
                )
                email.save()
        except Exception as e:
            errorCode = ErrorCode.SAVE_EMAIL
            logger.error('Not successfully save email: %s' % str(e))

        # Send email
        try:
            message = (
                "Customer id : %d\nCustomer email : %s\n" % 
                (email.id, email.email))
            message += "Date : %s\n" % datetime.datetime.now()
            message += "From : %s\n" % fromUrl
            
            if not form:
                title = (
                    '[Landing %s] - customer use landing page' % 
                    request.META['HTTP_HOST'])
            else:
                title = (
                    '[Landing %s] - %s' % (
                    request.META['HTTP_HOST'], landingForm.name))
            
            if not form:
                send_mail(
                    title,
                    message,
                    settings.DEFAULT_FROM_EMAIL,
                    settings.ZEPHYRUS_LANDING_SEND_TO,
                    fail_silently=False
                )
            else:
                message += "Other Data : \n"
                
                for key in dataDict:
                    value = dataDict[key]['value']
                    field = dataDict[key]['field']
                    message += "\t%s : %s\n" % (field.label, value)
                send_mail(
                    title,
                    message,
                    settings.DEFAULT_FROM_EMAIL,
                    landingForm.email_to.split(','),
                    fail_silently=False
                )
            
            replaceStr = {
                'email': email.email,
            }
            if not form:
                mail = get_template('landing/ThankEmail.txt')
                
                title = get_template('landing/ThankEmailSubject.txt')
                
                send_mail(
                    title.render(),
                    mail.render(replaceStr),
                    settings.DEFAULT_FROM_EMAIL,
                    [email.email,],
                    fail_silently=False
                )
            else:
                if fileId:
                    replaceStr.update(fileData)
                    mail = get_template('landing/FileVerification.txt')
                    message = mail.render(replaceStr)
                else:
                    mail = Template(landingForm.post_submit_msg)
                    message = mail.render(Context(replaceStr))
                title = landingForm.email_subject
                send_mail(
                    title,
                    message,
                    settings.DEFAULT_FROM_EMAIL,
                    [email.email,],
                    fail_silently=False
                )
                
        except Exception as e:
            errorCode = ErrorCode.SEND_EMAIL
            logger.error('Not successfully send email: %s' % str(e))
    except ValidationError as e:
        errorCode = ErrorCode.EMAIL_FORMAT 
        logger.error('Email format error: %s' % str(e))

    # Reditrct
    if redirectTo:     
        if six.PY3:
            url = urllib.parse.quote(redirectTo)
        else:
            url = urllib.quote(redirectTo)
    else:
        if hasattr(settings, "ZEPHYRUS_LANDING_THANK_URL"):
            url = fromUrl + settings.ZEPHYRUS_LANDING_THANK_URL
        else:
            url = fromUrl + "thank"
        
    url = url + "?from_url=%s&error=%d" % (fromUrl, errorCode)
    
    if not form:
        return redirect(url)
    else:
        if landingForm.page_redirect:
            url = landingForm.page_redirect.get_public_url() + \
                "?from_url=%s&error=%d" % (fromUrl, errorCode)
        return redirect(url)
    
class EmailView(View):  
    def post(self, request, *args, **kwargs):
        email = request.POST.get('email')
        fromUrl = request.POST.get('fromUrl')
        redirectTo = request.POST.get('redirectTo')
        return sendAndRedirect(request, email, fromUrl, redirectTo)
            
class LandingSocialView(View):
    def get(self, request, *args, **kwargs):
        user = request.user
        email = user.email
        fromUrl = request.GET.get('fromUrl')
        redirectTo = request.GET.get('redirectTo')
        formId = request.GET.get('formId')
        auth.logout(request)
        return sendAndRedirect(
            request, email, fromUrl, redirectTo, formId=formId)



def showActionFile(request):
    key = request.GET.get('key')
    if request.GET.get('token'):
        actionRecord = get_object_or_404(ActionRecord, action__file__sha1=key, token=request.GET.get('token'))
        return redirect(settings.MEDIA_URL + str(actionRecord.action.file.file))

    else:
        return render(request, 'landing/FileVerification.html', {'key': key})
