#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: admin.py 11699 2019-09-05 08:41:06Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2019-09-05 16:41:06 +0800 (週四, 05 九月 2019) $
# $Revision: 11699 $

import os
import datetime
import hashlib
import shutil

from functools import update_wrapper
from django.contrib import admin
from django import forms
from django.conf import settings
from django.utils.html import mark_safe
from django.utils.translation import ugettext_lazy as _
from django.contrib.admin.templatetags.admin_static import static
from django.contrib.auth.signals import user_logged_in
from django.forms import Media
from django.db import transaction
from django.conf.urls import url
from django.http import HttpResponseRedirect
from django.contrib import messages
from django.core.urlresolvers import reverse

# 介面拖拉上傳

# ------------------------------------------------------------------------------
from Iuno.template_editor.models import Static, Template, Version, FOLDER, FILE
from Iuno.template_editor import BACKUP_FILE_DIR, toBytes
from Iuppiter.Logging import createLogger

TEXT_SUB_FILE_NAME = [
    'txt', 'css', 'js', 'html', 'htm',
]

IMAGE_SUB_FILE_NAME = [
    'jpg', 'png', 'jpeg', 'bmp', 'gif', 'tif',
]

logger = createLogger(__name__)

class FolderFrom(forms.ModelForm):
    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class TextFrom(forms.ModelForm):
    content = forms.CharField(
        label=_('Content'), widget=forms.Textarea, required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

class ImageFrom(forms.ModelForm):
    image = forms.ImageField(label=_('Image'), required=False)

    class Meta(object):
        model = Static
        exclude = ('parent', 'type', 'name')

def createVersion(model, node, conflict=False):
    fileName = node.name.split('.')[0]
    subFileName = node.subFileName

    fullPath = node.absolutePath
    path = node.path

    with open(fullPath, 'rb') as f:
        data = f.read()

    dirName = hashlib.md5(toBytes(path)).hexdigest()
    checksum = hashlib.md5(data).hexdigest()

    createFile = False

    if Version.objects.filter(dirName=dirName).exists():
        # conflict or not
        preVersion = node.version
        maxVersion = preVersion.number

        if subFileName:
            backupName = "%s_%d.%s" % (fileName, maxVersion + 1, subFileName)
        else:
            backupName = "%s_%d" % (fileName, maxVersion + 1)

        if os.path.exists(preVersion.absoluteBackupPath):
            if not preVersion.checksum == checksum:
                preVersionUpdateTime = datetime.datetime.fromtimestamp(
                    os.stat(preVersion.absoluteBackupPath).st_mtime)
                currentFileUpdateTime = datetime.datetime.fromtimestamp(
                    os.stat(fullPath).st_mtime)
                
                if preVersionUpdateTime < currentFileUpdateTime: 
                    # 當前檔案比版控最新版本的新時，建立新版本並判斷是否衝突
                    if not maxVersion == 0 or not conflict:
                        version = Version.objects.create(
                            dirName=dirName,
                            checksum=checksum,
                            number=maxVersion + 1,
                            fileName=backupName,
                            conflict=conflict,
                            preVersion=preVersion,
                        )
                        preVersion.nextVersion = version
                        preVersion.save()
                    else:
                        preVersion.checksum = checksum
                        preVersion.save()
                        if subFileName:
                            backupName = "%s_%d.%s" % (
                                fileName, maxVersion, subFileName)
                        else:
                            backupName = "%s_%d" % (fileName, maxVersion)

                        version = preVersion

                    createFile = True
                else:
                    # 當前檔案比版控最新版本的舊時，不建新 Version，並且換回最新版控
                    version = preVersion
                    shutil.copyfile(preVersion.absoluteBackupPath, fullPath)
            else:
                version = preVersion
        else:
            # 備分資料夾意外被刪除，自動刪除對不上的版本
            logger.warning(
                "Template Editor: %s's version backup files had be deleted. "
                "create new versions for %s" % (node.name, node.name)
            )
            
            # delete all version
            Version.objects.filter(dirName=dirName).delete()
            
            # 重新建 version
            maxVersion = 0
            
            if subFileName:
                backupName = "%s_%d.%s" % (fileName, maxVersion, subFileName)
            else:
                backupName = "%s_%d" % (fileName, maxVersion)
                
            version = Version.objects.create(
                dirName=dirName,
                checksum=checksum,
                number=maxVersion,
                fileName=backupName,
            )
            createFile = True 
    else:
        # 沒有 version，建 version
        maxVersion = 0
        if subFileName:
            backupName = "%s_%d.%s" % (fileName, maxVersion, subFileName)
        else:
            backupName = "%s_%d" % (fileName, maxVersion)

        version = Version.objects.create(
            dirName=dirName,
            checksum=checksum,
            number=maxVersion,
            fileName=backupName,
        )
        createFile = True 

    if createFile:
        dirPath = os.path.join(BACKUP_FILE_DIR, dirName)
        if not os.path.isdir(dirPath):
            os.mkdir(dirPath)

        backupPath = os.path.join(dirPath, backupName)

        with open(backupPath, 'wb') as f:
            f.write(data)

    return version

@transaction.atomic
def checkVersion(model):
    # 確保 BACKUP_FILE_DIR 存在
    if not os.path.isdir(BACKUP_FILE_DIR):
        os.mkdir(BACKUP_FILE_DIR)
        
    conflictNodeList = []
    for node in model.objects.filter(type=FILE):
        version = createVersion(model, node, conflict=True)

        path = node.path
        dirName = hashlib.md5(toBytes(path)).hexdigest()

        for version in Version.objects.filter(
            dirName=dirName, conflict=True):
            conflictNodeList.append((node, version))
            logger.info(
                "%s conflict: version %d and version %s" % (
                path, version.number, version.preVersion.number))

    return conflictNodeList

def loginCheck(sender, user, request, **kwargs):
    if user.is_superuser:
        def createMessage(ele, model):
            node = ele[0]
            version = ele[1]

            info = model._meta.app_label, model._meta.model_name
            url = reverse('admin:%s_%s_change' % info, args=[node.id,])

            msg = _("Version %(version)d "
                    "conflict Version %(preVersion)d !!: "
                    "<a href='%(url)s'>%(path)s</a>") % {
                    'version': version.number, 
                    'preVersion': version.preVersion.number,
                    'url': url,
                    'path': node.path,
                }
            return mark_safe(msg)

        try:
            conflictNodeList = checkVersion(Static)
            for ele in conflictNodeList:
                msg = createMessage(ele, Static)
                messages.warning(request, msg)

            conflictNodeList = checkVersion(Template)
            for ele in conflictNodeList:
                msg = createMessage(ele, Template)
                messages.warning(request, msg)
        except Exception as e:
            logger.error(str(e) + ", please try use build_tree and "
                "clean_version command to fix it.")

user_logged_in.connect(loginCheck)

class BaseFileAdmin(admin.ModelAdmin):
    change_list_template = 'template_editor/change_list.html'
    change_form_template = 'template_editor/change_form.html'

    def get_urls(self):
        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        info = self.model._meta.app_label, self.model._meta.model_name
        
        urlpatterns = [
            url(r'^version/(?P<id>.+)/removeConflict/$', 
                wrap(self.removeConflictView), 
                name="%s_%s_removeConflict" % info),
        ] + super(BaseFileAdmin, self).get_urls()

        return urlpatterns

    def removeConflictView(self, request, extra_context=None, **kwargs):
        version = Version.objects.get(id=kwargs['id'])
        version.conflict = False
        version.save()

        info = self.model._meta.app_label, self.model._meta.model_name
        url = reverse('admin:%s_%s_changelist' % info)
        messages.success(request, _('Conflict solved.'))
        return HttpResponseRedirect(url)


    def get_form(self, request, obj=None, **kwargs):
        if not obj:
            # add Form 新增能夠上傳檔案來新增 static 檔案
            return super(BaseFileAdmin, self).get_form(request, obj, **kwargs)

        # is folder
        if obj.type == FOLDER:
            return FolderFrom

        # is file
        versionNum = request.GET.get('version', None)
        if versionNum:
            version = obj.getVersion(versionNumber=versionNum)
        else:
            version = obj.version

        fullPath = version.absoluteBackupPath
        subFileName = obj.subFileName

        if subFileName in TEXT_SUB_FILE_NAME:
            form = TextFrom
            with open(fullPath, 'rb') as f:
                data = f.read()
            form.base_fields['content'].initial = data
        elif subFileName in IMAGE_SUB_FILE_NAME:
            form = ImageFrom
        else:
            form = FolderFrom

        return form

    def change_view(self, request, object_id, form_url='', extra_context=None):
        extra_context = {}
        obj = self.model.objects.get(id=object_id)

        versionNum = request.GET.get('version', None)
        if versionNum:
            extra_context['version'] = obj.getVersion(versionNumber=versionNum)
        else:
            extra_context['version'] = obj.version

        subFileName = obj.subFileName

        if subFileName in TEXT_SUB_FILE_NAME:
            extra_context['isImg'] = False
        elif subFileName in IMAGE_SUB_FILE_NAME:
            extra_context['isImg'] = True
        else:
            extra_context['isImg'] = False

        extra_context['moduleName'] = obj._meta.verbose_name_plural

        return super(BaseFileAdmin, self).change_view(
            request, object_id, form_url=form_url, extra_context=extra_context)

    def response_change(self, request, obj):
        versionNum = request.GET.get('version', None)
        if versionNum:
            version = obj.getVersion(versionNumber=versionNum)
        else:
            version = obj.version

        if obj.type == FILE:
            subFileName = obj.subFileName
            fullPath = obj.absolutePath

            if subFileName in TEXT_SUB_FILE_NAME:
                # 能支援上傳修改檔案
                content = request.POST.get('content')
                
                with open(fullPath, 'wb') as f:
                    data = f.write(content.encode('utf-8'))
            elif subFileName in IMAGE_SUB_FILE_NAME:
                # 圖片轉檔
                image = request.FILES.get('image')
                
                if image:
                    data = image.read()
                    image.close()

                    with open(fullPath, 'wb') as f:
                        data = f.write(data)
                elif not obj.version == version:
                    backupPath = version.absoluteBackupPath
                    with open(backupPath, 'rb') as f:
                        data = f.read()
                    
                    with open(fullPath, 'wb') as f:
                        data = f.write(data)
                else:
                    pass
            logger.info("File has modified: %s" % obj.path)
            version = createVersion(self.model, obj)

        return super(BaseFileAdmin, self).response_change(request, obj)

    def changelist_view(self, request, extra_context=None):
        conflictNodeList = checkVersion(self.model)

        roots = self.model.objects.filter(level=0)
         
        extra_context = {
            'roots': roots,
            'conflictNodeList': conflictNodeList,
        }
        
        return super(BaseFileAdmin, self).changelist_view(
            request,  extra_context=extra_context)

class StaticAdmin(BaseFileAdmin):
    pass

class TemplateAdmin(BaseFileAdmin):
    pass

admin.site.register(Static, StaticAdmin)
admin.site.register(Template, TemplateAdmin)