#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: start_app.py 11244 2018-10-10 15:19:58Z Lavender $
#
# Copyright (c) 2016 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2018-10-11 00:19:58 +0900 (週四, 11 十月 2018) $
# $Revision: 11244 $

import os
import sys
import hashlib
import platform
import six
import json
import re

import requests

from django.conf import settings
from django.core.management.base import BaseCommand, CommandError
from django.core.management import get_commands
from django.template.loader import get_template

from Iuppiter.Logging import createLogger

logger = createLogger(__name__)

def toBytes(s):
    return s if six.PY2 else s.encode()
    
HEALTHCHECK_URL = "https://healthchecks.nuwainfo.com"
HEALTHCHECK_API_KEY = getattr(settings, 'HEALTHCHECK_API_KEY', "p8w-S3Au7KBw2bRUAgtyenTziz-zdWyl")

validateCrontabTimeFormatRegex = re.compile(
    r"^{0}\s+{1}\s+{2}\s+{3}\s+{4}$".format(
        r"(?P<minute>[\d\*]{1,2}([\,\-\/][\d\*]{1,2})*)",
        r"(?P<hour>[\d\*]{1,2}([\,\-\/][\d\*]{1,2})*)",
        r"(?P<day>[\d\*]{1,2}([\,\-\/][\d\*]{1,2})*)",
        r"(?P<month>[\d\*]{1,2}([\,\-\/][\d\*]{1,2})*)",
        r"(?P<day_of_week>[0-6\*]([\,\-\/][0-6\*])*)"
    )
)

class Command(BaseCommand):
    
    help = 'Add cron job for the project.'
    
    def add_arguments(self, parser):
        parser.add_argument('command', type=str, help="Command")
        parser.add_argument('time', type=str, help="Time")
        parser.add_argument(
            '-c', '--commandScript', 
            action='store_true',
            help="if command is shellscript. This parameter must be add.")

    def handle(self, *args, **options):
        command = options['command']
        time = options['time']
        commandScript = options['commandScript']
        
        result = validateCrontabTimeFormatRegex.match(time)
        if not result:
            errorMsg = "Cronjob time syntax error: %s" % time
            logger.error(errorMsg)
            raise CommandError(errorMsg)
        
        sysStr = platform.system()
        if sysStr == "Linux":
            # healthcheck
            checkName = "[%s] %s" % (getattr(settings, "APP_NAME", "Unknown"), command)
            headers = {
                "X-Api-Key": HEALTHCHECK_API_KEY,
            }
            res = requests.get("%s/api/v1/checks/" % HEALTHCHECK_URL, headers=headers)
            if not res.status_code == 200:
                raise RuntimeError("Can't access %s: %s" % (HEALTHCHECK_URL, res.text))
            data = json.loads(res.text)
            
            if not checkName in [check.get("name") for check in data.get("checks", []) if check.get("name")]:
                postData = {
                    "name": checkName,
                    "tags": "cron",
                    "timeout": 86400, # 一天
                    "grace": 3600, # 一小時
                    "channels": "*",
                }
                res = requests.post("%s/api/v1/checks/" % HEALTHCHECK_URL, 
                    headers=headers, data=json.dumps(postData))
                if not res.status_code in [200, 201]: # OK or Created
                    raise RuntimeError("Can't access %s: %s" % (HEALTHCHECK_URL, res.text))
                data = json.loads(res.text)
                hchk = data.get("update_url").replace("%s/api/v1/checks/" % HEALTHCHECK_URL, "")
            else:
                for check in data.get("checks", []):
                    if check.get("name") == checkName:
                        hchk = check.get("update_url").replace(
                            "%s/api/v1/checks/" % HEALTHCHECK_URL, "")
                        break
                    
                
            # create shell
            if commandScript:
                hashData = hashlib.md5(
                        toBytes("%s %s" % (command, time))).hexdigest()
                        
                jobName = "%s_%s" % (command.split()[0], hashData)
            else:
                # 判斷 command
                if not command.split()[0] in get_commands():
                    raise ValueError(
                        "This command not in command list of the project: %s" % 
                        command)
                jobName = command.split()[0]
                command = "python manage.py %s" % command
                
            # 跳脫
            command = command.replace("'", "'\\''")
                
            temp = get_template("deploy/Cron.sh")
            c = {
                "command": command,
                "time": time,
                "jobName": jobName,
                "hchk": hchk,
            }
            
            
            shell = temp.render(c).replace("\r\n", "\n")
            
            with open("%s.sh" % jobName, 'w') as f:
                f.write(shell)
            
            # Crontask.sh
            response = requests.get(
                "https://github.com/nuwainfo/crontask/raw/Iuno/Crontask.sh")
            crontask = response.text
            if not os.path.exists("cron"):
                os.mkdir("cron")
            with open("cron/Crontask.sh", 'w', encoding="utf8") as f:
                f.write(crontask)
            
            # execute shell
            result = os.system("chmod +x %s.sh" % jobName)
            result = os.system("chmod +x cron/Crontask.sh")
            result = os.system("./%s.sh" % jobName)
            
            if result == 0:
                logger.info(
                    "Success set '%s' '%s' to cronjob." %(command, time))
            else:
                logger.error("Error set '%s' '%s' to cronjob." %(command, time))
                
            os.remove("%s.sh" % jobName)
        else:
            logger.info("Only support for Linux.")


            
                    