#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: cms_plugins.py 12420 2020-07-23 08:06:48Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-07-23 16:06:48 +0800 (週四, 23 七月 2020) $
# $Revision: 12420 $

import datetime

from collections import OrderedDict
from cms.models.pluginmodel import CMSPlugin
from cms.models.pagemodel import Page, TreeNode
from cms.models.titlemodels import Title
from cms.plugin_base import CMSPluginBase
from cms.plugin_pool import plugin_pool
from django.conf import settings
from django.contrib import admin
from django.utils.translation import ugettext as _
from django.template.loader import select_template
from django.template.response import SimpleTemplateResponse
from django.db.models import Q
from django.conf.urls.static import static
from djangocms_forms.models import Form, FormDefinition
from djangocms_forms.cms_plugins import FormFieldInline
from djangocms_forms.forms import FormBuilder
from djangocms_forms.utils import int_to_hashid


from aldryn_newsblog.models import Article, Category
from taggit.models import Tag

from Zephyrus.blog_extension import settings as _settings
from Zephyrus.blog_extension.models import FilterPlugin, DatePluginConfig, CallToActionModel




class FilterArticlePlugin(CMSPluginBase):
    model = FilterPlugin
    module = 'News & Blog'
    name = _("Filter Articles")
    render_template =_settings.ZEPHYRUS_BLOG_MAYLIKE_DEFAULT_TEMPLATE

    def get_fieldsets(self, request, obj=None):
        fieldsets = (
            (None, {
                'fields': (
                    'limit',
                    'filterMethod',
                    'renderTemplate',
                )
            }),
            (_('Categories'), {
                'classes': ('collapse', 'wide', 'extrapretty'),
                'fields': (
                    ('allCatagory', 'catagory'),
                )
            }),
            (_('Tag'), {
                'classes': ('collapse', 'wide', 'extrapretty'),
                'fields': (
                    ('allTag', 'tag'),
                )
            }),
        )
        return fieldsets

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            _settings.ZEPHYRUS_BLOG_MAYLIKE_TEMPLATES,
        ])

    def render(self, context, instance, placeholder):
        if instance.allCatagory:
            categoryList = []
        else:
            categoryList = []      
            if instance.catagory:
                stack = [instance.catagory, ]
                while stack:
                    category = stack.pop()
                    if not category in categoryList:
                        categoryList.append(category)

                    for child in category.get_children():
                        stack.append(child)
            else:
                categoryList.append(None)

        if instance.allTag:
            tagList = []
        else:
            tagList = []
            if instance.tag.all():
                tagList = [tag.name for tag in instance.tag.all()]
            else:
                tagList.append(None)

        if instance.filterMethod == 'or':
            query = None
            for category in categoryList:
                if query:
                    query = query | Q(categories=category)
                else:
                    query = Q(categories=category)
            for tag in tagList:
                if query:
                    if tag:
                        query = query | Q(tags__name__in=[tag,])
                    else:
                        query = query | Q(tags=None)
                else:
                    if tag:
                        query = Q(tags__name__in=[tag,])
                    else:
                        query = Q(tags=None)
        else:
            query = None
            for category in categoryList:
                if query:
                    query = query & Q(categories=category)
                else:
                    query = Q(categories=category)
            for tag in tagList:
                if query:
                    if tag:
                        query = query & Q(tags__name__in=[tag,])
                    else:
                        query = query & Q(tags=None)
                else:
                    if tag:
                        query = Q(tags__name__in=[tag,])
                    else:
                        query = Q(tags=None)

        if query:
            queryset = Article.objects.filter(query).distinct()
        else:
            queryset = Article.objects.all()
            
        if queryset.count() >= instance.limit:
            articles = queryset[:instance.limit]
        else:
            articles = queryset

        context.update({
            'articles': articles,
        })
        return context

class DatePlugin(CMSPluginBase):
    model = DatePluginConfig
    module = 'News & Blog'
    name = _("Date List")
    render_template =_settings.ZEPHYRUS_BLOG_DATE_DEFAULT_TEMPLATE

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.renderTemplate,
            _settings.ZEPHYRUS_BLOG_DATE_TEMPLATES,
        ])

    def render(self, context, instance, placeholder):
        dateDict = OrderedDict()
        for article in Article.objects.order_by("-publishing_date"):
            year = article.publishing_date.year
            month = article.publishing_date.month
            
            # only use year month, default day = 1
            date = datetime.date(year=year, month=month, day=1) 

            if date in dateDict:
                dateDict[date] += 1
            else:
                dateDict[date] = 1

        context.update({
            'dateDict': dateDict,
        })
        return context
        
        

class CallToActionForm(CMSPluginBase):
    max_plugins = 1

    def add_view(self, request, form_url='', extra_context=None):
        return SimpleTemplateResponse('blog_extension/call_to_action/OverChildPlugin.html', {
                    'num_allowed': 1,
                })
        # return super(CMSPluginMaxPluginAllowed, self).add_view(request, form_url, extra_context)


class ActionForm(CallToActionForm):
    model = FormDefinition
    render_template = _settings.CALL_TO_ACTION_FORM_DEFAULT_TEMPLATE
    require_parent = True
    parent_classes = ['CallToActionPlugin']
    max_plugins = 1
    max_num = 1
    inlines = (FormFieldInline, )

    def get_fieldsets(self, request, obj=None):
        fieldsets = (
            (None, {'fields': ('name', )}),

            (None, {
                'description': _('You can change this template whatever you want'),
                'fields': ('form_template', )
            }),
            (None, {
                'fields': ('success_redirect', ('page_redirect', 'external_redirect'), 'redirect_delay',),
            }),
            (None, {
                'description': '<strong>Submission Settings</strong> &mdash; '
                               'Choose storage options to capture form data. You can enter '
                               'an email address to have the form submissions emailed to you or '
                               'log all the form submissions to the database.',
                'fields': ('email_to', 'email_from', 'email_subject',
                           'email_uploaded_files', 'save_data', 'spam_protection', ),
            }),
        )
        return fieldsets

    def get_render_template(self, context, instance, placeholder):
        return select_template([
            instance.form_template,
            # settings.DJANGOCMS_FORMS_DEFAULT_TEMPLATE,
            # 'djangocms_forms/form_template/default.html',
            _settings.CALL_TO_ACTION_FORM_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        context = super(ActionForm, self).render(context, instance, placeholder)
        request = context['request']

        form = FormBuilder(
            initial={'referrer': request.path_info}, form_definition=instance,
            label_suffix='', auto_id='%s')

        redirect_delay = instance.redirect_delay or \
            getattr(settings, 'DJANGOCMS_FORMS_REDIRECT_DELAY', 1000)

        context.update({
            'form': form,
            'recaptcha_site_key': settings.DJANGOCMS_FORMS_RECAPTCHA_PUBLIC_KEY,
            'redirect_delay': redirect_delay,
            'actionId': instance.parent.blog_extension_calltoactionmodel.id,
            'actionFormId': int_to_hashid(instance.id),
            'externalFileLink': instance.parent.blog_extension_calltoactionmodel.linkUrl,
        })

        return context


plugin_pool.register_plugin(ActionForm)




class CallToActionPlugin(CMSPluginBase):
    model = CallToActionModel
    module = 'News & Blog'
    name = _("Call To Action")
    render_template = _settings.CALL_TO_ACTION_DEFAULT_TEMPLATE
    child_classes = ['ActionForm', ]
    allow_children = True
    max_num = 1
    # inlines = (CallToActionFormInline, )
    text_enabled = True

    def icon_src(self, instance):
        return static('images/1.jpg')

    def save_model(self, request, obj, form, change):
        super(CallToActionPlugin, self).save_model(request, obj, form, change)

        # the child plugin actionform is created
        # normally it's only possible to be 1
        # obj.num_children()
        # if obj.get_children_count() >= 1:
        if CMSPlugin.objects.filter(parent_id=obj.id).exists():
            pass
            
        # the child plugin actionform is not created
        else:
        
            # test whether the thank page is created
            
            thankNode = TreeNode.objects.filter(
                cms_pages__title_set__slug='thank',
                cms_pages__template=_settings.THANK_DEFAULT_TEMPLATE
            ).distinct()
            if thankNode.exists():
                redirectPage = thankNode.all()[0].cms_pages.all()[0]
            else:
                
                thankNode = TreeNode(site_id=settings.SITE_ID, depth=1)
                lan = obj.language
                thankNode.save()
                childPage = Page(node=thankNode, template=_settings.THANK_DEFAULT_TEMPLATE, created_by=request.user, 
                                 languages=lan)
                childPage.save()
                childTitle = Title(language=lan, title='Thank', slug='thank', page=childPage, 
                                   path='thank')
                childTitle.save()
                childPage.publish(language=lan)
                redirectPage = childPage

            
            time = datetime.datetime.now()
            now = time.strftime("%m/%d/%Y|%H:%M:%S")
            name = f'{request.path}-{request.user.username}-{now}'
            
            referencedForm = Form(name=name)
            referencedForm.save()
            formDefinition = FormDefinition(
                name=name,
                success_redirect=True,
                page_redirect=redirectPage,
                form_template=_settings.CALL_TO_ACTION_FORM_DEFAULT_TEMPLATE,
                plugin_reference=referencedForm,
                email_subject=_(_settings.CALL_TO_ACTION_EMAIL_SUBJECT),
                # r
                # cmsplugin_ptr_id=childPlugin.id
            )
            formDefinition.save()


            

            childPlugin = CMSPlugin.objects.get(id=formDefinition.cmsplugin_ptr_id)


            childPlugin.language = obj.language

            childPlugin.position=0
            childPlugin.plugin_type=u'ActionForm'
            childPlugin.placeholder=obj.placeholder
            childPlugin.parent=obj
            childPlugin.depth=obj.depth + 1
            childPlugin.path = obj.path + '0001'
            childPlugin.save()

        return


    def get_render_template(self, context, instance, placeholder):


        return select_template([
            # instance.renderTemplate,
            _settings.CALL_TO_ACTION_DEFAULT_TEMPLATE,
        ])

    def render(self, context, instance, placeholder):
        context.update({
                'instance': instance,
            })
        return context





plugin_pool.register_plugin(CallToActionPlugin)

plugin_pool.register_plugin(FilterArticlePlugin)
plugin_pool.register_plugin(DatePlugin)