#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 9742 2017-02-08 01:01:19Z David $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $ (last)
# $Date: 2017-02-08 09:01:19 +0800 (Wed, 08 Feb 2017) $
# $Revision: 9742 $

import os
import sys
import warnings

from Iuno import warnWithout, update, require
from Iuppiter.DjangoUtil import (extendInstalledApps, 
                                 extendMiddlewareClasses, 
                                 extendTemplateContextProcessors,
                                 extendTemplateDirs, 
                                 extendDatabaseDefaults, 
                                 extendAuthenticationBackend)

from oscar import get_core_apps
from oscar import OSCAR_MAIN_TEMPLATE_DIR

from django.utils.translation import ugettext_lazy as _

def attachSettings(settingsDict, enablePaypal=False):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    @param enablePaypal enable pay with paypal or not
    """

    #=========================================================================
    # Django plugins.

    ADDING_INSTALLED_APPS = (
        'django.contrib.sites',
        'django.contrib.flatpages',
        'compressor',
        'widget_tweaks',
    ) + tuple(get_core_apps())
    
    ADDING_MIDDLEWARE_CLASSES = (
        'oscar.apps.basket.middleware.BasketMiddleware',
        'django.contrib.flatpages.middleware.FlatpageFallbackMiddleware',
    )

    installed = settingsDict['INSTALLED_APPS']
    keys = [k for k in installed if k.startswith('django.contrib')]
    extendInstalledApps(settingsDict, ADDING_INSTALLED_APPS,
                        key=list(keys)[-1])

    extendMiddlewareClasses(settingsDict, ADDING_MIDDLEWARE_CLASSES,
                            key='django.middleware.security.SecurityMiddleware')
    #=========================================================================
    
    #=========================================================================
    # Email.
    emailDefaults = {
        'ACCOUNT_EMAIL_CONFIRMATION_REQUIRED' : True,
        'EMAIL_BACKEND' : 'django.core.mail.backends.smtp.EmailBackend',
    }
    
    warnWithout('EMAIL_BACKEND', settingsDict, emailDefaults)

    update(settingsDict, emailDefaults)
    settingsDict['OSCAR_FROM_EMAIL'] = settingsDict['DEFAULT_FROM_EMAIL']
    #=========================================================================
    
    #=========================================================================
    # authentication backends.
    authenticationBackends = [            
        'django.contrib.auth.backends.ModelBackend',
        'oscar.apps.customer.auth_backends.EmailBackend',
    ]

    authenticationBackends = tuple(authenticationBackends)
    
    extendAuthenticationBackend(settingsDict, authenticationBackends)
    #=========================================================================
    
    #=========================================================================
    #  Django templates.
    #  Context processors
    ADDING_TEMPLATES_CONTEXT_PROCESSORS = (
        'oscar.apps.search.context_processors.search_form',
        'oscar.apps.promotions.context_processors.promotions',
        'oscar.apps.checkout.context_processors.checkout',
        'oscar.apps.customer.notifications.context_processors.notifications',
        'oscar.core.context_processors.metadata',                       
    )
    
    extendTemplateContextProcessors(settingsDict, 
                                    ADDING_TEMPLATES_CONTEXT_PROCESSORS)

    # Template tags
    ADDING_TEMPLATES_DIRS = [
        os.path.join(settingsDict['BASE_DIR'], 'templates'),
        OSCAR_MAIN_TEMPLATE_DIR,
    ]
    extendTemplateDirs(settingsDict, ADDING_TEMPLATES_DIRS)
    #=========================================================================
    # Django database default values.
    ADDING_DATABASES_DEFAULTS = {
        'ATOMIC_REQUESTS': True
    }
    extendDatabaseDefaults(settingsDict, ADDING_DATABASES_DEFAULTS)
    #=========================================================================
    # Django haystack connections.
    ADDING_HAYSTACK_CONNECTIONS = {
        'HAYSTACK_CONNECTIONS' : {
            'default': {
                'ENGINE': 'haystack.backends.simple_backend.SimpleEngine',
            },
        }
    }
    update(settingsDict, ADDING_HAYSTACK_CONNECTIONS)
    #=========================================================================

    #=========================================================================
    # Static files setting for Iuno.shop
    location = lambda x: os.path.join(
        os.path.dirname(os.path.realpath(settingsDict['__file__'])), x)

    if settingsDict['DEBUG']:
        STATIC_ROOT = location('static')
    #=========================================================================

    #=========================================================================
    # Oscar default settings 
    addOscarDefaultsPackages = {}
    oscarDefaultsPackages = __import__(
            'oscar.defaults', globals(), locals(), [], -1
        ).__dict__['defaults']
    for attr in dir(oscarDefaultsPackages):
        addOscarDefaultsPackages[attr] = getattr(
            oscarDefaultsPackages, attr)
    update(settingsDict, addOscarDefaultsPackages)
    #=========================================================================

    #=========================================================================
    if enablePaypal:
        require('PAYPAL_API_USERNAME', settingsDict)
        require('PAYPAL_API_PASSWORD', settingsDict)
        require('PAYPAL_API_SIGNATURE', settingsDict)

        ADDING_PAYPAL_INSTALLED_APPS = (
            'paypal',
        )
        extendInstalledApps(settingsDict, ADDING_PAYPAL_INSTALLED_APPS,
                        key=list(keys)[-1])


        ADDING_OSCAR_SHOP_TAGLINE = {
            'OSCAR_SHOP_TAGLINE' : 'Paypal'
        }
        update(settingsDict, ADDING_OSCAR_SHOP_TAGLINE)

        ADDING_OSCAR_DASHBOARD_NAVIGATION = {
            'label': _('PayPal'),
            'icon': 'icon-globe',
            'children': [
                {
                    'label': _('Express transactions'),
                    'url_name': 'paypal-express-list',
                },
            ]
        }
        if 'OSCAR_DASHBOARD_NAVIGATION' in settingsDict:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'].append(ADDING_OSCAR_DASHBOARD_NAVIGATION)
        else:
            settingsDict['OSCAR_DASHBOARD_NAVIGATION'] = ADDING_OSCAR_DASHBOARD_NAVIGATION
    #=========================================================================

def attachShopURLs(settings, urlpatterns):
    """
    Attach shop related urlpatterns.
    It must be called in urls.py by passing urlpatterns.
 
    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """

    from django.conf.urls import url, include, patterns
    from oscar.app import application as shop

    urlpatterns += patterns('',
        (r'^i18n/', include('django.conf.urls.i18n')),
        (r'', include(shop.urls)),

    )

def attachPaypalURLs(settings, urlpatterns):
    """
    Attach paypal related urlpatterns.
    It must be called in urls.py by passing urlpatterns.
 
    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """

    from django.conf.urls import url, include, patterns
    from paypal.express.dashboard.app import application

    urlpatterns += patterns('',
        (r'^checkout/payment-details/', include('paypal.express.urls')),
        (r'^shop/dashboard/paypal/express/', include(application.urls)),

    )