#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 14845 2024-08-12 10:03:59Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2024-08-12 18:03:59 +0800 (週一, 12 八月 2024) $
# $Revision: 14845 $

import logging
import os
from typing import List, Tuple
from .compile import generatePatchedWASM
from .RunWASM import runWASM
from .DataTypes import WasmResultStatus
from .utils.Exceptions import WasmPatchingException
from .utils import getMd5Checksum, isChecksumFileExist

logger = logging.getLogger(__name__)
DEFAULT_FUEL = 500_000_000
DEFAULT_MEMORY_SIZE = 32 * 1024 * 1024 # 32 MB
DEFAULT_FILE_SIZE_LIMIT = 10 * 1024 * 1024 # for workspace
DEFAULT_FILE_COUNT_LIMIT = 1000 # for workspace

class WasmHandler():
    def __init__(
            self,
            wasmFilePath: str,
            outputDir: str = None,
            allowedURLs: List[str] = ["*"],
            keepSource: bool = True,
            timeout: int = 180
        ):
        self.wasmFilePath = wasmFilePath
        self.outputDir = outputDir
        self.allowedURLs = allowedURLs

        # check checksum of the wasm file already exist => if exists then the wasm has been patched
        checksum = getMd5Checksum(wasmFilePath)
        workspaceDir = os.path.dirname(wasmFilePath) if outputDir is None else outputDir

        if not isChecksumFileExist(
            checksum=checksum,
            dir=workspaceDir
        ):
            isGenerated, _ = generatePatchedWASM(
                wasmFilePath=wasmFilePath,
                outputDir=workspaceDir,
                keepSource=keepSource,
                timeout=timeout
            )
            if not isGenerated:
                raise WasmPatchingException()

        self.patchedWasmFilePath = os.path.join(workspaceDir, f"{checksum}.wasm")
        self.checksum = checksum

    def run(
            self,
            workspaceDir: str,
            fuel: int = DEFAULT_FUEL,
            memorySize: int = DEFAULT_MEMORY_SIZE,
            fileSizeLimit: int = DEFAULT_FILE_SIZE_LIMIT,
            fileCountLimit: int = DEFAULT_FILE_COUNT_LIMIT,
            debug: bool = False
        ) -> Tuple[bool, str]:

        result = runWASM(
            workspaceDir=workspaceDir,
            wasmFilePath=self.patchedWasmFilePath,
            fuel=fuel,
            memorySize=memorySize,
            fileSizeLimit=fileSizeLimit,
            fileCountLimit=fileCountLimit,
            allowedURLs=self.allowedURLs,
            debug=debug
        )
        if result["status"] == WasmResultStatus.SUCCESS:
            return True, result["message"]
        return False, result["message"]

def run(
    wasmFilePath: str,
    outputDir: str,
    workspaceDir: str,
    fuel: int = DEFAULT_FUEL,
    memorySize: int = DEFAULT_MEMORY_SIZE,
    fileSizeLimit: int = DEFAULT_FILE_SIZE_LIMIT,
    fileCountLimit: int = DEFAULT_FILE_COUNT_LIMIT,
    allowedURLs: List[str] = ["*"],
    keepSource: bool = True,
    debug: bool = False
) -> Tuple[bool, str]:

    isGenerated, filePath = generatePatchedWASM(
        wasmFilePath=wasmFilePath,
        outputDir=outputDir,
        keepSource=keepSource
    )

    if not isGenerated:
        raise WasmPatchingException()

    result = runWASM(
        workspaceDir=workspaceDir,
        wasmFilePath=filePath,
        fuel=fuel,
        memorySize=memorySize,
        fileSizeLimit=fileSizeLimit,
        fileCountLimit=fileCountLimit,
        allowedURLs=allowedURLs,
        debug=debug
    )
    if result["status"] == WasmResultStatus.SUCCESS:
        return True, result["message"]
    return False, result["message"]


