#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 14849 2024-08-13 09:13:15Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2024-08-13 17:13:15 +0800 (週二, 13 八月 2024) $
# $Revision: 14849 $

import os
import logging
from typing import Tuple
from wasm_file_http.utils.Wasm2wat import generateWATByWasmFile, generateWasmByWatFile
from wasm_file_http.utils.PatchWAT import patchWatCallbacks
from wasm_file_http.utils import getMd5Checksum

logger = logging.getLogger(__name__)

currentDir = os.path.dirname(os.path.abspath(__file__))
WABT_PATH = os.path.abspath(os.path.join(currentDir, "../wabt"))

logger.debug(f"{WABT_PATH=}")

def generatePatchedWASM(
        wasmFilePath: str,
        outputDir: str = None,
        keepSource: bool = True,
        timeout: int = 180,
    ) -> Tuple[bool, str | None]:

    """
        generate the patched WASM file from wasmFilePath
        if the patched WASM file is successfully generated then return True and the file path
        wasmFilePath: abs path
        outputDir: where the patched wasm will be generated, default to dir of the wasm file path
    """

    outputFileName = f"{getMd5Checksum(wasmFilePath)}.wasm"
    isWATGenerated, watFilePath = generateWATByWasmFile(wasmFilePath=wasmFilePath, wabtPath=WABT_PATH, timeout=timeout)

    if isWATGenerated:
        logger.debug(f"{watFilePath} generated\n")
        isWatPatched, patchedWatPath = patchWatCallbacks(watFilePath)

        if isWatPatched:
            logger.debug(f"{patchedWatPath} generated\n")

            isNewWasmGenerated, generatedFilePath = generateWasmByWatFile(
                patchedWatPath,
                WABT_PATH,
                outputFileName,
                outputDir,
                timeout
            )

            if isNewWasmGenerated:
                logger.debug(f"{generatedFilePath} generated\n")

                # remove intermediate files
                intermidiateFiles = [watFilePath, patchedWatPath]

                if not keepSource:
                    intermidiateFiles.append(wasmFilePath)

                for filePath in intermidiateFiles:
                    try:
                        os.remove(filePath)
                    except Exception as e:
                        logger.exception(str(e))

                return True, generatedFilePath
    return False, None


