#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: ProxyAPITest.py 14794 2024-08-01 04:19:43Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2024-08-01 12:19:43 +0800 (週四, 01 八月 2024) $
# $Revision: 14794 $

import unittest
import json
import threading
import os
import base64
from wasm_file_http.tests.utils import getResponseFromFile, makeConcurrentProxyRequest

TESTUSER1_WORKSPACE_PATH = "tests/fixtures/workspace/testuser1"
TESTUSER2_WORKSPACE_PATH = "tests/fixtures/workspace/testuser2"

TEST_API_URL = "https://dog.ceo/api/breed/hound/list"
EXPECTED_API_RESPONSE = {
    "status_code": 200,
    "data": {
        "message": ["afghan", "basset", "blood", "english", "ibizan", "plott", "walker"],
        "status": "success"
    }
}

TEST_SECOND_API_URL = "https://dog.ceo/api/breed/poodle/list"
EXPECTED_SECOND_API_RESPONSE = {
    "status_code": 200,
    "data": {"message": ["medium","miniature","standard","toy"], "status": "success"}
}

TEST_THIRD_API_URL = "https://dog.ceo/api/breed/bulldog/list"
EXPECTED_THIRD_API_RESPONSE = {
    "status_code": 200,
    "data": {"message":["boston","english","french"],"status":"success"}
}

RANDOM_API_URL = "https://random-data-api.fermyon.app/animals/json"

class DataIntegrityTestCase(unittest.TestCase):

    maxDiff = None

    def setUp(self):
        '''
        for each test an arbitrary api is sent and response is first modified
        to let other tests to check if the http request response is correct (not the default one)
        '''
        self.getPath = f"{TESTUSER1_WORKSPACE_PATH}/requests/get.txt"
        self.postPath = f"{TESTUSER1_WORKSPACE_PATH}/requests/post.txt"
        self.responsePath = f"{TESTUSER1_WORKSPACE_PATH}/requests/response.txt"

        self.user2GetPath = f"{TESTUSER2_WORKSPACE_PATH}/requests/get.txt"
        self.user2ResponsePath = f"{TESTUSER2_WORKSPACE_PATH}/requests/response.txt"

    def testCorrectResponseOneRequest(self):
        with open(self.getPath, "w", encoding="utf8") as f:
            f.write(
                json.dumps({ "url": TEST_API_URL })
            )

        # READ response.txt
        response = getResponseFromFile(self.responsePath)

        self.assertEqual(response, EXPECTED_API_RESPONSE)

    def testCorrectResponseConcurrentRequests(self):
        '''
            make sure there is no race condition issue with concurrent requests for one user
        '''

        urls = [
            TEST_API_URL, TEST_SECOND_API_URL, TEST_THIRD_API_URL, \
            TEST_SECOND_API_URL, TEST_THIRD_API_URL, TEST_API_URL
        ]
        threads = []
        results = [None] * len(urls)

        # Create and start threads
        for i, url in enumerate(urls):
            thread = threading.Thread(
                target=makeConcurrentProxyRequest,
                args=(url, self.getPath, self.responsePath, results, i)
            )
            threads.append(thread)
            thread.start()

        # Wait for all threads to complete
        for thread in threads:
            thread.join()

        self.assertEqual(results[0], EXPECTED_API_RESPONSE)
        self.assertEqual(results[1], EXPECTED_SECOND_API_RESPONSE)
        self.assertEqual(results[2], EXPECTED_THIRD_API_RESPONSE)
        self.assertEqual(results[3], EXPECTED_SECOND_API_RESPONSE)
        self.assertEqual(results[4], EXPECTED_THIRD_API_RESPONSE)
        self.assertEqual(results[5], EXPECTED_API_RESPONSE)

    def testCorrectResponseConsecutiveRequests(self):
        # first request
        with open(self.getPath, "w", encoding="utf8") as f:
            f.write(
                json.dumps({ "url": TEST_API_URL })
            )

        response = getResponseFromFile(self.responsePath)
        self.assertEqual(response, EXPECTED_API_RESPONSE)

        # second request
        with open(self.getPath, "w", encoding="utf8") as f:
            f.write(
                json.dumps({ "url": TEST_SECOND_API_URL })
            )

        response = getResponseFromFile(self.responsePath)
        self.assertEqual(response, EXPECTED_SECOND_API_RESPONSE)

    def testCorrectResponsesMultipleUsersMultipleRequests(self):
        '''
            make sure there is no race condition issue with concurrent requests for multiple users
        '''

        requestConfigs = [
            {"getPath": self.getPath, "responsePath": self.responsePath, "url": TEST_API_URL},
            {"getPath": self.getPath, "responsePath": self.responsePath, "url": TEST_SECOND_API_URL},
            {"getPath": self.user2GetPath, "responsePath": self.user2ResponsePath, "url": TEST_API_URL},
            {"getPath": self.user2GetPath, "responsePath": self.user2ResponsePath, "url": TEST_SECOND_API_URL},
        ]

        threads = []
        results = [None] * len(requestConfigs)

        # Create and start threads
        for i, config in enumerate(requestConfigs):
            thread = threading.Thread(
                target=makeConcurrentProxyRequest,
                args=(config["url"], config["getPath"], config["responsePath"], results, i)
            )
            threads.append(thread)
            thread.start()

        # Wait for all threads to complete
        for thread in threads:
            thread.join()

        self.assertEqual(results[0], EXPECTED_API_RESPONSE)
        self.assertEqual(results[1], EXPECTED_SECOND_API_RESPONSE)
        self.assertEqual(results[2], EXPECTED_API_RESPONSE)
        self.assertEqual(results[3], EXPECTED_SECOND_API_RESPONSE)

class APIServerIntegrationTestCase(unittest.TestCase):

    '''
        make sure API server, SMTP serve are all up running before running this testcase!
    '''

    def setUp(self) -> None:
        self.taskId = 5
        self.mailboxId = 5
        self.sender = "tim.lai@nuwainfo.com"
        self.receivers = ["tim.lai@nuwainfo.com"]
        self.subject = "test subject"
        self.body = "test body"
        self.postPath = f"{TESTUSER1_WORKSPACE_PATH}/requests/post.txt"
        self.responsePath = f"{TESTUSER1_WORKSPACE_PATH}/requests/response.txt"

        API_USERNAME = "api"
        API_PASSWORD = "G7k$2vR#9tQ!w1Z@4jP*"

        b64EncodedAuth = base64.b64encode(
            (API_USERNAME + ":" + API_PASSWORD).encode("utf-8")
        ).decode("utf-8")

        self.API_HEADERS = {
            "Authorization": f"Basic {b64EncodedAuth}"
        }

    def testExecutionActionSuccess(self):
        if os.path.isfile(self.responsePath):
            os.remove(self.responsePath)

        with open(self.postPath, "w", encoding="utf8") as f:
            f.write(
                json.dumps(
                    {
                        "url": f"http://localhost:8000/api/task-configs/{self.taskId}/action/execute/",
                        "json": {
                            "mailData": {
                                "sender": self.sender,
                                "receivers": self.receivers,
                                "subject": self.subject,
                                "body": self.body,
                        }},
                        "timeout": 10,
                        "headers": self.API_HEADERS
                    }
                )
            )
        response = getResponseFromFile(self.responsePath)
        self.assertEqual(response["status_code"], 201)
