#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 14844 2024-08-12 09:59:11Z Tim $
#
# Copyright (c) 2024 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Tim $
# $Date: 2024-08-12 17:59:11 +0800 (週一, 12 八月 2024) $
# $Revision: 14844 $

import hashlib
import logging
import json
import os
from urllib.parse import urlparse
from typing import List, Optional
from wasm_file_http.DataTypes import GETConfig, POSTConfig

logger = logging.getLogger()

def getGETConfigFromRawFileContent(config: str) -> Optional[GETConfig]:
    try:
        loadedConfig = json.loads(config)
        return GETConfig(
            url=loadedConfig["url"],
            headers=loadedConfig.get("headers"),
            timeout=loadedConfig.get("timeout"),
            params=loadedConfig.get("params"),
        )
    except Exception as e:
        logger.warning(str(e), exc_info=True)
        return None

def getMd5Checksum(filePath: str) -> str:
    md5Hash = hashlib.md5()
    with open(filePath, "rb") as file:
        for byteBlock in iter(lambda: file.read(4096), b""):
            md5Hash.update(byteBlock)
    return md5Hash.hexdigest()

def isChecksumFileExist(checksum: str, dir: str) -> bool:
    checksumFilePath = os.path.join(dir, f"{checksum}.wasm")
    if os.path.exists(checksumFilePath):
        return True
    return False

def isUrlAllowed(url: str, allowedUrls: List[str]) -> bool:
    # Parse the input URL
    parsed_url = urlparse(url)
    url_host = f"{parsed_url.hostname}:{parsed_url.port}" if parsed_url.port else parsed_url.hostname
    url_path = parsed_url.path

    for pattern in allowedUrls:
        # Handle the wildcard pattern for allowedUrls
        if pattern == "*":
            return True

        # Split pattern into host and path
        if "/" in pattern:
            pattern_host, pattern_path = pattern.split("/", 1)
            pattern_path = f"/{pattern_path}"
        else:
            pattern_host, pattern_path = pattern, ""

        # Handle trailing slashes in pattern paths
        if pattern_path.endswith("*"):
            pattern_path = pattern_path.rstrip("*")

        # Check if host matches
        if pattern_host == "*":
            return True
        if url_host == pattern_host or (pattern_host.endswith('*') and url_host.startswith(pattern_host[:-1])):
            # Check if path matches
            if pattern_path == "" or url_path.startswith(pattern_path):
                return True
        if parsed_url.hostname == pattern_host:
            # Check if path matches
            if pattern_path == "" or url_path.startswith(pattern_path):
                return True

    return False
