#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 13083 2021-07-20 09:04:36Z Mint $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Mint $ (last)
# $Date: 2021-07-20 17:04:36 +0800 (週二, 20 七月 2021) $
# $Revision: 13083 $

import os
import sys
import warnings

from Iuno import warnWithout, update, require
from Iuppiter.DjangoUtil import (extendInstalledApps, extendMiddlewareClasses,
                                 extendTemplateContextProcessors,
                                 extendAuthenticationBackend, DJANGO_VERSION,
                                 patterns)


def attachSettings(settingsDict, enableFacebook=False, enableGoogle=False,
                   enableLive=False, enableYahoo=False, enableLocale=False,
                   enableTimezone=False):
    """
    Attach default settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsDict locals() in settings.py.
    @param enableFacebook enable Facebook social login or not
    @param enableGoogle enable Google social login or not
    @param enableLive enable Live social login or not
    @param enableYahoo enable Yahoo social login or not
    """
    #=========================================================================
    # Django plugins.

    ADDING_INSTALLED_APPS = (
        'account',
        'pinax_theme_bootstrap',
        'bootstrapform',
        'social_django',
        'django.contrib.sites',
    )

    if enableLocale and not enableTimezone:
        ADDING_MIDDLEWARE_CLASSES = (
            'account.middleware.LocaleMiddleware',
        )
    elif not enableLocale and enableTimezone:
        ADDING_MIDDLEWARE_CLASSES = (
            'account.middleware.TimezoneMiddleware',
        )
    elif enableLocale and enableTimezone:
        ADDING_MIDDLEWARE_CLASSES = (
            'account.middleware.LocaleMiddleware',
            'account.middleware.TimezoneMiddleware',
        )
    else:
        # 皆不要
        ADDING_MIDDLEWARE_CLASSES = ()

    installed = settingsDict['INSTALLED_APPS']
    keys = [k for k in installed if k.startswith('django.contrib')]
    extendInstalledApps(settingsDict, ADDING_INSTALLED_APPS,
                        key=list(keys)[-1])

    extendMiddlewareClasses(settingsDict, ADDING_MIDDLEWARE_CLASSES,
                            key='django.contrib.auth.middleware.AuthenticationMiddleware')
    #=========================================================================

    #=========================================================================
    # Email.
    emailDefaults = {
        'ACCOUNT_EMAIL_CONFIRMATION_REQUIRED' : True,
    }

    update(settingsDict, emailDefaults)
    #=========================================================================

    #=========================================================================
    # authentication backends.
    authenticationBackends = [
        'account.auth_backends.EmailAuthenticationBackend',
        'django.contrib.auth.backends.ModelBackend',
        'social_core.backends.open_id.OpenIdAuth',
    ]

    if enableFacebook:
        authenticationBackends.append('social_core.backends.facebook.FacebookOAuth2')

    if enableGoogle:
        authenticationBackends.append('social_core.backends.google.GoogleOAuth2')

    if enableYahoo:
        authenticationBackends.append('social_core.backends.yahoo.YahooOAuth2')

    if enableLive:
        authenticationBackends.append('social_core.backends.live.LiveOAuth2')

    authenticationBackends = tuple(authenticationBackends)

    extendAuthenticationBackend(settingsDict, authenticationBackends)

    defaultSessionSerializers = {
        'SESSION_SERIALIZER':
        'django.contrib.sessions.serializers.PickleSerializer'
    }
    update(settingsDict, defaultSessionSerializers)
    #=========================================================================

    #=========================================================================
    #  Django templates.
    if DJANGO_VERSION >= 11000:
        contextProcessorsText = "django.template.context_processors.request"
    else:
        contextProcessorsText = "django.core.context_processors.request"

    ADDING_TEMPLATES_CONTEXT_PROCESSORS = (
        "account.context_processors.account",
        contextProcessorsText,
        "pinax_theme_bootstrap.context_processors.theme",
        "social_django.context_processors.backends",
        "social_django.context_processors.login_redirect",
    )

    extendTemplateContextProcessors(settingsDict,
                                    ADDING_TEMPLATES_CONTEXT_PROCESSORS)
    #=========================================================================

    #=========================================================================
    #  social_auth pipeline.
    socialAuthPipelineDefaults = {
        'SOCIAL_AUTH_PIPELINE' : (
            'social_core.pipeline.social_auth.social_details',
            'social_core.pipeline.social_auth.social_uid',
            'social_core.pipeline.social_auth.social_user',
            'social_core.pipeline.user.get_username',
            'social_core.pipeline.social_auth.associate_by_email',
            'social_core.pipeline.user.create_user',
            'social_core.pipeline.social_auth.associate_user',
            'social_core.pipeline.social_auth.load_extra_data',
            'social_core.pipeline.user.user_details',
        )
    }

    update(settingsDict, socialAuthPipelineDefaults)
    #=========================================================================

    #=========================================================================
    #  socail_auth basic settings.
    socailAuthSettingsDefaults = {
        'SOCIAL_AUTH_UID_LENGTH' : 255,
        'SOCIAL_AUTH_ASSOCIATION_HANDLE_LENGTH' : 255,
        'SOCIAL_AUTH_NONCE_SERVER_URL_LENGTH' : 255,
        'SOCIAL_AUTH_ASSOCIATION_SERVER_URL_LENGTH' : 255,
    }

    if enableFacebook:
        socailAuthSettingsDefaults[
            'SOCIAL_AUTH_FACEBOOK_PROFILE_EXTRA_PARAMS'] = {
          'locale': 'zh-tw',
          'fields': 'id, name, first_name, last_name, email, age_range'
        }
        socailAuthSettingsDefaults['SOCIAL_AUTH_FACEBOOK_SCOPE'] = ['email']
        require('SOCIAL_AUTH_FACEBOOK_KEY', settingsDict)
        require('SOCIAL_AUTH_FACEBOOK_SECRET', settingsDict)

    if enableGoogle:
        require('SOCIAL_AUTH_GOOGLE_OAUTH2_KEY', settingsDict)
        require('SOCIAL_AUTH_GOOGLE_OAUTH2_SECRET', settingsDict)

    if enableYahoo:
        require('SOCIAL_AUTH_YAHOO_OAUTH2_KEY', settingsDict)
        require('SOCIAL_AUTH_YAHOO_OAUTH2_SECRET', settingsDict)

    if enableLive:
        socailAuthSettingsDefaults['SOCIAL_AUTH_LIVE_SCOPE'] = \
            ['wl.basic', 'wl.emails']
        require('SOCIAL_AUTH_LIVE_KEY', settingsDict)
        require('SOCIAL_AUTH_LIVE_SECRET', settingsDict)

    update(settingsDict, socailAuthSettingsDefaults)
    #=========================================================================
    # default login related value settings.
    # TODO: There are lots of default values yet to set, it's better to set them
    # all. See
    # https://docs.djangoproject.com/en/1.11/ref/settings/#login-redirect-url
    # http://django-user-accounts.readthedocs.io/
    # en/latest/settings.html#settings
    # http://python-social-auth-docs.readthedocs.io/
    # en/latest/configuration/settings.html#urls-options

    if 'LOGIN_URL' not in settingsDict:
        settingsDict['LOGIN_URL'] = '/accounts/login/'
    if 'LOGIN_REDIRECT_URL' not in settingsDict:
        settingsDict['LOGIN_REDIRECT_URL'] = '/accounts/profile/'

    if 'SOCIAL_AUTH_LOGIN_URL' not in settingsDict:
        settingsDict['SOCIAL_AUTH_LOGIN_URL'] = settingsDict['LOGIN_URL']
    if 'SOCIAL_AUTH_LOGIN_REDIRECT_URL' not in settingsDict:
        settingsDict['SOCIAL_AUTH_LOGIN_REDIRECT_URL'] = settingsDict[
                                                         'LOGIN_REDIRECT_URL']
    if 'ACCOUNT_LOGIN_URL' not in settingsDict:
        settingsDict['ACCOUNT_LOGIN_URL'] = settingsDict['LOGIN_URL']
    if 'ACCOUNT_LOGIN_REDIRECT_URL' not in settingsDict:
        settingsDict['ACCOUNT_LOGIN_REDIRECT_URL'] = settingsDict[
                                                     'LOGIN_REDIRECT_URL']

    #=========================================================================
    # django user accounts http protocal settings.
    # see details in account/models.py EmailConfirmation >> send()
    # reference: https://docs.djangoproject.com/en/1.8/ref/
    # settings/#secure-proxy-ssl-header

    if 'SECURE_PROXY_SSL_HEADER' in settingsDict:
        settingsDict['DEFAULT_HTTP_PROTOCOL'] = settingsDict[
                                                   'SECURE_PROXY_SSL_HEADER'][1]
    else:
        https = os.environ.get('HTTPS', None)
        if https == 'on':
            settingsDict['DEFAULT_HTTP_PROTOCOL'] = 'https'
            
    #=========================================================================
    if getattr(settingsDict, "IUNO_MEMBER_TRAC_ENABLE", False):
        # trac login
        tracDefaults = {
            'DEFAULT_TRAC_AUTH_URL': 'https://code.nuwainfo.com/aec359bc3cfadb7504f26461a4c06625',
            'DEFAULT_TRAC_AUTH_USERNAME': 'Deploy',
            'DEFAULT_TRAC_AUTH_PASSWORD': 'ei0W1aUeP1pcRvJoV4X8ZC2xm',
            "SET_TRAC_LOGIN_AS_STAFF": False,
            "SET_TRAC_LOGIN_AS_SUPERUSER": False,
        }
        
        settingsDict["AUTHENTICATION_BACKENDS"] = [
            'Iuno.member.Backends.TracBackend',
        ]

        update(settingsDict, tracDefaults)

def attachURLs(settings, urlpatterns):
    from django.conf.urls import include, url

    urlpatterns += (
        url(r'accounts/', include('Iuno.member.urls')),
    )
