#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Encryption.py 11385 2019-03-13 06:03:05Z Lavender $
#
# Copyright (c) 2009 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $
# $Date: 2019-03-13 14:03:05 +0800 (週三, 13 三月 2019) $
# $Revision: 11385 $
"""
Encryption utilities.
"""

from Crypto.Cipher import AES

class Encryptor(object):
    """
    Provides encryption functionalities.
    """

    def __init__(self, key):
        """
        Constructor.

        @param key Security key.
        """
        # The length of key of AES algorithm must be multiple of 16.
        if len(key) > 32:
            self.key = key[:32]
        else:
            self.key = self._complete16(key)
            
        self.mode = AES.MODE_ECB

    def _complete16(self, byte):
        """
        Complete string length to multiple of 16.

        @param s String.
        @return Completed string.
        """
        return byte + (16 - (len(byte) % 16)) * b'?'

    def encrypt(self, data):
        """
        Encrypt data.

        @param data The data which will be encrypted. It must be a string type.
        @param return Encrypted data.
        """
        key = AES.new(self.key, self.mode)
        # The length of string which will be encrypted also must be multiple of
        # 16, we stores the original length in front of string and fills '?' to
        # end of string to complete it adequately.
        # Store the length in front of string and use '%' as separator.
        ms = str(len(data)).encode() + b'%' + data
        # Complete string length.
        ms = self._complete16(ms)
        # After AES encryption, the length of encrypted string will be the same
        # as the length of compressed string.
        return key.encrypt(ms)

    def decrypt(self, encrypted):
        """
        Decrypt encrypted data to original data.

        @param encrypted Encrypted data.
        @return Original data.
        """
        key = AES.new(self.key, self.mode)
        ms = key.decrypt(encrypted)
        
        # Find the position of length separator in front of decrypted string.
        i = ms.find(b'%')
        
        # Get the original compressed string by cutting the length in front of
        # string and useless '?' in end of it.
        return ms[i + 1: i + 1 + int(ms[:i])]