#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 10932 2018-03-26 07:56:30Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last) 
# $Date: 2018-03-26 15:56:30 +0800 (週一, 26 三月 2018) $
# $Revision: 10932 $
"""
Debug facilities.
"""

def timeit(repeat=1, prompt='elapsed time:'):
    """
    Decorator to calculate execution time for function with repeat times.

    @param repeat Repeat times.
    @param prompt Prompt text to print execution time, None if no prompt.
    """
    import time
    import sys
    
    if hasattr(time, 'perf_counter'):
        measure = time.perf_counter
    elif sys.platform.startswith('win'):
        measure = time.clock
    else:
        measure = time.time

    def _t2(func):
        def _t(*args, **kws):            
            c = measure()
            for x in range(repeat):
                r = func(*args, **kws)
            t = measure() - c
            _t.elapsed = t
            if prompt:
                print(func.__name__, prompt, t, 'secs.')
            return r
        return _t
    return _t2

def cp(f1, f2, repeat=1, repeat2=1, returnResult=False, *args, **kws):
    """
    Compare performance of two function (f1 and f2) with repeat times.

    @param f1 Function 1.
    @param f2 Function 2.
    @param repeat Repeat times.
    @param repeat2 Repeat times.
    @param args Arguments.
    @param kws Keyword arguments.
    @return Boolean whether if f1 is faster than f2
    """
    tf1 = timeit(repeat, None)(f1)
    tf2 = timeit(repeat, None)(f2)

    t1, t2 = (0, 0)
    for x in range(repeat2):
        tf1(*args, **kws)
        t1 += tf1.elapsed
        tf2(*args, **kws)
        t2 += tf2.elapsed

    avgt1 = t1 / (repeat2 * 1.0)
    avgt2 = t2 / (repeat2 * 1.0)

    print((('Total time to execute %s (repeat %ld*%ld times): '
            '%f (AVG: %f)\n') %
                (f1.__name__, repeat, repeat2, t1, avgt1)))
    print((('Total time to execute %s (repeat %ld*%ld times): '
            '%f (AVG: %f)\n') %
                (f2.__name__, repeat, repeat2, t2, avgt2)))

    if avgt1 > avgt2:
        print(('%s is slower than %s about %f%%\n' %
            (f1.__name__, f2.__name__, (((avgt1 / avgt2) - 1) * 100))))

        if returnResult == True:
            return False

    elif avgt1 < avgt2:
        print(('%s is faster than %s about %f%%\n' %
            (f1.__name__, f2.__name__, (((avgt2 / avgt1) - 1) * 100))))

        if returnResult == True:
            return True

def breakpoint():
    """
    Insert a breakpoint in code by calling PyCrust.
    """
    import sys
    import inspect
    import wx.py.PyCrust
    import Iuppiter.PyCrust
 
    if sys.displayhook != Iuppiter.PyCrust.myDisplayhook:
        print('Patch sys.displayhook...')
        sys.displayhook = Iuppiter.PyCrust.myDisplayhook
 
    cur = inspect.currentframe()
    Iuppiter.PyCrust.PyCrustApp(cur.f_back).MainLoop()
    del cur

def wxInspector():
    """
    Activate the wxPython inspection tool.
    """
    import wx
    from wx.lib.inspection import InspectionTool
    if not InspectionTool().initialized:
        InspectionTool().Init()

    # Find a widget to be selected in the tree.
    # Use either the one under the cursor, if any, or this frame.
    wnd = wx.FindWindowAtPointer()
    if not wnd:
        wnd = wx.GetApp().GetTopWindow()
    InspectionTool().Show(wnd, True)

try:
    import Iuppiter.debug.views # Import it to do some hack.
except:
    pass
