#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Runner.py 12255 2020-05-21 08:25:37Z Lavender $
#
# Copyright (c) 2016 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Lavender $ (last)
# $Date: 2020-05-21 16:25:37 +0800 (週四, 21 五月 2020) $
# $Revision: 12255 $

import os
import sys
import shutil
import webbrowser
import glob
import six
import warnings

from optparse import OptionParser

from Iuppiter.DistUtil import getScript
from Iuppiter.Util import colored, cprint

class TestRunner(object):
    """
    This class can run all tests in project's test directory.
    """

    REPORT_XML = "Report.xml"
    PYLINT_CFG = "pylint.cfg"

    def __init__(self, workingDir="."):
        """
        Constructor.

        @param workingDir Working directory.
        """

        self.usage = 'runtests [options]'
        self.parser = OptionParser(self.usage)
        self.parser.add_option(
            '-c',
            '--coverage',
            dest='coverage',
            default=False,
            action='store_true',
            help='Run coverage to test code quality.',
        )
        self.parser.add_option(
            '-p',
            '--pylint',
            dest='pylint',
            default=False,
            action='store_true',
            help='Run pylint to test code quality.',
        )
        self.parser.add_option(
            '-j',
            '--jslint',
            dest='jslint',
            default=False,
            action='store_true',
            help='Run jslint to test code quality.',
        )
        self.parser.add_option(
            '-v',
            '--view',
            dest='view',
            default=False,
            action='store_true',
            help='View test report after running test cases.',
        )
        self.parser.add_option(
            '--settings',
            help=(
                'The Python path to a settings module, e.g. '
                '"myproject.settings.main". If this isn\'t provided, the '
                'DJANGO_SETTINGS_MODULE environment variable will be used.'
            ),
            default=None,
        )

        self.curdir = os.path.abspath(workingDir)
        self.here = os.path.abspath(os.path.dirname(__file__))
        self.reportLoc = os.path.join(self.here, TestRunner.REPORT_XML)
        self.pylintcfgLoc = os.path.join(self.here, TestRunner.PYLINT_CFG)
        self.path = os.path.join(self.curdir, "tests", "test-reports")
        if os.path.exists(self.path):
            shutil.rmtree(self.path)
        os.makedirs(self.path)

        self.options = None

    def enterDir(self, dest):
        """
        Change directory to destination, make directory if it doesn't exist.

        @param dest Destination directory.
        """
        if not os.path.exists(dest):
            os.makedirs(dest)
        os.chdir(dest)

    def runPylint(self):
        """
        Run pylint for working directory's project.        
        """
        print("Pylint running! please wait...")
        reportdir = os.path.join(self.path, "pylint")
        self.enterDir(reportdir)

        (parent, package) = os.path.split(self.curdir)
        os.chdir(parent)

        if self.options.view:
            format = 'html'
        else:
            format = 'parseable'

        rawReport = os.path.join(reportdir, "_pylint_raw_.txt")
        report = os.path.join(reportdir, "pylint.txt")
        pylint = getScript('pylint')
        cmd = ('%s --rcfile="%s" -f %s %s > "%s"' %
               (pylint, self.pylintcfgLoc, format, package, rawReport))
        print(cmd)
        os.system(cmd)

        print('Filter unnecessary files from _pylint_raw_.txt to => pylint.txt')

        sitePackagesKey = six.b('%s%s%s' % (os.path.sep, 'site-packages', os.path.sep))

        with open(rawReport, 'rb') as f:
            with open(report, 'wb') as f2:
                for line in f:
                    if sitePackagesKey in line:
                        continue
                    f2.write(line)

        if self.options.view:
            # Open pylint report automatically.
            os.chdir(reportdir)
            webbrowser.open_new_tab("pylint.txt")

        os.chdir(self.curdir)

    def runJslint(self):
        """
        Run jslint for working directory's project.
        """
        JAR = os.path.abspath(glob.glob(
            os.path.join(os.path.dirname(__file__), 'jslint4java-*.jar'))[-1])

        CFG = os.path.join(os.path.dirname(__file__), 'jslint.cfg')
        with open(CFG, 'r') as f:
            options = f.read()
        options = options.replace('\n', ' ').replace('\r', '')

        JSLINT4JAVA = ('-jar "%s" %s') % (JAR, options)

        print("Jslint running! please wait...")
        reportdir = os.path.join(self.path, "jslint")
        self.enterDir(reportdir)

        if self.options.view:
            format = 'plain'
            report = os.path.join(reportdir, "jslint.txt")
        else:
            format = 'xml'
            report = os.path.join(reportdir, "jslint.xml")

        from Iuppiter.Util import rglob, run

        jsFiles = rglob(self.curdir, '*.js', quote=True)
        jsFiles = [f for f in jsFiles
                   if '.min.js' not in f and 'jquery' not in f.lower()]

        if jsFiles:
            args = ('%s --report %s %s' % (
                    JSLINT4JAVA, format, ' '.join(jsFiles)))
            out = run('java', args, printCmd=True)

            with open(report, 'wb') as f:
                f.write(out)

            if self.options.view:
                webbrowser.open_new_tab(report)

        os.chdir(self.curdir)

    def fixEncoding(self):
        """
        Fix utf-8 encoding for test reports.
        Test cases might output non-utf8 characters to console, we must convert
        it into utf-8 encoding.
        """
        from Iuppiter.Util import rglob
        xmlFiles = rglob(self.path, '*.xml')

        from Iuppiter.Encoding import utf8

        for fn in xmlFiles:
            with open(fn, "rb") as f:
                content = f.read()

            content = utf8(content)

            with open(fn, 'wb') as f:
                f.write(content)

    def checkEnvironment(self):
        """
        Check required environment settings and files.

        @return True if all required is OK, otherwise False.
        """
        requiredEnv = {
            'JAVA_HOME': 'java',
            'ANT_HOME': 'ant'
        }

        for e, c in list(requiredEnv.items()):
            if not os.environ.get(e, None):                                
                print()
                print(((
                    'Environment "%s" is required, '
                    'or please make the command "%s" is runnable from '
                    'console.'
                ) % (
                    colored(e, 'yellow', attrs=['bold']),
                    colored(c, 'cyan', attrs=['bold']),
                )))
                print()
                
                # At this time, not all command are truly necessary for run 
                # tests, so we pass it just give a warning.
                # return False

        return True

    def run(self, args=None):
        """
        Run tests in working directory's test directory.

        @param args Arguments, default: sys.argv[1:].
        @return True if all tests passed.
        """
        if not self.checkEnvironment():
            sys.exit(1)

        if not args:
            args = sys.argv[1:]

        options = self.parser.parse_args(args)[0]
        self.options = options

        os.chdir(self.curdir)

        # Run tests.
        argv = ["discover",
                ".",
                "*Test.py"]

        try:
            stdout = sys.stdout

            if self.options.coverage:
                import coverage

                from Iuppiter import DistUtil
                sitePackagesDir = DistUtil.getSitePackagesDir()

                if self.curdir.lower().startswith(sitePackagesDir.lower()):
                    cov = coverage.coverage(cover_pylib=True,
                                            include=[os.path.join(
                                                        self.curdir, '*')])
                else:
                    cov = coverage.coverage()

                cov.start()
            
            import unittest
            import xmlrunner
            
            try:            
                from django.conf import settings
                import django
                django.setup()
                                
                settings.TEST_OUTPUT_DIR = self.path
                from xmlrunner.extra import djangotestrunner
                
                try:
                    s = options.get("settings")
                except Exception as e:
                    s = None
                testRunner = djangotestrunner.XMLTestRunner(
                    settings=s,
                    pythonpath=None,
                    traceback=False,
                    testrunner=None,
                    no_color=False,
                    pattern='*Test.py',
                    liveserver=None
                )
                result = testRunner.run_tests(None)
                print("In Django environment.")
            except:
                testRunner = xmlrunner.XMLTestRunner(output=self.path)
                print("Not in Django environment.")
            
                result = unittest.TestProgram(module=None,
                                              argv=sys.argv[:1] + argv,
                                              testRunner=testRunner, exit=False)

            # Fix utf-8 encoding bug for test report.
            self.fixEncoding()

            if self.options.coverage:
                cov.stop()
                reportdir = os.path.join(self.path, "coverage")

                # Create it if not exists.
                self.enterDir(reportdir)
                os.chdir(self.curdir)

                if self.options.view:
                    cov.html_report(directory=reportdir)
                else:
                    cov.xml_report(outfile=os.path.join(reportdir,
                                                        'coverage.xml'))
                cov.save()

                if self.options.view:
                    os.chdir(reportdir)
                    webbrowser.open_new_tab("index.html")
                    os.chdir(self.path)

            # Generate test reports.
            self.enterDir(self.path)

            if not os.path.exists(os.path.join(
                        self.path, TestRunner.REPORT_XML)):
                shutil.copy(self.reportLoc, self.path)

            os.system('ant -f Report.xml > log.txt')
            os.remove('log.txt')

            if self.options.view:
                # Open test report automatically.
                os.chdir(os.path.join(self.path, "tests"))
                webbrowser.open_new_tab("index.html")
                os.chdir(self.path)
        finally:
            pass

        os.chdir(self.curdir)
        sys.stdout = stdout

        print("\n")

        # Run pylint.
        if self.options.pylint:
            self.runPylint()

        # Run jslint.
        if self.options.jslint:
            self.runJslint()

        return result

if __name__ == '__main__':
    TestRunner(os.path.join("..", "..")).run()
