#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: DecoratorTest.py 11990 2020-03-20 11:19:07Z Bear $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Bear $ (last)
# $Date: 2020-03-20 19:19:07 +0800 (週五, 20 三月 2020) $
# $Revision: 11990 $
"""
Test Decorator module.
"""

import unittest

from Iuppiter import Decorator

class RemoveDecoratorTest(unittest.TestCase):
    """
    Test case for removeDecorator function.
    """

    def testRemoveDecorator(self):
        """
        Test for decorator.
        """
        def dec(func):
            def wrapper(*args, **kws):
                return func(*args, **kws) + 1
            return wrapper

        def f(i, j, k):
            return i + j + k

        def _test(of):
            self.assertEqual(of, f)
            self.assertEqual(of(1, 2, 3), 1 + 2 + 3)

        df = dec(f)

        of = Decorator.removeDecorator(df, "f")
        _test(of)

        # Double decorate it.
        ddf = dec(df)
        of = Decorator.removeDecorator(ddf, "f")
        _test(of)

        # Test well-behavior decorator removing.
        from decorator import decorator

        @decorator
        def tracing(f, *args, **kw):
            print("calling %s with args %s, %s" % (f.__name__, args, kw))
            return f(*args, **kw)

        wf = tracing(f)
        of = Decorator.removeDecorator(wf)
        _test(of)

        # Nested.
        wddf = tracing(ddf)
        dwddf = dec(wddf)

        of = Decorator.removeDecorator(dwddf, "f")
        _test(of)

        # Test decorated by Function.enhance.
        def e(f, *args, **kws):
            print("enhance")
            return f(*args, **kws)

        from Iuppiter import Function
        ef = Function.enhance(f, e)
        of = Decorator.removeDecorator(ef)
        _test(of)

    def testRemovePermalink(self):
        """
        Test to remove permalink decorator.
        """
        # from django.db.models import permalink
        # The django.db.models.permalink() decorator is removed.
        # https://docs.djangoproject.com/en/2.2/releases/2.1/
        from functools import wraps
        def permalink(func):
            @wraps(func)
            def inner(*args, **kwargs):
                bits = func(*args, **kwargs)
                return "permalink"
            return inner


        @permalink
        def f():
            return "?"

        try:
            f()
            self.fail("@permalink seems not attached?")
        except:
            pass

        rf = Decorator.removeDecorator(f, 'f')
        self.assertEqual(rf(), "?")

        class God(object):
            def __init__(self):
                self.id = 1

            @permalink
            def get_absolute_url(self):
                t = ('test', self.__class__.__name__)
                return ('%s.views.get%s' % t , [str(self.id)])

        # Test remove from class level.
        g = God()
        f = Decorator.removeDecorator(God.get_absolute_url, 'get_absolute_url')
        self.assertEqual(f(g), ('test.views.getGod', ['1']))

        # Test remove from object level.
        f = Decorator.removeDecorator(g.get_absolute_url, 'get_absolute_url')
        self.assertEqual(f(g), ('test.views.getGod', ['1']))

    def testDecorator(self):
        """
        Test for decorator decorator.
        """
        @Decorator.decorator
        def d(f):
            def _d(*args, **kws):
                return f(*args, **kws)
            return _d

        @d
        def s(i, j, k=1, *args, **kws):
            pass

        import inspect
        (args, varargs, varkw, defaults) = inspect.getargspec(s)
        self.assertEqual(args, ['i', 'j', 'k'])
        self.assertEqual(varargs, 'args')
        self.assertEqual(varkw, 'kws')
        self.assertEqual(defaults, (1,))     

if __name__ == '__main__':
    unittest.main()
