#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: ModelTest.py 11267 2018-11-23 08:23:13Z Kevin $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, and individual contributors.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
#   3. Neither the name of Nuwa Information nor the names of its contributors
#      may be used to endorse or promote products derived from this software
#      without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Author: Kevin $ (last)
# $Date: 2018-11-23 16:23:13 +0800 (週五, 23 十一月 2018) $
# $Revision: 11267 $

import six
import logging
   
logging.basicConfig(level=logging.INFO,
    format="%(asctime)s - %(levelname)s - "
           "Thread: %(threadName)s(%(thread)d) - %(message)s")

import unittest

# from django.test import TestCase
from django.test import TransactionTestCase
  
from Iuppiter import Model
  
class ModelTest(TransactionTestCase):
# class ModelTest(TestCase):
    """
    Test case for Model fields.
    """
    
    def tearDown(self):
        from django.contrib.auth.models import User
        User.objects.all().delete()
        TransactionTestCase.tearDown(self)
  
    def testUpdate(self):
        """
        Test delete method in ServiceManager.
        """
        logging.info('testUpdate')
       
        from django.contrib.auth.models import User
       
        User.objects.create_user('a', 'a@a.com', 'a')
        User.objects.create_user('b', 'test@test.com', 'b')
        User.objects.create_user('c', 'test@test.com', 'c')
       
       
        # By object without change statements.
        userA = User.objects.get(username="a")
        newUserA = User(id=userA.id, username="a", email="bbb@bbb.com")        
        six.get_unbound_function(Model.ServiceManager.update)(
            User.objects, newUserA)
        self.assertEqual(newUserA.email, "bbb@bbb.com")
       
        # By object with change statements.
        userA = User.objects.get(username="a")
        six.get_unbound_function(Model.ServiceManager.update)(
            User.objects, userA, email="aaa@aaa.com")
        self.assertEqual(userA.email, "aaa@aaa.com")
       
        # By query with change statements.
        query = { "email" : "test@test.com" }
        users = User.objects.filter(**query)
        ids = [u.id for u in users]
        six.get_unbound_function(Model.ServiceManager.update)(
            User.objects, query, email="ttt@ttt.com")
        updatedUsers = User.objects.filter(email="ttt@ttt.com")
        updatedIds = [u.id for u in updatedUsers]
        self.assertEqual(ids, updatedIds)
  
   
    def testDelete(self):
        """
        Test delete method in ServiceManager.
        """
        logging.info('testDelete')
        
        from django.contrib.auth.models import User
        User.objects.all().delete()
        
        User.objects.create_user('a', 'a@a.com', 'a')
        User.objects.create_user('b', 'b@b.com', 'b')
        User.objects.create_user('c', 'c@c.com', 'c')
        User.objects.create_user('d', 'd@d.com', 'd')
        User.objects.create_user('e', 'e@e.com', 'e')
        User.objects.create_user('aa', 'aa@aa.com', 'aa')
        
        self.assertEqual(User.objects.all().count(), 6)
        
        # By id.
        six.get_unbound_function(Model.ServiceManager.delete)(
            User.objects, User.objects.all()[1].id)
        self.assertEqual(User.objects.all().count(), 5)
        
        # By conditions.
        six.get_unbound_function(Model.ServiceManager.delete)(
            User.objects, username__contains='a')
        self.assertEqual(User.objects.all().count(), 3)
        
        # By object.
        u = User.objects.get(username='d')
        six.get_unbound_function(Model.ServiceManager.delete)(User.objects, u)
        self.assertEqual(User.objects.all().count(), 2)
        
        self.assertEqual(User.objects.all()[0].username, 'c')
        self.assertEqual(User.objects.all()[1].username, 'e')
      
    def testCheckExist(self):
        """
        Test checkExist method in ServiceManager.
        """
        logging.info('testCheckExist')
       
        from django.contrib.auth.models import User
        User.objects.create_user('a', 'a@a.com', 'a')
        User.objects.create_user('b', 'b@b.com', 'b')
       
        self.assertTrue(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, username='a'))
        self.assertTrue(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, email='b@b.com'))
       
        self.assertFalse(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, username='c'))
       
        from django.db.models import Q
       
        self.assertTrue(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, Q(username='b') | Q(username='a')))
       
        self.assertTrue(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, ~Q(username='b'), username='a'))

        self.assertFalse(
            six.get_unbound_function(Model.ServiceManager.checkExist)(
                User.objects, ~Q(username='b'), ~Q(username='a')))
   
    def testPreciseDateTimeField(self):
        """
        Test for PreciseDateTimeField.
        """
        import django

        if django.__version__ != '1.8':
            return

        logging.info('testPreciseDateTimeField')

        from Iuppiter.tests.models import TestPreciseDateTimeField
        import datetime

        v = datetime.datetime(2009, 10, 10)
        t = TestPreciseDateTimeField()
        t.time = v
        t.save()

        self.assertEqual(t.time, v)
        self.assertTrue(t.updateTime != None)

        t = TestPreciseDateTimeField.objects.all()[0]
        self.assertEqual(t.time, v)

        v = datetime.datetime(2009, 10, 11)
        t.time = v
        pastUpdateTime = t.updateTime

        import time
        time.sleep(0.1)

        t.save() # Update.
        self.assertEqual(t.time, v)
        self.assertTrue(t.updateTime > pastUpdateTime)

        #updateTime = t.updateTime
        t = TestPreciseDateTimeField.objects.all()[0]
        self.assertEqual(t.time, v)
        # FIXME: There will be a very small difference...
        #self.assertTrue(t.updateTime == updateTime)

    
    
    def testTransaction(self):
        """
        Test for transacton management Eventum #704.
        """
        logging.info('testTransaction')

        from django.contrib.auth.models import User
        from django.db import transaction

        User.objects.all().delete()

        self.c = 1

        # Try rollback.
        @Model.transaction
        def f():
            name = str(self.c)
            mail = "%d@%d.com" % (self.c, self.c)
            User.objects.create_user(name, mail, mail)
            self.c += 1

        @Model.transaction
        def f2():
            f()
            raise RuntimeError()

        try:
            f2()
            self.fail('RuntimeError should be raised.')
        except RuntimeError:
            pass

        self.assertEqual(len(User.objects.all()), 0)

        # Try commit.
        f()
        self.assertEqual(len(User.objects.all()), 1)

        @Model.transaction
        def f3():
            f()
            f()

        f3()
        self.assertEqual(len(User.objects.all()), 3)

        @Model.transaction
        def f4():
            f3()
            1 / 0
            f()

        try:
            f4()
            self.fail('Exception should be raised.')
        except:
            pass

        self.assertEqual(len(User.objects.all()), 3)

        # Test for whether it is well-behavior decorator or not.
        @Model.transaction
        def s(i, j, k=1, *args, **kws):
            f4()
 
        import inspect
        (args, varargs, varkw, defaults) = inspect.getargspec(s)
        self.assertEqual(args, ['i', 'j', 'k'])
        self.assertEqual(varargs, 'args')
        self.assertEqual(varkw, 'kws')
        self.assertEqual(defaults, (1,))

    def testVoid(self):
        pass
  
if __name__ == '__main__':
#     print "123"
    unittest.main()

