#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 1340 2022-05-10 23:57:40Z Jacky $
#
# Copyright (c) 2019 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Jacky $ (last)
# $Date: 2022-05-11 07:57:40 +0800 (週三, 11 五月 2022) $
# $Revision: 1340 $

import os
import sys
import six
import json
import datetime
import re

import requests

from pathlib import Path

from Crypto.PublicKey import RSA

from decorator import decorator
from termcolor import colored, cprint
from fabric import task
from fabric import Connection as _Connection
from fabric import connection
from fabric.connection import opens

from contextlib import contextmanager
from rich.progress import Progress, SpinnerColumn, TextColumn, BarColumn, TimeRemainingColumn, TimeElapsedColumn

# patch
from invoke.runners import Runner

DEFAULT_MARIADB_PASSOWRD = os.environ.get(
    'DEFAULT_MARIADB_PASSOWRD', "SuXeMA3oTeXWgfjz2G1BGrPm0v0OKw")

DEPLOY_TEST_SERVER_DOMAIN = os.environ.get(
    'DEPLOY_TEST_SERVER_DOMAIN', 'test.nuwainfo.com')
DEPLOY_TEST_SITE_DOMAIN = os.environ.get(
    'DEPLOY_TEST_SITE_DOMAIN', 'scaffold.nuwainfo.com')

DEPLOY_TEST_SERVER_USER = os.environ.get(
    'DEPLOY_TEST_SERVER_USER', 'nuwa')
DEPLOY_TEST_SERVER_PASSWORD = os.environ.get(
    'DEPLOY_TEST_SERVER_PASSWORD', '@%)@5529')

_read_our_stdin = Runner.read_our_stdin

def read_our_stdin(self, input_):
    bytes_ = _read_our_stdin(self, input_)
    if bytes_:
        return bytes_
    else:
        return None

Runner.read_our_stdin = read_our_stdin

def getBuildData(shPath, dataName, alternatives=[]):
    with open(shPath, 'r') as f:
        for line in f.readlines():
            lineWithoutSpace = line.lstrip()
            if lineWithoutSpace:
                firstInLine = lineWithoutSpace[0]

            if firstInLine != "#": # 沒有註解的才會進來
                if "%s=" % dataName in line:
                    index = line.index("=") + 1
                    return line[index:].replace('\r\n', '').strip()
                elif alternatives:
                    for alternative in alternatives:
                        if "%s=" % alternative in line:
                            index = line.index("=") + 1
                            return line[index:].replace('\r\n', '').strip()

    return None

# https://zh.wikipedia.org/wiki/%E4%B8%93%E7%94%A8%E7%BD%91%E7%BB%9C
def isVirtualMachine(ip):
    if ip.startswith("192.168.") or ip.startswith("10."):
        return True
    for i in range(16, 32):
        if ip.startswith("172.%s." % i):
            return True
    return False

@contextmanager
def createProgress(name="[red]Processing"):
    class Task:
        def __init__(self, progress, task):
            self.progress = progress
            self.task = task
        def update(self, advance):
            self.progress.update(self.task, advance=advance)

    if hasattr(createProgress, 'live'):
        progress = createProgress.live
        task = progress.add_task(name, total=100)
        yield Task(progress, task)
    else:
        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            BarColumn(),
            TextColumn("[progress.percentage]{task.percentage:>3.0f}%"),
            TimeRemainingColumn(),
            TimeElapsedColumn(),
        ) as progress:
            createProgress.live = progress
            task = progress.add_task(name, total=100)
            yield Task(progress, task)

def showSplash():
    import pyfiglet
    cprint("-" * 79, "white", attrs=["bold"],)
    cprint(pyfiglet.figlet_format('Nuwa Info', font="standard"),
           "blue", attrs=["bold"],)

    cprint(pyfiglet.figlet_format('Deploy', font="standard"),
           "yellow", attrs=["bold"],)
    cprint("-" * 79, "white", attrs=["bold"],)

    if six.PY2:
        cprint(
            "Deploy is only run by python3.",
            "blue",
            "on_white",
            attrs=["bold", "underline"],
        )
        sys.exit()
    else:
        cprint(
            colored(
                "When enter the shell, "
                "you must press Enter twice to execute command.",
                "yellow"
            )
        )

def getConnectKwargs(password=None, key=os.path.join("PrivateKey.key")):
    # put key
    if password:
        return {"password": password,}
    else:
        if os.path.exists(key):
            return {"key_filename": key,}
        else:
            return None

def getOrCreateKey(username, keyPath=os.path.join("Keys")):
    if not os.path.exists(keyPath):
        os.mkdir(keyPath)
    userPath = os.path.join(keyPath, username)
    privateKeyPath = os.path.join(userPath, "PrivateKey.key")
    publicKeyPath = os.path.join(userPath, "PublicKey.key")

    # check key
    if os.path.isdir(userPath):
        # get key
        with open(publicKeyPath, 'r') as f:
            publicKey = f.read()
        with open(privateKeyPath, 'r') as f:
            privateKey = f.read()
    else:
        # make key
        os.mkdir(userPath)

        key = RSA.generate(4096)

        publicKey = key.publickey().exportKey('OpenSSH').decode("utf-8")
        privateKey = key.exportKey("PEM").decode("utf-8")

        with open(publicKeyPath, 'w') as f:

            f.write(publicKey)

        with open(privateKeyPath, 'w') as f:
            privateKey = privateKey.replace(
                "BEGIN PRIVATE KEY", "BEGIN RSA PRIVATE KEY")
            privateKey = privateKey.replace(
                "END PRIVATE KEY", "END RSA PRIVATE KEY")

            f.write(privateKey)


    return (publicKey, privateKey, publicKeyPath, privateKeyPath)

@decorator
def record(func, c, *args, **kwargs):
    rt = None
    try:
        if not hasattr(c, 'recordLogs'):
            # 沒有連到遠端的指令，EX: python Deploy.py mode stage
            def _recordLogs():
                # 因為沒有連遠端，不用記 log
                pass
            c.recordLogs = _recordLogs
        if not hasattr(c, 'addLog'):
            # 沒有連到遠端的指令，EX: python Deploy.py mode stage
            def _addLog(msg):
                # 因為沒有連遠端，直接把 log print 出來
                cprint(
                    colored(
                        f'Log: {msg}',
                        "white"
                    )
                )
            c.addLog = _addLog

        # 執行指令
        rt = func(c, *args, **kwargs)
    except Exception as e:
        cprint(
            colored(
                "ERROR: %s" % str(e),
                "red"
            )
        )
        c.addLog("ERROR: %s" % str(e))
        raise
    finally:
        c.recordLogs()

    return rt

# patch
@opens
def run(self, command, notRecord=False, hideStdout=False, **kwargs):
    """
    Execute a shell command on the remote end of this connection.

    This method wraps an SSH-capable implementation of
    `invoke.runners.Runner.run`; see its documentation for details.

    .. warning::
        There are a few spots where Fabric departs from Invoke's default
        settings/behaviors; they are documented under
        `.Config.global_defaults`.

    .. versionadded:: 2.0
    """
    if "hide" in kwargs and kwargs["hide"] == True:
        pass
    else:
        cprint(
            command,
            "blue",
            "on_white",
            attrs=["bold", "underline"],
        )

    if six.PY2 and not kwargs.get("pty"):
        kwargs["hide"] = True

    if hideStdout:
        kwargs["hide"] = True

    if not notRecord:
        c = "%s: %s" % (command, str(datetime.datetime.now().time()))
        if hasattr(self, "commands"):
            self.commands.append(c)
        else:
            self.commands = [c]

    if not "encoding" in kwargs:
        kwargs["encoding"] = "utf8"

    # with open("test.log", 'a+') as f:
        # result = self._run(self._remote_runner(), command, out_stream=f, err_stream=f, **kwargs)
    result = self._run(self._remote_runner(), command, **kwargs)

    return result

def recordLogs(self):
    # get user
    if "key_filename" in self.connect_kwargs:
        from Crypto.PublicKey import RSA

        with open(self.connect_kwargs["key_filename"], "r") as f:
            key = RSA.import_key(f.read())

        publicKey = key.publickey().exportKey('OpenSSH').decode("utf-8")

        result = self.run(
            "cat ~/.ssh/authorized_keys | grep '%s'" % publicKey,
            hide=True, notRecord=True)
        result = result.stdout
        username = re.findall(r"USERNAME=(\w+)", result)[0]
    else:
        username = "someone"

    # get ip
    # request https://api.ipify.org/?format=json to get ip
    try:
        ip = requests.get("https://api.ipify.org/?format=json")
        ip = json.loads(ip.text)
    except Exception as e:
        ip = {}
    finally:
        username = "%s(%s)" % (username, ip.get("ip", "unknown"))

    # 跳脫
    commands = [cmd.replace("'", "'\\''") for cmd in self.commands]

    # record
    date = datetime.datetime.now().date()
    msg = "Execute command by %s at %s" % (username, date)
    logs = (
        "-----%s-----"
        "\\n%s\\n"
        "-----%s-----\\n") % (
            msg, "\\n".join(commands), "-" * len(msg))
    cmd = "echo -e '%s' >> ~/.deploy_history" % logs
    self.run(cmd, hide=True, notRecord=True)

    # clean
    self.commands = []

    return

def addLog(self, log):
    log = "%s: %s" % (log, str(datetime.datetime.now().time()))
    if hasattr(self, 'commands'):
        self.commands.append(log)
    else:
        self.commands = [log,]

connection.Connection.run = run
connection.Connection.addLog = addLog
connection.Connection.recordLogs = recordLogs

def unixPath(*args):
    return os.path.join(*args).replace("\\", "/")

def prompt(msg):
    if six.PY2:
        result = raw_input(msg)
    else:
        result = input(msg)
    return result

def utf8(s, encodings=None, throw=True):
    """
    Convert a string (UNICODE or ANSI) to a utf8 string.

    @param s String.
    @param encodings Native encodings for decode. It will be tried to decode
                     string, try and error.
    @param throw Raise exception if it fails to convert string.
    @return UTF8 string.
    """
    if isinstance(s, six.text_type):
        return s.encode('utf-8')
    else:
        return _unicode(s, encodings=encodings, throw=throw).encode('utf-8')

def configureDjango(*args, **kws):
    """
    Configure django settings by given arguments.

    @param *args Arguments.
    @param **kws Keyword arguments.
    """
    import django.template.loader
    from django.conf import settings as djangoSettings
    from django.utils.functional import empty

    djangoSettings._wrapped = empty
    djangoSettings.configure(*args, **kws)

    django.setup()

def checkRequirementsExisted(path):
    """
    Check if there's any 'requirements.txt' or 'REQUIREMENTS.txt' file in the path.

    @param path: The path that may have a 'requirements.txt' or REQUIREMENTS.txt' file.
    @return: A boolean. The path have ('requirements.txt' or 'REQUIREMENTS.txt') or not.
    """
    return Path(path, 'requirements.txt').exists() or Path(path, 'REQUIREMENTS.txt').exists()

def findSettings(path):
    """
    Find 'settings.py' in the path or in sub-directory of the path.

    @param path: The project path that should contains a 'settings.py' file.
    @return: The path to 'settings.py' if the path really contains a 'settings.py file. Otherwise, None.
    """
    for root, dirs, files in os.walk(path):
        for f in files:
            if f == 'settings.py':
                return os.path.join(root, f)
    return None

def grepData(filePath, pattern, encoding='UTF-8'):
    """
    Get string which leaded by a certain pattern in the file.
    For example: you can get value of 'APP_NAME' in 'settings.py' by using 'grepData(settingPath, "APP_NAME = ")'

    @param filePath: Path to the file.
    @param pattern: Certain pattern, and you want to get the string
                    which is in the same line and leaded by this pattern.
    @param encoding: The file's encoding method.
    @return: String after the pattern if the file exists and the file contains the pattern. Otherwise, None.
    """
    if not Path(filePath).exists():
        return None
    with open(filePath, 'r', encoding=encoding) as f:
        for line in f.readlines():
            if line.startswith(pattern):
                return line[len(pattern):].strip()
    return None

def getRepository(projectPath):
    """
    Get the repository URL of the project.

    @param projectPath: Path to the project.
    @return: The repository URL if no exception. Otherwise, None.
    """
    os.system("svn info %s > info.txt" % projectPath)
    repository = grepData('info.txt', 'URL: ', encoding=None)
    os.remove('info.txt')
    return repository

def validateDomainFormat(domain):
    """
    Check if the domain name is in valid format.
    Only lowercase characters, numbers, "-" and "." are valid in domain name. 

    @param domain: The domain name.
    @return: A boolean. The domain name is in valid format or not.
    """
    # Using regex: re.match()
    # If match, the function returns a "re.Match" object,
    # which can be converted to a boolean type object with "True" value.
    # Else, returns "None".
    pattern = r'([a-z0-9]([a-z0-9-]{0,61}[a-z0-9])?\.)+[a-z0-9][a-z0-9-]{0,61}[a-z0-9]'
    return bool(re.match(pattern, domain))
