#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: create_pages.py 11840 2020-01-07 13:45:58Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-01-07 21:45:58 +0800 (Tue, 07 Jan 2020) $
# $Revision: 11840 $

import os
import json
import traceback

import cms

from django.core.management.base import BaseCommand
from django.conf import settings
from django.db import transaction
from cms.api import create_page
from cms.models.titlemodels import Title

from Zephyrus.boilerplate.management.commands import (
        TEMPLATES_DIR_PATH, ROOT_DIR_PATH)
from Zephyrus.boilerplate.management.commands import createLogger
from Zephyrus.boilerplate.management.commands import open3 as open

logger = createLogger(__name__, "CreatePages.log")

class Command(BaseCommand):

    help = 'Auto create all pages.'

    @transaction.atomic
    def handle(self, *args, **options):
        try:
            if os.path.isfile(os.path.join(ROOT_DIR_PATH, 'info.json')):
                with open(os.path.join(ROOT_DIR_PATH, 'info.json')) as jsonfile:
                    data = json.load(jsonfile)
                    slug = data['slug']
            else:
                slug = None
                
            createPage = False
            rootPage = None

            for root, dirs, files in os.walk(TEMPLATES_DIR_PATH):
                if not createPage:
                    # 還沒找到 root page
                    if 'index.html' in [f.lower() for f in files]:
                        for i, f in enumerate([f.lower() for f in files], 0):
                            if f == 'index.html':
                                createPage = True
                                rootPage = files[i]
                    else:
                        for f in files:
                            if f.endswith('.html'):
                                rootPage = f
                                createPage = True
                                logger.warn(
                                    ("'%s' template use '%s' as root page.") % 
                                    (slug, rootPage))
                                break

                    if createPage:
                        # create root page
                        path = os.path.join(root, rootPage)
                        data = self.getPageData(path)
                        
                        title = data['title']
                        description = data['description']
                        
                        if (TEMPLATES_DIR_PATH.endswith('/') or 
                            TEMPLATES_DIR_PATH.endswith('\\')):
                            rootPagePath = os.path.join(
                                root.replace(TEMPLATES_DIR_PATH, ''), 
                                rootPage
                            )
                        else:
                            # 去掉第一個 / or \
                            rootPagePath = os.path.join(
                                root.replace(TEMPLATES_DIR_PATH, '')[1:], 
                                rootPage
                            )

                        homePage = create_page(
                            'index', rootPagePath, settings.LANGUAGE_CODE, 
                            published=True, meta_description=description)

                        if not cms.__version__.startswith('3.4'):
                            homePage.is_home = True
                            homePage.save()

                        page = homePage.get_public_object()
                        pageTitle1 = Title.objects.get(page=homePage)
                        pageTitle2 = Title.objects.get(page=page)
                        for t in [pageTitle1, pageTitle2]:
                            t.page_title = title
                            if not cms.__version__.startswith('3.4'):
                                t.path = ''
                            t.save()

                        logger.info('Create homepage by %s' % rootPage)

                if createPage:
                    # 已經有找到 root page
                    for template in files:
                        if f.endswith('.html'):
                            name = template[:template.index(".html")]
                           
                            if template == rootPage:
                                continue
                                
                            path = os.path.join(root, template)
                            data = self.getPageData(path)
                            
                            title = data['title']
                            description = data['description']
                            
                            if (TEMPLATES_DIR_PATH.endswith('/') or 
                                TEMPLATES_DIR_PATH.endswith('\\')):
                                templatePath = os.path.join(
                                    root.replace(TEMPLATES_DIR_PATH, ''), 
                                    template
                                )
                            else:
                                # 去掉第一個 / or \
                                templatePath = os.path.join(
                                    root.replace(TEMPLATES_DIR_PATH, '')[1:], 
                                    template
                                )
                            
                            page1 = create_page(
                                name, templatePath, settings.LANGUAGE_CODE, 
                                parent=homePage, published=True, 
                                meta_description=description)

                            page2 = page1.get_public_object()
                            pageTitle1 = Title.objects.get(page=page1)
                            pageTitle2 = Title.objects.get(page=page2)
                            for t in [pageTitle1, pageTitle2]:
                                t.page_title = title
                                if "/" in templatePath or "\\" in templatePath:
                                    t.path = templatePath.replace(
                                        ".html", "").replace("\\", "/")
                                t.save()

                            logger.info(
                                'Create %s page by %s' % (name, template))
                    
            if not createPage:
                logger.error("%s No templates to create pages." % slug)
            else:
                logger.info('All OK!!')
        except Exception as e:
            logger.error("create_pages error:%s" % e)
            with open('./AllError.log', 'w+') as log:
                log.write(traceback.format_exc())
            raise

    def getPageData(self, path):
        with open(path) as f:
            content = f.read()

        startIndex = content.index('<!--CMS_PAGE_DATA') +\
                     len('<!--CMS_PAGE_DATA') 
        endIndex = content.index('END_CMS_PAGE_DATA-->')

        data = content[startIndex:endIndex]

        content = content.replace(
            '<!--CMS_PAGE_DATA%sEND_CMS_PAGE_DATA-->' % data, '')

        with open(path, 'w') as f:
            f.write(content)

        return json.loads(data)
        

