#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 12044 2020-04-05 11:36:35Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-04-05 19:36:35 +0800 (Sun, 05 Apr 2020) $
# $Revision: 12044 $

from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.core.validators import MaxValueValidator, MinValueValidator
from solo.models import SingletonModel

from cms.models import CMSPlugin

from Zephyrus.facebook import settings

class FacebookSettings(SingletonModel):
    accessToken = models.CharField(_('Access token'), max_length=512, blank=True, null=True, default=None)

class FacebookFan(CMSPlugin):
    url = models.URLField(_("Facebook fan page URL"), max_length=400)
    limit = models.PositiveIntegerField(_("Number of articles"), default=0, 
        validators=[MaxValueValidator(20),])

    renderTemplate = models.CharField(
        _('Fan Template'), max_length=150, blank=True,
        choices=settings.ZEPHYRUS_FACEBOOK_FAN_TEMPLATES,
        default=settings.ZEPHYRUS_FACEBOOK_FAN_DEFAULT_TEMPLATE,
    )

    # display:
    unknownDisplay = models.BooleanField(
        _('Display unknown article'), default=True)
    videoDisplay = models.BooleanField(
        _('Display video article'), default=True)
    albumDisplay = models.BooleanField(
        _('Display album photo article'), default=True)
    shareDisplay = models.BooleanField(
        _('Display share article'), default=True)
    commerceDisplay = models.BooleanField(
        _('Display commerce article'), default=True)

    onlyMessageDisplay = models.BooleanField(
        _('Only display article has message'), default=True)

    class Meta:
        verbose_name = _("Facebook Fan Page")
        verbose_name_plural = _("Facebook Fan Page")
    
class FacebookPhoto(CMSPlugin):
    url = models.URLField(_("Facebook fan page album URL"), max_length=400)
    limit = models.PositiveIntegerField(_("Number of photos"), default=0,
        validators=[MaxValueValidator(20),])
    renderTemplate = models.CharField(
        _('Photo Template'), max_length=150, blank=True,
        choices=settings.ZEPHYRUS_FACEBOOK_PHOTO_TEMPLATES,
        default=settings.ZEPHYRUS_FACEBOOK_PHOTO_DEFAULT_TEMPLATE,
    )

    class Meta:
        verbose_name = _("Facebook Fan Photo")
        verbose_name_plural = _("Facebook Fan Photo")

LAYOUT_CHOICE = (
    ('standard',  _("Standard")),
    ('button_count',  _("Button Count")),
    ('button',  _("Button")),
    ('box_count',  _("Box Count")),
)

ACTION_CHOICE = (
    ('like',  _("Like")),
    ('recommend',  _("Recommend")),
)

SIZE_CHOICE = (
    ('small',  _("Small")),
    ('large',  _("Large")),
)

class ShareLike(CMSPlugin):
    layout = models.CharField(
        _("Layout"),
        max_length=15,
        choices=LAYOUT_CHOICE,
        default='standard',
    )

    action = models.CharField(
        _("Action"),
        max_length=10,
        choices=ACTION_CHOICE,
        default='like',
    )

    size = models.CharField(
        _("Size"),
        max_length=5,
        choices=SIZE_CHOICE,
        default='small',
    )

    showFaces = models.BooleanField(_("Show faces"),)
    share = models.BooleanField(_("Share"),)
    appId = models.CharField(
        _("Facebook API app ID"), max_length=20, null=True, blank=True)

    width = models.PositiveIntegerField(_("Width"), default=450)
    height = models.PositiveIntegerField(_("Height"), default=80)

TAB_CHOICE = (
    ('timeline',  _("Timeline")),
    ('events',  _("Events")),
    ('messages',  _("Messages")),
)

class FanPage(CMSPlugin):
    url = models.URLField(max_length=400)
    appId = models.CharField(
        _("Facebook API app ID"), max_length=20, null=True, blank=True)

    width = models.PositiveIntegerField(
        _("Width"), 
        default=340,
        validators=[
            MaxValueValidator(500),
            MinValueValidator(180)
        ])
    height = models.PositiveIntegerField(
        _("Height"), 
        default=500,
        validators=[
            MinValueValidator(70),
        ])

    tab = models.CharField(
        _("Tab"),
        max_length=10,
        choices=TAB_CHOICE,
        default='timeline',
    )

    hideCover = models.BooleanField(_("Hide cover"), default=False)
    showFacepile = models.BooleanField(_("Show facepile"), default=True)
    hideCta = models.BooleanField(_("Hide cta"), default=False)
    smallHeader = models.BooleanField(_("Small header"), default=False)
    adaptContainerWidth = models.BooleanField(
        _("Adapt Container width"), default=False)

class APIData(models.Model):
    url = models.CharField(max_length=400, editable=False, db_index=True)
    likeCount = models.PositiveIntegerField(
        verbose_name=_("Like count"), default=0)
    commentCount = models.PositiveIntegerField(
        verbose_name=_("Comment count"), default=0)
    updateTime = models.DateTimeField(auto_now=True)