#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: __init__.py 11186 2018-08-05 05:18:00Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2018-08-05 14:18:00 +0900 (週日, 05 八月 2018) $
# $Revision: 11186 $

import json
import traceback
import logging

import requests

from des.backends import ConfiguredEmailBackend as _ConfiguredEmailBackend
from des.models import DynamicEmailConfiguration
from django.conf import settings
from django.core.mail.backends.smtp import EmailBackend as SmtpEmailBackend

from Iuppiter.Logging import createLogger

logger = createLogger(__name__)

NEWSKETTER_BLACKCONTACT_SEARCH_URL = "https://newsletter.nuwainfo.com/newsletters/newsletter/blackcontact/search/"

def filterBlacklist(emailMessages):
    # filter black contact 
    emailMessagesAfterFilter = []
    for message in emailMessages:
        try:
            payload = {
                "emails": message.to,
            }
            response = json.loads(
                requests.post(NEWSKETTER_BLACKCONTACT_SEARCH_URL, data=payload, 
                timeout=getattr(settings, "IUNO_EMAIL_BACKEND_SEARCH_BLACKCONTACT_TIMEOUT", 1)
            ).text)
            emails = [result['email'] for result in response.get("results", [])]
            if emails:
                message.to = [email for email in message.to if not email in emails]
            
            if message.to:
                emailMessagesAfterFilter.append(message) 
        except Exception as e:
            logger.error(str(e))
            emailMessagesAfterFilter.append(message)
            
    return emailMessagesAfterFilter
    

class ConfiguredEmailBackend(_ConfiguredEmailBackend):
    def send_messages(self, emailMessages):
        # filter black contact 
        emailMessages = filterBlacklist(emailMessages)
        
        fromEmailList = []
        try:
            # 用 DynamicEmailConfiguration 寄信
            fromEmail = DynamicEmailConfiguration.objects.get().from_email
            for message in emailMessages:
                fromEmailList.append(message.from_email)
                message.from_email = fromEmail

            return super(
                ConfiguredEmailBackend, self).send_messages(emailMessages)
        except Exception as e:
            # 不能用 DynamicEmailConfiguration 寄信，
            # 因此將 from_email 拿回來試試 SMTP 寄信
            for i, message in enumerate(emailMessages, 0):
                message.from_email = fromEmailList[i]
            errMsg = (("Use dynamic email configuration to send email error: "
                       "%s\n%s\nUse SmtpEmailBackend to send email") % 
                    (str(e), traceback.format_exc()))
            logger.warning(errMsg)
            smtp = SmtpEmailBackend()
            try:
                r = smtp.send_messages(emailMessages)
                return r
            except Exception as e:
                # 不能用 DynamicEmailConfiguration 寄信，
                # 而且將 from_email 拿回來也不能用 SMTP 寄信，因此將所有信的
                # fromEmail 用 settings.DEFAULT_FROM_EMAIL 再用 SMTP 寄一次
                errMsg = (("Use SMTP to send email error: "
                           "%s\n%s\nUse SmtpEmailBackend to send email "
                           "by %s as from Email") % 
                    (str(e), traceback.format_exc(), 
                     settings.DEFAULT_FROM_EMAIL))
                logger.warning(errMsg)
                for message in emailMessages:
                    if not message.from_email == settings.DEFAULT_FROM_EMAIL:
                        message.from_email = settings.DEFAULT_FROM_EMAIL
                
                try:                
                    r = smtp.send_messages(emailMessages)
                    return r
                except Exception as e:
                    # 完全寄不了信
                    errMsg = ("ERROR: %s\n%s\nError send email" % 
                    (str(e), traceback.format_exc()))
                    logger.error(errMsg)
                    return
