#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 10369 2017-10-14 09:46:18Z David $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $
# $Date: 2017-10-14 17:46:18 +0800 (Sat, 14 Oct 2017) $
# $Revision: 10369 $

from django.conf import settings
from django.shortcuts import redirect

from Iuppiter.extension.views import jsonpCallback, view
from Iuppiter.dispatch import ThreadPool
from Iuppiter.Logging import createLogger

from Theophrastus.Utility import _slugify
from Theophrastus.cloud.Mailer import CloudMailer

from models import MailingList, Newsletter, ContactMailingStatus, Contact

@jsonpCallback(noPadding=True)
def getSlugName(request):
    """
    Get slug name.

    @param request Request instance.
    """
    title = request.GET.get('title', '')
    if title:
        return {'slug': _slugify(title)}
    else:
        return {}

@view('FeedComplete.html')
def feedComplete(request):
    """
    Redirect to this view when feed completed.

    If feed successfully, we will get a argument "post_id".
    Sample: post_id=100000187377979_135405309958471

    @param request Request instance.
    """
    postId = request.GET.get('post_id', '')
    return {'postId': postId}

@jsonpCallback()
def getMailingListInfo(request):
    """
    Get MailingList information.

    @param request Request instance.
    """
    mailingListId = request.GET.get('mailingList', '')
    newsletterIds = request.GET.get('newsletters', '')
    newsletterIds = newsletterIds.split('_')
    try:
        if mailingListId:
            mailingList = MailingList.objects.get(id=mailingListId)
            return {
                'subscribersCount': mailingList.subscribers_count(),
                'name': mailingList.name
            }
        elif newsletterIds:
            info = []
            for n in Newsletter.objects.filter(id__in=newsletterIds):
                mailingList = n.mailing_list
                info.append({
                    'subscribersCount': mailingList.subscribers_count(),
                    'name': n.title
                })

            return {'info': info}
    except Exception as e:
        pass
    return {'error': str(e)}

def resendErrorContact(requset, newsletterId):
    """
    Resend newsletter to the contacts which has error when
    sending mail previously.

    @param request Request instance.
    """
    SENDBOX_THREAD_POOL_COUNT = settings.SENDBOX_THREAD_POOL_COUNT
    ENABLE_FAKE_MAIL = settings.ENABLE_FAKE_MAIL
    ENABLE_FAKE_SMTP = settings.ENABLE_FAKE_SMTP
    LOG_LEVEL = settings.LOG_LEVEL
    MAILER_LOGFILE = settings.MAILER_LOGFILE

    sendboxPool = ThreadPool(SENDBOX_THREAD_POOL_COUNT)

    mailerLogger = createLogger(
                'theophrastusMailer', MAILER_LOGFILE, loggingLevel=LOG_LEVEL)

    newsletter = Newsletter.objects.filter(id=newsletterId)[0]

    contactIdSet = set(
        ContactMailingStatus.objects.filter(
            newsletter_id=newsletter.id, status=1).values_list('contact_id'))
    contactIdList = [ele[0] for ele in list(contactIdSet)]
    contactList = Contact.objects.filter(id__in=contactIdList)

    mailer = CloudMailer(newsletter, pool=sendboxPool,
                                     fakeMail=ENABLE_FAKE_MAIL,
                                     fakeSMTP=ENABLE_FAKE_SMTP,
                                     logger=mailerLogger)

    mailer.useCloud = True if newsletter.server.id == 1 else False
    mailer.smtp_connect()
    mailer.attachments = mailer.getAttachments()

    for contact in contactList:
        mailer.send(contact)
        
    return redirect(newsletter.get_historic_url())